/*****************************************************************************
 * Copyright Avaya Inc., All Rights Reserved.
 * THIS IS UNPUBLISHED PROPRIETARY SOURCE CODE OF Avaya Inc.
 * The copyright notice above does not evidence any actual or intended publication of such source code.
 * Some third-party source code components may have been modified from their original versions by Avaya Inc.
 * The modifications are Copyright Avaya Inc., All Rights Reserved.
 * Avaya  Confidential & Restricted. May not be distributed further without written permission of
 * the Avaya owner.
 ****************************************************************************/

package com.avaya.collaboration.authorization.http;

import java.io.IOException;
import java.net.UnknownHostException;
import java.security.KeyManagementException;
import java.security.NoSuchAlgorithmException;
import java.util.concurrent.Executors;
import java.util.concurrent.ScheduledExecutorService;
import java.util.concurrent.TimeUnit;

import javax.net.ssl.SSLContext;

import org.apache.http.client.HttpClient;
import org.apache.http.client.config.RequestConfig;
import org.apache.http.config.Registry;
import org.apache.http.config.RegistryBuilder;
import org.apache.http.conn.socket.ConnectionSocketFactory;
import org.apache.http.conn.ssl.AllowAllHostnameVerifier;
import org.apache.http.conn.ssl.SSLConnectionSocketFactory;
import org.apache.http.impl.client.CloseableHttpClient;
import org.apache.http.impl.client.HttpClients;
import org.apache.http.impl.conn.PoolingHttpClientConnectionManager;

import com.avaya.collaboration.authorization.modules.SslProvider;
import com.google.inject.Inject;
import com.google.inject.Singleton;

/**
 * Initializes an HTTP Connection Manager and provides an HttpClient instance. Also starts an audit task to periodically close stale
 * connections.
 *
 * @author Avaya
 *
 */
@Singleton
public class HttpClientProviderImpl implements HttpClientProvider
{
    private final SslProvider<SSLContext> sslContextProvider;

    private PoolingHttpClientConnectionManager clientConnectionManager;
    private ScheduledExecutorService staleConnectionAuditService;
    private CloseableHttpClient clientInstance;

    private static final long AUDIT_PERIOD = 45; // seconds
    private static final long INITIAL_DELAY = 5; // seconds
    private static final int TIMEOUT = 30 * 1000; // ms

    @Inject
    public HttpClientProviderImpl(final SslProvider<SSLContext> sslContextProvider) throws UnknownHostException, KeyManagementException,
            NoSuchAlgorithmException
    {
        this.sslContextProvider = sslContextProvider;

        initializeConnectionManager();

        initializeConnectionManagerAudit();

        initializeHttpClient();
    }

    private void initializeConnectionManager() throws KeyManagementException, NoSuchAlgorithmException
    {
        final SSLContext sslContext = sslContextProvider.get();

        final SSLConnectionSocketFactory sslsf =
                new SSLConnectionSocketFactory(
                        sslContext,
                        SSLConnectionSocketFactory.ALLOW_ALL_HOSTNAME_VERIFIER);

        final Registry<ConnectionSocketFactory> registry = RegistryBuilder.<ConnectionSocketFactory> create()
                .register("https", sslsf)
                .build();

        clientConnectionManager = new PoolingHttpClientConnectionManager(registry);

        // Set max total connections to 200
        clientConnectionManager.setMaxTotal(200);

        // Set max connections per route to 30
        clientConnectionManager.setDefaultMaxPerRoute(30);
    }

    private void initializeConnectionManagerAudit()
    {
        staleConnectionAuditService = Executors.newSingleThreadScheduledExecutor();

        staleConnectionAuditService.scheduleAtFixedRate(new Runnable()
        {
            @Override
            public void run()
            {
                // Close expired connections
                clientConnectionManager.closeExpiredConnections();

                // Close connections that have been idle longer than 45 sec
                clientConnectionManager.closeIdleConnections(45, TimeUnit.SECONDS);
            }
        }, INITIAL_DELAY, AUDIT_PERIOD, TimeUnit.SECONDS);
    }

    private void initializeHttpClient() throws UnknownHostException
    {

        final RequestConfig requestConfig = RequestConfig.custom()
                .setSocketTimeout(TIMEOUT)
                .setConnectTimeout(TIMEOUT)
                .setConnectionRequestTimeout(TIMEOUT)
                .build();

        clientInstance = HttpClients.custom().setHostnameVerifier(new AllowAllHostnameVerifier())
                .setConnectionManager(clientConnectionManager).setDefaultRequestConfig(requestConfig).build();
    }

    @Override
    public HttpClient getClientIntance()
    {
        return clientInstance;
    }

    @Override
    public void shutDown() throws IOException
    {
        clientInstance.close();
        clientConnectionManager.close();
        staleConnectionAuditService.shutdown();
    }
}