/*****************************************************************************
 * Copyright Avaya Inc., All Rights Reserved.
 * THIS IS UNPUBLISHED PROPRIETARY SOURCE CODE OF Avaya Inc.
 * The copyright notice above does not evidence any actual or intended publication of such source code.
 * Some third-party source code components may have been modified from their original versions by Avaya Inc.
 * The modifications are Copyright Avaya Inc., All Rights Reserved.
 * Avaya  Confidential & Restricted. May not be distributed further without written permission of
 * the Avaya owner.
 ****************************************************************************/

package com.avaya.collaboration.authorization.http;

import java.io.IOException;

import org.apache.http.HttpEntity;
import org.apache.http.HttpStatus;
import org.apache.http.client.HttpClient;
import org.apache.http.client.methods.CloseableHttpResponse;
import org.apache.http.client.methods.HttpPost;
import org.apache.http.entity.BufferedHttpEntity;
import org.apache.http.util.EntityUtils;

import com.avaya.collaboration.authorization.AccessToken;
import com.avaya.collaboration.authorization.AuthorizationHelperException;
import com.avaya.collaboration.authorization.HttpResponseException;
import com.avaya.collaboration.authorization.RefreshToken;
import com.avaya.collaboration.authorization.TokenAggregate;
import com.avaya.collaboration.authorization.impl.AccessTokenImpl;
import com.avaya.collaboration.authorization.impl.RefreshTokenImpl;
import com.google.gson.FieldNamingPolicy;
import com.google.gson.Gson;
import com.google.gson.GsonBuilder;

/**
 * Makes a POST HTTP request to the /token endpoint of the Authorization Service. Upon successful response, the execute() method returns an
 * {@link AccessToken}
 *
 * @author Avaya
 *
 */
public class HttpRequestExecutor
{
    private final HttpClient httpClient;
    private final HttpPost httpPost;

    public HttpRequestExecutor(final HttpClient httpClient, final HttpPost postRequest)
    {
        this.httpClient = httpClient;
        httpPost = postRequest;
    }

    public final AccessToken execute() throws HttpResponseException,
            AuthorizationHelperException
    {
        try
        {
            final CloseableHttpResponse response = (CloseableHttpResponse) httpClient.execute(httpPost);
            try
            {
                if (response.getStatusLine().getStatusCode() == HttpStatus.SC_OK)
                {
                    final HttpEntity entity = response.getEntity();
                    final String entityContent = EntityUtils.toString(entity);
                    final Gson tokenGson = new GsonBuilder().setFieldNamingPolicy(FieldNamingPolicy.LOWER_CASE_WITH_UNDERSCORES).create();
                    final AccessTokenImpl tokenResponse = tokenGson.fromJson(entityContent, AccessTokenImpl.class);

                    EntityUtils.consume(entity);
                    return tokenResponse;
                }
                else
                {
                    final HttpEntity entity = response.getEntity();
                    final String responseString = EntityUtils.toString(entity, "UTF-8");
                    final HttpResponseException responseException =
                            new HttpResponseException(response.getStatusLine().getStatusCode(),
                                    responseString);

                    EntityUtils.consume(entity);
                    throw responseException;
                }
            }
            finally
            {
                response.close();
            }
        }
        catch (final IOException ex)
        {
            throw new AuthorizationHelperException("Caught exception while executing request: ", ex);
        }
    }

    public final com.avaya.collaboration.authorization.TokenAggregate getTokenAggregate()
            throws HttpResponseException, AuthorizationHelperException
    {
        try (final CloseableHttpResponse response = (CloseableHttpResponse) httpClient.execute(httpPost))
        {
            if (response.getStatusLine().getStatusCode() == HttpStatus.SC_OK)
            {
                final HttpEntity entity = new BufferedHttpEntity(response.getEntity());

                final String entityContent = EntityUtils.toString(entity);
                final Gson tokenGson = new GsonBuilder().setFieldNamingPolicy(FieldNamingPolicy.LOWER_CASE_WITH_UNDERSCORES).create();
                final AccessTokenImpl accessToken = tokenGson.fromJson(entityContent, AccessTokenImpl.class);

                final Gson refreshGson = new GsonBuilder().setFieldNamingPolicy(FieldNamingPolicy.LOWER_CASE_WITH_UNDERSCORES).create();
                final RefreshTokenImpl refreshToken = refreshGson.fromJson(entityContent, RefreshTokenImpl.class);

                EntityUtils.consume(entity);

                return new TokenAggregate()
                {
                    @Override
                    public RefreshToken getRefreshToken()
                    {
                        return refreshToken;
                    }

                    @Override
                    public AccessToken getAccessToken()
                    {
                        return accessToken;
                    }
                };
            }
            else
            {
                final HttpEntity entity = response.getEntity();
                final String responseString = EntityUtils.toString(entity, "UTF-8");
                final HttpResponseException responseException =
                        new HttpResponseException(response.getStatusLine().getStatusCode(),
                                responseString);

                EntityUtils.consume(entity);
                throw responseException;
            }
        }
        catch (final IOException ex)
        {
            throw new AuthorizationHelperException("Caught exception while executing request: ", ex);
        }
    }
}
