/****************************************************************************
 * Copyright Avaya Inc., All Rights Reserved.
 ****************************************************************************/
package com.avaya.zephyr.sdk.authorization.samples.client.interceptors;

import java.io.FileInputStream;
import java.io.IOException;
import java.io.InputStream;
import java.security.KeyStore;
import java.security.KeyStoreException;
import java.security.NoSuchAlgorithmException;
import java.security.cert.CertificateException;
import java.util.Calendar;
import java.util.Date;

import javax.servlet.http.Cookie;
import javax.servlet.http.HttpServletRequest;
import javax.servlet.http.HttpServletResponse;

import org.apache.commons.lang.StringUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.core.env.Environment;
import org.springframework.web.servlet.HandlerInterceptor;
import org.springframework.web.servlet.ModelAndView;

import com.avaya.collaboration.authorization.AccessToken;
import com.avaya.collaboration.authorization.AuthorizationClientHelper;
import com.avaya.collaboration.authorization.AuthorizationHelperException;
import com.avaya.collaboration.authorization.HttpResponseException;
import com.avaya.zephyr.services.sample_services.Authorization.types.CurrentUser;
import com.avaya.zephyr.services.sample_services.Authorization.types.UserSession;
import com.fasterxml.jackson.core.JsonProcessingException;
import com.fasterxml.jackson.databind.ObjectMapper;

/**
 * AccessTokenInterceptor checks if a request has cookie named ${sessionCookieName}
 * If so, it allows the request to proceed - which presents the index page to the user.
 *
 * If the cookie is not present, but if the request contains an authorization code,
 * it uses the {@link AuthorizationClientHelper} library to get an access token using the code.
 * The access token is then used to set a cookie named ${sessionCookieName} as a response header
 * and the flow is redirected to the index page.
 *
 * Upon redirection, this interceptor is again called, which will check for the cookie and allow
 * the request to proceed.
 *
 * @author Avaya
 */
public class AccessTokenInterceptor implements HandlerInterceptor
{
    private static final String AUTH_CODE_GRANT_TYPE = "code";
    private final Environment env;

    private AuthorizationClientHelper authorizationClientHelper;
    private final Logger logger = LoggerFactory.getLogger(AccessTokenInterceptor.class);

    public AccessTokenInterceptor(final Environment env)
    {
        this.env = env;

        try (InputStream keyStoreStream = new FileInputStream(env.getProperty("clientKeystorePath"));
                InputStream trustStoreStream = new FileInputStream(env.getProperty("clientTruststorePath")))
        {
            final KeyStore clientKeyStore = KeyStore.getInstance("JKS");
            clientKeyStore.load(keyStoreStream, env.getProperty("clientKeystorePassword").toCharArray());

            final KeyStore clientTrustStore = KeyStore.getInstance("JKS");
            clientTrustStore.load(trustStoreStream, env.getProperty("clientTruststorePassword").toCharArray());

            buildAuthorizationClientHelper(clientKeyStore, clientTrustStore);
        }
        catch (AuthorizationHelperException | NoSuchAlgorithmException | CertificateException | IOException | KeyStoreException e)
        {
            logger.error("AccessTokenInterceptor: Caught exception during init. Application might not perform as intended.", e);
        }
    }

    /*
     * Retrieves endpoint and keystore related attributes from application.properties file
     * and builds the client helper.
     */
    private void buildAuthorizationClientHelper(final KeyStore clientKeyStore, final KeyStore clientTrustStore) throws AuthorizationHelperException
    {
        authorizationClientHelper = new AuthorizationClientHelper.Builder()
                .tokenEndpoint(env.getProperty("tokenEndpointURL"))
                .clientIdentifier(env.getProperty("clientId"))
                .keyStore(clientKeyStore, env.getProperty("clientKeystorePassword"), env.getProperty("clientCertificateAlias"))
                .trustStore(clientTrustStore)
                .build();
    }

    @Override
    public boolean preHandle(final HttpServletRequest servletRequest, final HttpServletResponse servletResponse, final Object arg2) throws IOException
    {
        if (hasValidSessionCookie(servletRequest)) // Checks for a session cookie in the request.
        {
            return true;
        }
        else if (hasAuthorizationCode(servletRequest)) // Checks for the authorization code request parameter in the request.
        {
            fetchAccessTokenAndRedirectWithCookie(servletRequest, servletResponse);
        }
        return false;
    }

    private boolean hasValidSessionCookie(final HttpServletRequest httpServletRequest)
    {
        final Cookie[] cookies = httpServletRequest.getCookies();
        if (cookies != null && cookies.length > 0)
        {
            for (final Cookie cookie : cookies)
            {
                if (cookie.getName().equals(env.getProperty("sessionCookieName")))
                {
                    return isSessionValid(cookie.getValue());
                }
            }
            return false;
        }
        else
        {
            return false;
        }
    }

    private boolean isSessionValid(final String value)
    {
        try
        {
            final UserSession userSession = new ObjectMapper().readValue(value, UserSession.class);
            if(userSession != null)
            {
                final CurrentUser currentUser = userSession.getCurrentUser();
                if(currentUser != null && currentUser.getAuthdata() != null)
                {
                    return true;
                }
            }
        }
        catch (final IOException e)
        {
            logger.error("AccessTokenInterceptor: Caught exception during session validation", e);
        }
        return false;
    }

    private boolean hasAuthorizationCode(final HttpServletRequest servletRequest)
    {
        return StringUtils.isNotBlank(servletRequest.getParameter(AUTH_CODE_GRANT_TYPE));
    }

    private void fetchAccessTokenAndRedirectWithCookie(final HttpServletRequest servletRequest, final HttpServletResponse servletResponse)
            throws IOException
    {
        try
        {
            final AccessToken accessToken = authorizationClientHelper.getAccessTokenForUser(servletRequest);
            if (accessToken != null)
            {
                final String userSession = buildUserSession(accessToken);
                redirectWithCookie(userSession, accessToken.getExpiresIn(), servletRequest, servletResponse);
            }
        }
        catch (AuthorizationHelperException | HttpResponseException e)
        {
            logger.error("AccessTokenInterceptor: Caught exception when fetching access token", e);
            servletResponse.sendError(HttpServletResponse.SC_INTERNAL_SERVER_ERROR, e.getMessage());
        }
    }

    private String buildUserSession(final AccessToken accessToken) throws JsonProcessingException
    {
        final CurrentUser currentUser = new CurrentUser();
        currentUser.setUsername(accessToken.getSubject());
        currentUser.setAuthdata(accessToken.toString());
        currentUser.setExpiry(getExpiryTime(accessToken.getExpiresIn()));

        final UserSession userSession = new UserSession();
        userSession.setCurrentUser(currentUser);

        return new ObjectMapper().writeValueAsString(userSession);
    }

    private String getExpiryTime(final int expiryInSeconds)
    {
        final Calendar cal = Calendar.getInstance();
        cal.setTime(new Date());
        cal.add(Calendar.SECOND, expiryInSeconds);
        return String.valueOf(cal.getTimeInMillis());
    }

    private void redirectWithCookie(final String userSession, final int cookieExpiry, final HttpServletRequest httpServletRequest,
            final HttpServletResponse httpServletResponse)
                    throws IOException
    {
        final Cookie sessionCookie = new Cookie(env.getProperty("sessionCookieName"), userSession);
        sessionCookie.setMaxAge(cookieExpiry);

        final String clientRedirectUri = httpServletRequest.getRequestURL().toString();
        httpServletResponse.addCookie(sessionCookie);
        httpServletResponse.sendRedirect(clientRedirectUri);
    }

    @Override
    public void afterCompletion(final HttpServletRequest arg0, final HttpServletResponse arg1, final Object arg2, final Exception arg3)
    {
    }

    @Override
    public void postHandle(final HttpServletRequest arg0, final HttpServletResponse arg1, final Object arg2, final ModelAndView arg3)
    {
    }

    // for JUnits
    AuthorizationClientHelper getAuthorizationClientHelper()
    {
        return authorizationClientHelper;
    }
}