/****************************************************************************
 * Copyright Avaya Inc., All Rights Reserved.
 ****************************************************************************/
package com.avaya.zephyr.sdk.authorization.samples.client.interceptors;

import java.io.IOException;
import java.net.URISyntaxException;
import java.nio.ByteBuffer;
import java.util.UUID;

import javax.servlet.ServletRequest;
import javax.servlet.http.Cookie;
import javax.servlet.http.HttpServletRequest;
import javax.servlet.http.HttpServletResponse;

import org.apache.commons.codec.binary.Base64;
import org.apache.commons.lang.StringUtils;
import org.apache.http.client.utils.URIBuilder;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.core.env.Environment;
import org.springframework.web.servlet.HandlerInterceptor;
import org.springframework.web.servlet.ModelAndView;

/**
 * AuthorizationCodeInterceptor checks if a request has cookie named ${sessionCookieName}
 * If so, it allows the request to proceed, if not, it checks if the request contains
 * an authorization code parameter.
 *
 * If an authorization code is present, it allows the request to proceed. If not, it redirects
 * the flow to the Authorization Service to fetch an authorization code.
 *
 * @author Avaya
 */
public class AuthorizationCodeInterceptor implements HandlerInterceptor
{
    private static final String AUTH_CODE_GRANT_TYPE = "code";

    private final Environment env;

    private final Logger logger = LoggerFactory.getLogger(AuthorizationCodeInterceptor.class);

    public AuthorizationCodeInterceptor(final Environment env)
    {
        this.env = env;
    }

    @Override
    public boolean preHandle(final HttpServletRequest servletRequest, final HttpServletResponse servletResponse, final Object arg2)
    {
        if (hasSessionCookie(servletRequest)) // Checks for a session cookie in the request.
        {
            return true;
        }
        else if (hasAuthorizationCode(servletRequest)) // Checks for the authorization code request parameter in the request.
        {
            return true;
        }
        else
        {
            try
            {
                redirectToFetchAuthCode(servletRequest, servletResponse); // Redirects to the Authorization Service to fetch an auth code.
            }
            catch (URISyntaxException | IOException e)
            {
                logger.error("AuthorizationCodeInterceptor: Caught exception during init. Application might not perform as intended.", e);
            }
            return false;
        }
    }

    private boolean hasSessionCookie(final HttpServletRequest servletRequest)
    {
        final Cookie[] cookies = servletRequest.getCookies();
        if (cookies != null && cookies.length > 0)
        {
            for (final Cookie cookie : cookies)
            {
                if (cookie.getName().equals(env.getProperty("sessionCookieName")))
                {
                    return true;
                }
            }
            return false;
        }
        else
        {
            return false;
        }
    }

    private boolean hasAuthorizationCode(final ServletRequest servletRequest)
    {
        return StringUtils.isNotBlank(servletRequest.getParameter(AUTH_CODE_GRANT_TYPE));
    }

    /*
     * This method builds a URI to redirect the flow to the Authorization Service.
     * The URI also contains query parameters needed while conforming to 'Authorization Code Grant Type'
     * when making an authorization code request.
     */
    private void redirectToFetchAuthCode(final HttpServletRequest servletRequest, final HttpServletResponse servletResponse)
            throws URISyntaxException, IOException
    {
        final URIBuilder authorizeEndpointUriBuilder = new URIBuilder(env.getProperty("authorizeEndpointURL"));
        authorizeEndpointUriBuilder.addParameter("response_type", AUTH_CODE_GRANT_TYPE);
        authorizeEndpointUriBuilder.addParameter("client_id", env.getProperty("clientId"));
        authorizeEndpointUriBuilder.addParameter("redirect_uri", servletRequest.getRequestURL().toString());
        authorizeEndpointUriBuilder.addParameter("state", getStateParameter());
        servletResponse.sendRedirect(authorizeEndpointUriBuilder.build().toString());
    }

    /*
     * Generates a UUID for the state parameter
     */
    private String getStateParameter()
    {
        return uuidToBase64(UUID.randomUUID());
    }

    private static String uuidToBase64(final UUID uniqueId)
    {
        final ByteBuffer bb = ByteBuffer.wrap(new byte[16]);
        bb.putLong(uniqueId.getMostSignificantBits());
        bb.putLong(uniqueId.getLeastSignificantBits());
        return Base64.encodeBase64URLSafeString(bb.array());
    }

    @Override
    public void afterCompletion(final HttpServletRequest arg0, final HttpServletResponse arg1, final Object arg2, final Exception arg3)
    {
    }

    @Override
    public void postHandle(final HttpServletRequest arg0, final HttpServletResponse arg1, final Object arg2, final ModelAndView arg3)
    {
    }
}