/*****************************************************************************
 * Copyright Avaya Inc., All Rights Reserved.
 * THIS IS UNPUBLISHED PROPRIETARY SOURCE CODE OF Avaya Inc.
 * The copyright notice above does not evidence any actual or intended publication of such source code.
 * Some third-party source code components may have been modified from their original versions by Avaya Inc.
 * The modifications are Copyright Avaya Inc., All Rights Reserved.
 * Avaya  Confidential & Restricted. May not be distributed further without written permission of
 * the Avaya owner.
 ****************************************************************************/

package com.avaya.collaboration.authorization;

import java.util.List;

import javax.servlet.ServletRequest;

/**
 * Interface to be implemented by Access Token Providers.
 *
 * @author Avaya
 * @since 3.2
 */
public interface AccessTokenProvider
{
    /**
     * Authenticates the client with the Authorization Service and returns the token.
     *
     * @return AccessTokenResponse
     * @throws AuthorizationHelperException
     *             Indicates error conditions in the Authorization Helper APIs.
     * @throws HttpResponseException
     *             Indicates HTTP error responses propagated by the Authorization Helper APIs.
     */
    AccessToken getAccessToken() throws AuthorizationHelperException, HttpResponseException;

    /**
     * Same function as above, but allows client to ask for specific scopes in the request
     *
     * @param scopes
     *            The list of scopes the client wants to specify while making the token request.
     * @return AccessTokenResponse
     * @throws AuthorizationHelperException
     *             Indicates error conditions in the Authorization Helper APIs.
     * @throws HttpResponseException
     *             Indicates HTTP error responses propagated by the Authorization Helper APIs.
     */
    AccessToken getAccessToken(List<String> scopes) throws AuthorizationHelperException, HttpResponseException;

    /**
     * Authenticates the client and the user with the Authorization Service and returns the token.
     *
     * @param userName
     *            User name
     * @param userPassword
     *            User password
     * @return AccessTokenResponse
     * @throws AuthorizationHelperException
     *             Indicates error conditions in the Authorization Helper APIs.
     * @throws HttpResponseException
     *             Indicates HTTP error responses propagated by the Authorization Helper APIs.
     */
    AccessToken getAccessTokenForUser(String userName, String userPassword) throws AuthorizationHelperException,
            HttpResponseException;

    /**
     * Same function as above, but allows client to ask for specific scopes in the request
     *
     * @param userName
     *            User name
     * @param userPassword
     *            User password
     * @param scopes
     *            The list of scopes the client wants to specify while making the token request.
     * @return AccessTokenResponse
     * @throws AuthorizationHelperException
     *             Indicates error conditions in the Authorization Helper APIs.
     * @throws HttpResponseException
     *             Indicates HTTP error responses propagated by the Authorization Helper APIs.
     */
    AccessToken getAccessTokenForUser(String userName, String userPassword, List<String> scopes)
            throws AuthorizationHelperException, HttpResponseException;

    /**
     *
     * @param servletRequest
     *            The ServletRequest which wraps the request made by the browser.
     * @return AccessTokenResponse
     *
     * @throws HttpResponseException
     *             Indicates HTTP error responses propagated by the Authorization Helper APIs.
     * @throws AuthorizationHelperException
     *             Indicates error conditions in the Authorization Helper APIs.
     */
    AccessToken getAccessTokenForUser(ServletRequest servletRequest) throws AuthorizationHelperException, HttpResponseException;

    /**
     * API to release any system resources associated.
     *
     * @throws AuthorizationHelperException
     *             Indicates error conditions in the Authorization Helper APIs.
     */
    void shutDown() throws AuthorizationHelperException;

    /**
     *
     * @return {@link TokenAggregate}
     * @throws AuthorizationHelperException
     *             Indicates error conditions in the Authorization Helper APIs.
     * @throws HttpResponseException
     *             Indicates HTTP error responses propagated by the Authorization Helper APIs.
     */
    TokenAggregate getTokenAggregate() throws AuthorizationHelperException, HttpResponseException;

    /**
     * @param scopes
     *            The list of scopes the client wants to specify while making the token request.
     * @return {@link TokenAggregate}
     * @throws AuthorizationHelperException
     *             Indicates error conditions in the Authorization Helper APIs.
     * @throws HttpResponseException
     *             Indicates HTTP error responses propagated by the Authorization Helper APIs.
     */
    TokenAggregate getTokenAggregate(List<String> scopes) throws AuthorizationHelperException, HttpResponseException;

    /**
     * Authenticates the client (along with the user's auth code) with the Authorization Service and returns the token aggregate.
     *
     * @param servletRequest
     *            The authorization code granted to the user.
     * @return {@link TokenAggregate}
     * @throws AuthorizationHelperException
     *             Indicates error conditions in the Authorization Helper APIs.
     * @throws HttpResponseException
     *             Indicates HTTP error responses propagated by the Authorization Helper APIs.
     */
    TokenAggregate getTokenAggregateForUser(ServletRequest servletRequest) throws AuthorizationHelperException, HttpResponseException;

    /**
     * Authenticates the client and the user with the Authorization Service and returns the token aggregate.
     *
     * @param userName
     *            The user name for whom the token is being requested.
     * @param userPassword
     *            The user's password.
     * @return {@link TokenAggregate}
     * @throws AuthorizationHelperException
     *             Indicates error conditions in the Authorization Helper APIs.
     * @throws HttpResponseException
     *             Indicates HTTP error responses propagated by the Authorization Helper APIs.
     */
    TokenAggregate getTokenAggregateForUser(String userName, String userPassword)
            throws AuthorizationHelperException, HttpResponseException;

    /**
     * Authenticates the client and the user with the Authorization Service and returns the token aggregate. Also allows client to ask for
     * specific scopes in the request.
     *
     * @param userName
     *            The user name for whom the token is being requested.
     * @param userPassword
     *            The user's password.
     * @param scopes
     *            The list of scopes the client wants to specify while making the token request.
     * @return {@link TokenAggregate}
     * @throws AuthorizationHelperException
     *             Indicates error conditions in the Authorization Helper APIs.
     * @throws HttpResponseException
     *             Indicates HTTP error responses propagated by the Authorization Helper APIs.
     */
    TokenAggregate getTokenAggregateForUser(String userName, String userPassword, List<String> scopes)
            throws AuthorizationHelperException, HttpResponseException;

    /**
     * @param refreshToken
     *            A refresh token obtained previously by the client by calling {@link #getTokenAggregate()} or
     *            {@link #getTokenAggregate(List)}
     *
     * @return {@link TokenAggregate}
     * @throws AuthorizationHelperException
     *             Indicates error conditions in the Authorization Helper APIs.
     * @throws HttpResponseException
     *             Indicates HTTP error responses propagated by the Authorization Helper APIs.
     */
    TokenAggregate getTokenAggregate(String refreshToken) throws AuthorizationHelperException, HttpResponseException;
}
