/* Copyright 2014 Avaya LLC. All rights reserved. */
package sample;

import java.util.ArrayList;
import java.util.List;

import javax.ws.rs.client.Client;
import javax.ws.rs.client.Entity;
import javax.ws.rs.client.Invocation.Builder;
import javax.ws.rs.client.WebTarget;
import javax.ws.rs.core.MediaType;
import javax.ws.rs.core.Response;

import org.junit.Test;

import com.avaya.mgmt.dmodel.nrp.rs.DialPattern;
import com.avaya.mgmt.dmodel.nrp.rs.DialPatterns;
import com.avaya.mgmt.dmodel.nrp.rs.RoutingPolicy;
import com.avaya.mgmt.dmodel.nrp.rs.RsEntityLink;

/*
 * Before this can be run make sure to add a routing policy and modify below POLICY to that routing policy name.
 */

public class DialPatternExample {
	final static String POLICY = "REST policy";

	@Test
	public void testDialPatternGet() throws Exception {
		Client client = Common.getClient();

		/*
		 * Query all dialpatterns.
		 */
		WebTarget target = client.target(Common.URLBASE).path("dialpatterns");
		Response response = target.request(MediaType.APPLICATION_XML).get();
		Common.checkError(response);

		/*
		 * In this case we are manually telling what entity type is to be
		 * unmarshalled. Many clients allow registration of the JAXB mapping
		 * classes so this is not needed.
		 */
		DialPatterns dialpatterns = response.readEntity(DialPatterns.class);
		System.out.println("Recieved " + dialpatterns.getCount() + " of "
				+ dialpatterns.getTotalcount() + " dialpatterns");

		/*
		 * Now take first dialpattern found and GET based on reference URI.
		 */
		DialPattern dialpattern = dialpatterns.getList().get(0);
		target = client.target(dialpattern.getLink().getHref());
		response = target.request(MediaType.APPLICATION_XML).get();
		Common.checkError(response);

		DialPattern dialpattern2 = response.readEntity(DialPattern.class);
		System.out.println("Recieved " + dialpattern2.getName());
		System.out.println("DialPattern has ETag: "
				+ response.getHeaderString("ETag"));
	}

	@Test
	public void testDialPatternAddModifyDelete() throws Exception {
		Client client = Common.getClient();

		/*
		 * Add a new dialpattern, if use the Routing GUI to add an entry you
		 * will notice that pattern/min/max/domain/location are the key values
		 * for dial pattern uniqueness. And also the unique identifiers for the
		 * query key.
		 * 
		 * Make sure to get the pattern in first. This JAXB class is pretty
		 * finicky.
		 */
		DialPattern dialpattern = new DialPattern();
		dialpattern.setDigitpattern("222");
		dialpattern.setMindigits(3);
		dialpattern.setMaxdigits(8);
		dialpattern.setTreatasemergency(false);
		dialpattern.setNotes("Notes for a REST dial pattern");

		/*
		 * Now we need to do a lookup to find our policy URI
		 */
		WebTarget target = client.target(Common.URLBASE)
				.path("routingpolicies").path("name").path(POLICY);
		Builder request = target.request(MediaType.APPLICATION_XML).accept(
				MediaType.APPLICATION_XML);
		Response response = request.get();
		Common.checkError(response);

		/*
		 * This is actually the hard but consistent way to do this.
		 * 
		 * An alternate would be to use the name we already have and not do the
		 * round trip request:
		 * 
		 * RsLink link = new RsLink(); link.setHRefName(POLICY); RsEntityLink
		 * elink = new RsEntityLink(link);
		 */
		RoutingPolicy policy = response.readEntity(RoutingPolicy.class);
		List<RsEntityLink> policies = new ArrayList<RsEntityLink>();
		policies.add(new RsEntityLink(policy.getLink()));
		dialpattern.setPolicies(policies);

		/*
		 * We are going to use this policy for all locations. This approach to
		 * expressing all locations is quite different then the import mechanism
		 * of inputing a null.
		 */
		dialpattern.setLocationall("ALL");

		target = client.target(Common.URLBASE).path("dialpatterns");
		request = target.request(MediaType.APPLICATION_XML).accept(
				MediaType.APPLICATION_XML);
		response = request.post(Entity.xml(dialpattern));
		Common.checkError(response);

		String etag = response.getHeaderString("ETag");
		DialPattern dialpatternAdd = response.readEntity(DialPattern.class);
		System.out.println("DialPattern " + dialpatternAdd.getDigitpattern()
				+ " added at URI " + dialpatternAdd.getLink().getHref());

		/*
		 * Modify a dialpattern, for java client it is simpler to fetch full
		 * dialpattern entity and PUT "replace" then build URI with matrix
		 * parameters.
		 * 
		 * DialPattern.getLink() is only populated on server side.
		 */
		target = client.target(dialpatternAdd.getLink().getHref());
		request = target.request(MediaType.APPLICATION_XML).accept(
				MediaType.APPLICATION_XML);
		request.header("ETag", etag);

		dialpattern.setNotes("A different and modified note");
		response = request.put(Entity.xml(dialpattern));
		Common.checkError(response);

		DialPattern dialpatternModified = response
				.readEntity(DialPattern.class);

		/*
		 * Now delete. No payload is returned. ETag is ignored for deletes.
		 */
		target = client.target(dialpatternModified.getLink().getHref());
		request = target.request(MediaType.APPLICATION_XML).accept(
				MediaType.APPLICATION_XML);
		response = request.delete();
		Common.checkError(response);

		/*
		 * Now a get lookup based on unique lookup attributes documented in the
		 * API reference: pattern/min/max/location/domain
		 */
		target = client.target(Common.URLBASE).path("dialpatterns");
		target = target.path(dialpatternModified.getDigitpattern()).path("" + dialpatternModified.getMindigits());
		target = target.path("" + dialpatternModified.getMaxdigits()).path("ALL").path("ALL");

		request = target.request(MediaType.APPLICATION_XML).accept(
				MediaType.APPLICATION_XML);
		response = request.get();
		System.out.println("We should get a 404 not found return " + response.getStatus());
	}
}
