/* Copyright 2014 Avaya LLC. All rights reserved. */
package sample;

import java.util.ArrayList;
import java.util.List;

import javax.ws.rs.client.Client;
import javax.ws.rs.client.Entity;
import javax.ws.rs.client.WebTarget;
import javax.ws.rs.client.Invocation.Builder;
import javax.ws.rs.core.MediaType;
import javax.ws.rs.core.Response;

import org.junit.Test;

import com.avaya.mgmt.dmodel.nrp.rs.Location;
import com.avaya.mgmt.dmodel.nrp.rs.LocationPattern;
import com.avaya.mgmt.dmodel.nrp.rs.Locations;

public class LocationExample {
	/*
	 * Simple query of locations a re-fetch of first location by entity id.
	 */
	@Test
	public void testLocationGet() throws Exception {
		Client client = Common.getClient();

		/*
		 * Query all locations.
		 */
		WebTarget target = client.target(Common.URLBASE).path("locations");
		Response response = target.request(MediaType.APPLICATION_XML).get();
		Common.checkError(response);

		/*
		 * In this case we are manually telling what entity type is to be
		 * unmarshalled. Many clients allow registration of the JAXB mapping
		 * classes so this is not needed.
		 */
		Locations locations = response.readEntity(Locations.class);
		System.out.println("Recieved " + locations.getCount() + " of "
				+ locations.getTotalcount() + " locations");

		/*
		 * Now take first location found and GET based on reference URI.
		 */
		Location location = locations.getList().get(0);
		target = client.target(location.getLink().getHref());
		response = target.request(MediaType.APPLICATION_XML).get();
		Common.checkError(response);

		Location location2 = response.readEntity(Location.class);
		System.out.println("Recieved " + location2.getName());
		System.out.println("Location has ETag: "
				+ response.getHeaderString("ETag"));
	}

	@Test
	public void testLocationAddModifyDelete() throws Exception {
		Client client = Common.getClient();

		/*
		 * Add a new location, and location pattern. Track ETag to force possible optimistic lock.
		 * If you look at the GUI, you will see numerous required fields which
		 * have default values as seen on the GUI are implicitly set by the REST
		 * interface in a similar fashion.
		 */
		Location location = new Location();
		location.setName("REST location");
		location.setNotes("Notes for a new location");
		
		/*
		 * And now add a pattern. Remember if you want to remove a pattern
		 * or modify pattern you have to work on that sub resource URL directly.
		 * 
		 * See segment below for deletion.
		 */
		LocationPattern pattern = new LocationPattern();
		pattern.setIpaddresspattern("192.168.*");
		List<LocationPattern> patterns = new ArrayList<LocationPattern>();
		patterns.add(pattern);
		location.setLocationPatterns(patterns);

		WebTarget target = client.target(Common.URLBASE).path("locations");
		Builder request = target.request(MediaType.APPLICATION_XML).accept(
				MediaType.APPLICATION_XML);
		Response response = request.post(Entity.xml(location));
		Common.checkError(response);

		String etag = response.getHeaderString("ETag");
		Location locationAdd = response.readEntity(Location.class);
		System.out.println("Location " + locationAdd.getName() + " added at URI "
				+ locationAdd.getLink().getHref());

		/*
		 * Modify a location, for java client it is simpler to fetch full location
		 * entity and PUT "replace" then build URI with matrix parameters.
		 * 
		 * Location.getLink() is only populated on server side.
		 */
		target = client.target(locationAdd.getLink().getHref());
		request = target.request(MediaType.APPLICATION_XML).accept(
				MediaType.APPLICATION_XML);
		request.header("ETag", etag);

		location.setNotes("A different and modified note");
		response = request.put(Entity.xml(location));
		Common.checkError(response);

		Location locationModified = response.readEntity(Location.class);
		
		/*
		 * Let's now delete just the ListenPort sub resource
		 */
		target = client.target(locationModified.getLocationPatterns().get(0).getLink().getHref());
		request = target.request(MediaType.APPLICATION_XML).accept(
				MediaType.APPLICATION_XML);
		response = request.delete();
		Common.checkError(response);

		/*
		 * Now delete. No payload is returned. ETag is ignored for deletes.
		 */
		target = client.target(locationModified.getLink().getHref());
		request = target.request(MediaType.APPLICATION_XML).accept(
				MediaType.APPLICATION_XML);
		response = request.delete();
		Common.checkError(response);

		/*
		 * Now a get query based on name, should be no results.
		 */
		target = client.target(Common.URLBASE).path("locations");
		target.queryParam("name", "new.com");
		request = target.request(MediaType.APPLICATION_XML).accept(
				MediaType.APPLICATION_XML);
		response = request.get();
		Locations locations = response.readEntity(Locations.class);
		System.out.println("We should get a 200 return" + response.getStatus()
				+ " and 0 results in query: " + locations.getCount());
	}
}
