/* Copyright 2014 Avaya LLC. All rights reserved. */
package sample;

import javax.ws.rs.client.Client;
import javax.ws.rs.client.Entity;
import javax.ws.rs.client.Invocation.Builder;
import javax.ws.rs.client.WebTarget;
import javax.ws.rs.core.MediaType;
import javax.ws.rs.core.Response;

import org.junit.Test;

import com.avaya.mgmt.dmodel.nrp.rs.Domain;
import com.avaya.mgmt.dmodel.nrp.rs.Domains;
import com.avaya.mgmt.dmodel.nrp.rs.RsError;

/**
 * Add a single domain via the GUI before running testDomainGet()
 */
public class DomainExample {

	/*
	 * Simple query of domains a re-fetch of first domain by entity id.
	 */
	@Test
	public void testDomanGet() throws Exception {
		Client client = Common.getClient();

		/*
		 * Query all domains.
		 */
		WebTarget target = client.target(Common.URLBASE).path("domains");
		Response response = target.request(MediaType.APPLICATION_XML).get();
		Common.checkError(response);

		/*
		 * In this case we are manually telling what entity type is to be
		 * unmarshalled. Many clients allow registration of the JAXB mapping
		 * classes so this is not needed.
		 */
		Domains domains = response.readEntity(Domains.class);
		System.out.println("Recieved " + domains.getCount() + " of "
				+ domains.getTotalcount() + " domains");

		/*
		 * Now take first domain found and GET based on reference URI.
		 */
		Domain domain = domains.getList().get(0);
		target = client.target(domain.getLink().getHref());
		response = target.request(MediaType.APPLICATION_XML).get();
		Common.checkError(response);

		Domain domain2 = response.readEntity(Domain.class);
		System.out.println("Recieved " + domain2.getName());
		System.out.println("Domain has ETag: "
				+ response.getHeaderString("ETag"));
	}

	@Test
	public void testDomainAddModifyDelete() throws Exception {
		Client client = Common.getClient();

		/*
		 * Add a new domain, track ETag to force possible optimistic lock
		 */
		Domain domain = new Domain();
		// Force error, invalid name length.
		domain.setName("ba");
		domain.setNotes("Notes for a new domain");

		WebTarget target = client.target(Common.URLBASE).path("domains");
		Builder request = target.request(MediaType.APPLICATION_XML).accept(
				MediaType.APPLICATION_XML);
		Response response = request.post(Entity.xml(domain));
		System.out.println("Error length: " + response.getLength());
		if (response.getLength() > 0) {
			System.out.println(response.toString());
			RsError err = response.readEntity(RsError.class);
			Common.printRsError(err);
		}

		domain.setName("rest.domain.com");
		request = target.request(MediaType.APPLICATION_XML).accept(
				MediaType.APPLICATION_XML);
		response = request.post(Entity.xml(domain));

		String etag = response.getHeaderString("ETag");
		Domain domainAdd = response.readEntity(Domain.class);
		System.out.println("Domain " + domainAdd.getName() + " added at URI "
				+ domainAdd.getLink().getHref());

		/*
		 * Modify a domain, for java client it is simpler to fetch full domain
		 * entity and PUT "replace" then build URI with matrix parameters.
		 * 
		 * Domain.getLink() is only populated on server side.
		 */
		target = client.target(domainAdd.getLink().getHref());
		request = target.request(MediaType.APPLICATION_XML).accept(
				MediaType.APPLICATION_XML);
		request.header("ETag", etag);

		domain.setNotes("A different and modified note");
		response = request.put(Entity.xml(domain));
		Common.checkError(response);

		Domain domainModified = response.readEntity(Domain.class);

		/*
		 * Now delete. No payload is returned. ETag is ignored for deletes.
		 */
		target = client.target(domainModified.getLink().getHref());
		request = target.request(MediaType.APPLICATION_XML).accept(
				MediaType.APPLICATION_XML);
		response = request.delete();
		Common.checkError(response);

		/*
		 * Now a get query based on name, should be no results.
		 */
		target = client.target(Common.URLBASE).path("domains");
		target.queryParam("name", "new.com");
		request = target.request(MediaType.APPLICATION_XML).accept(
				MediaType.APPLICATION_XML);
		response = request.get();
		Domains domains = response.readEntity(Domains.class);
		System.out.println("We should get a 200 return" + response.getStatus()
				+ " and 0 results in query: " + domains.getCount());
	}

}
