/* Copyright 2014 Avaya LLC. All rights reserved. */
package sample;

import javax.ws.rs.client.Client;
import javax.ws.rs.client.Entity;
import javax.ws.rs.client.WebTarget;
import javax.ws.rs.client.Invocation.Builder;
import javax.ws.rs.core.MediaType;
import javax.ws.rs.core.Response;

import org.junit.Test;

import com.avaya.mgmt.dmodel.nrp.rs.Adaptation;
import com.avaya.mgmt.dmodel.nrp.rs.Adaptations;
import com.avaya.mgmt.dmodel.nrp.rs.IngressAdaptation;
import com.avaya.mgmt.dmodel.nrp.rs.RsError;

public class AdaptationExample {
	/*
	 * Simple query of adaptations a re-fetch of first adaptation by entity id.
	 */
	@Test
	public void testAdaptationGet() throws Exception {
		Client client = Common.getClient();

		/*
		 * Query all adaptations.
		 */
		WebTarget target = client.target(Common.URLBASE).path("adaptations");
		Response response = target.request(MediaType.APPLICATION_XML).get();
		Common.checkError(response);

		/*
		 * In this case we are manually telling what entity type is to be
		 * unmarshalled. Many clients allow registration of the JAXB mapping
		 * classes so this is not needed.
		 */
		Adaptations adaptations = response.readEntity(Adaptations.class);
		System.out.println("Recieved " + adaptations.getCount() + " of "
				+ adaptations.getTotalcount() + " adaptations");

		/*
		 * Now take first adaptation found and GET based on reference URI.
		 */
		Adaptation adaptation = adaptations.getList().get(0);
		target = client.target(adaptation.getLink().getHref());
		response = target.request(MediaType.APPLICATION_XML).get();
		Common.checkError(response);

		Adaptation adaptation2 = response.readEntity(Adaptation.class);
		System.out.println("Recieved " + adaptation2.getName());
		System.out.println("Adaptation has ETag: "
				+ response.getHeaderString("ETag"));
	}

	@Test
	public void testAdaptationAddModifyDelete() throws Exception {
		Client client = Common.getClient();

		/*
		 * Add a new adaptation, track ETag to force possible optimistic lock
		 */
		Adaptation adaptation = new Adaptation();
		adaptation.setName("REST added adaptation");
		adaptation.setNotes("Notes for a new adaptation");

		WebTarget target = client.target(Common.URLBASE).path("adaptations");
		Builder request = target.request(MediaType.APPLICATION_XML).accept(
				MediaType.APPLICATION_XML);
		Response response = request.post(Entity.xml(adaptation));
		Common.checkError(response);

		String etag = response.getHeaderString("ETag");
		Adaptation adaptationAdd = response.readEntity(Adaptation.class);
		System.out.println("Adaptation " + adaptationAdd.getName()
				+ " added at URI " + adaptationAdd.getLink().getHref());

		/*
		 * Modify a adaptation, for java client it is simpler to fetch full
		 * adaptation entity and PUT "replace" then build URI with matrix
		 * parameters.
		 * 
		 * Adaptation.getLink() is only populated on server side.
		 */
		target = client.target(adaptationAdd.getLink().getHref());
		request = target.request(MediaType.APPLICATION_XML).accept(
				MediaType.APPLICATION_XML);
		request.header("ETag", etag);

		adaptation.setNotes("A different and modified note");
		response = request.put(Entity.xml(adaptation));
		Common.checkError(response);

		Adaptation adaptationModified = response.readEntity(Adaptation.class);

		/*
		 * Now do some sub resource handling. Add an ingress adapatation, then
		 * modify it.
		 */
		IngressAdaptation ingress = new IngressAdaptation();
		ingress.setMatchingpattern("123");
		ingress.setMindigits(3);
		ingress.setMaxdigits(8);
		ingress.setInsertdigits("9");
		ingress.setAddressToModify("both");
		target = client.target(adaptationModified.getLink().getHref()).path("ingressadaptations");
		request = target.request(MediaType.APPLICATION_XML).accept(
				MediaType.APPLICATION_XML);
		response = request.post(Entity.xml(ingress));
		
		/*
		 * We are expecting a validation error. Errors come with in a RsError entity.
		 */
		System.out.println("Error length: " + response.getLength());
		if (response.getLength() > 0) {
			System.out.println(response.toString());
			RsError err = response.readEntity(RsError.class);
			Common.printRsError(err);
		}
		
		ingress.setDeletedigits(0);
		
		adaptationModified = response.readEntity(Adaptation.class);

		/*
		 * Now delete. No payload is returned. ETag is ignored for deletes.
		 */
		target = client.target(adaptationModified.getLink().getHref());
		request = target.request(MediaType.APPLICATION_XML).accept(
				MediaType.APPLICATION_XML);
		response = request.delete();
		Common.checkError(response);

		/*
		 * Now a get query based on name, should be no results.
		 */
		target = client.target(Common.URLBASE).path("adaptations");
		target.queryParam("name", "new.com");
		request = target.request(MediaType.APPLICATION_XML).accept(
				MediaType.APPLICATION_XML);
		response = request.get();
		Adaptations adaptations = response.readEntity(Adaptations.class);
		System.out.println("We should get a 200 return" + response.getStatus()
				+ " and 0 results in query: " + adaptations.getCount());
	}

}
