/**
 * alertsService adds the messages received from the server to the alertsData factory which persists this information
 * where required on the view.
 * alertsService provides helper methods to format console messages with source of message and a timestamp 
 *
 */

angular.module('refClientJS.alerts', [])
    .factory("alertsService", ["alertsData", function (alertsData) {

        var alertsService = {};

        /**
        * Adds an alert to an array of alerts fisplayed on the client
        * 
        */
        alertsService.addAlert = function (newAlert, newType) {
            if (newAlert == null) {
                return;
            }
            alertsData.alerts.push({
                msg: newAlert,
                type: newType
            });
        };
        /**
        * Alert has been acknowledged by the user so is removed from the array of alerts to be displayed on the screen
        * 
        */
        alertsService.closeAlert = function (index) {
            alertsData.alerts.splice(index, 1);
        };
        
        /**
        * All alerts have been acknowledged by the user
        * 
        */
        alertsService.clearAlerts = function () {
            alertsData.alerts = [];
        };

        /**
         * Returns a formatted message prepending the source of the message
         * @type {String}
        */
        function formatLogMessage(source, message) {
            var clientName = 'JSDemoClient';

            return clientName + ": " + source + ": " + message.toString();
        }

        /**
        * Returns a timestamp with the format "m/d/yy h:MM:ss TT"
        * @type {Date}
        */
        function timeStamp() {
        
            var i,
            now = new Date(),
            date = [ now.getFullYear(), now.getMonth() + 1, now.getDate() ],
            time = [ now.getHours(), now.getMinutes(), now.getSeconds() ],
            msec = now.getMilliseconds();

            // If month or day are less than 10, add a zero
            for (i = 1; i < 3; i++) {
                if (date[i] < 10) {
                    date[i] = "0" + date[i];
                }
            }
        
            // If hours, minutes or seconds are less than 10, add a zero
            for (i = 0; i < 3; i++) {
                if (time[i] < 10) {
                    time[i] = "0" + time[i];
                }
            }
        
            if (msec < 10) {
                msec = "00" + msec;
            } else if (msec < 100) {
                msec = "0" + msec;
            }

            // Return the formatted string
            return date.join("-") + " " + time.join(":") + "." + msec;
        }
    
        /**
        * Logs an error message to the console 
        *
        * @param   {Object}  source The object that will be the source of the trace logs.  This object must
        *                    provide a "LoggingName" property that identifies the object that is producing
        *                    the trace log.  This name will be prefixed to any trace message generated.
        * @param {String} message The message text to be logged as an error.
        *
        * @returns {void}
        */
        alertsService.logerror = function (source, message) {
            message = formatLogMessage(source, message);
            console.error(timeStamp() + " " + message);
        };
    
        /**
        * Logs a warning message to the console 
        *
        * @param   {Object}  source The object that will be the source of the trace logs.  This object must
        *                    provide a "LoggingName" property that identifies the object that is producing
        *                    the trace log.  This name will be prefixed to any trace message generated.
        * 
        * @param {String} message The message text to be logged as a warning.
        *
        * @returns {void}
        */
        alertsService.logwarn = function (source, message) {
            message = formatLogMessage(source, message);
            console.warn(timeStamp() + " " + message);
        };
    
        /**
         * Logs an informational message to the console and the server (if possible and in a lab environment).
         *
         * @param   {Object}  source The object that will be the source of the trace logs.  This object must
         *                    provide a "LoggingName" property that identifies the object that is producing
         *                    the trace log.  This name will be prefixed to any trace message generated.
         * 
         * @param {String} message The message text to be logged.
         *
         * @returns {void}
         */
        alertsService.loginfo = function (source, message) {
            message = formatLogMessage(source, message);
            console.info(timeStamp() + " " + message);
        };
    
        /**
        * Logs an error message to the console only.
        *
        * @param   {Object}  source The object that will be the source of the trace logs.  This object must
        *                    provide a "LoggingName" property that identifies the object that is producing
        *                    the log message.  This name will be prefixed to any trace message generated.
        * @param {String} message The message text to be logged.
        *
        * @returns {void}
        */
        alertsService.error = function (source, message) {
            console.error(timeStamp() + " " + formatLogMessage(source, message));
        };
    
        /**
         * 
         * Logs a warning message to the console only.
         *
         * @param   {Object}  source The object that will be the source of the trace logs.  This object must
         *                    provide a "LoggingName" property that identifies the object that is producing
         *                    the trace log.  This name will be prefixed to any trace message generated.
         * 
         * @param {String} message The message text to be logged.
         *
         * @returns {void}
        */
        alertsService.warn = function (source, message) {
            console.warn(timeStamp() + " " + formatLogMessage(source, message));
        };
    
        /**
        * Logs an informational message to the console only.
        *
        * @param   {Object}  source The object that will be the source of the trace logs.  This object must
        *                    provide a "LoggingName" property that identifies the object that is producing
        *                    the trace log.  This name will be prefixed to any trace message generated.        
        * @param {String} message The message text to be logged.
        *
        * @returns {void}
        */
        alertsService.info = function (source, message) {
            console.info(timeStamp() + " " + formatLogMessage(source, message));
        };
    
        /**
         * Logs a debug message to the console only.
         * 
         * @param   {Object}  source The object that will be the source of the trace logs.  This object must
         *                    provide a "LoggingName" property that identifies the object that is producing
         *                    the trace log.  This name will be prefixed to any trace message generated. 
         *
         * @param {String} message The message text to be logged.
         *
         * @returns {void}
         */
        alertsService.log = function (source, message) {
            console.log(timeStamp() + " " + formatLogMessage(source, message));
        };


        return alertsService;

    }])
    .factory("alertsData", function () {
        var alerts = [];
        window.alertsDataScope = alerts;
        return {
            'alerts': alerts
        };
    });