/* jshint -W097 */
"use strict";

angular.module('refClientJS.customerDetails', ['refClientJS.alerts'])
.directive('customerDetails', ["alertsService", function(alertsService) {
    /**
     * The customer details directive is responsible for display and control of the customer details for both a customer on a live interaction 
     * and a customer returned through customer search.
     * The directive has an isolated scope which makes it possible to re-use the directive from multiple locations in the client application.
     * For the details of a customer on an interaction, there is a customer details directive per interaction and therefore an isolated scope per 
     * interaction.
     * For the details of a customer returned from customer search, there is one customer details directive.
     */
    var customerDetailsDirective = {
        restrict: 'E',
        /*
         * Isolated scope for each instance of the customer details directive.
         */
        scope: {
            /**
             * The customer object is injected as an attribute to the customer details directive. The reference to the customer object is therefore 
             * passed in from the parent scope and here it is added to the directives isolated scope. 
             * The customer object is injected when displaying details for a customer returned from a customer search or from a customer 
             * on a live interaction.
             */
            selectedCustomer: '=customer',
            /*
             * The interaction object is injected as an attribute to the customer details directive. The reference to the interaction object is 
             * therefore passed in from the parent scope and here it is added to the directives isolated scope. 
             * The interaction object is injected when displaying details of a customer on a live interaction only. It is used to invoke the methods 
             * on the CCS API interaction object (such as interaction.UpdateCustomerPhone)
             */      
            selectedInteraction: '=interaction'
        },
        templateUrl: 'customer/customerDetails/customer-details.tpl.html',
        link: link
    };
    
    function link(scope) {
                
        alertsService.log('customerDetails: link()', ''); 
        
        // scope objects to store new email, phone or address entries that are being added to this customer
        scope.newPhone = {};
        scope.newEmailAddress = {};        
        scope.newPostalAddress = {};
        
        scope.showNewPhone = false;
        scope.showNewEmail = false;
        scope.showNewAddress = false;  
        
        /*
         * Asynchronous callback method invoked from the CCS API containing the customer object that was updated
         */
        scope.onCustomerUpdateReceived = function onCustomerResponseReceived(notifyType, notifyData) {
            if (notifyType === MobileCCS.NotifyTypes.RESPONSE) {
                alertsService.log("customerDetails: onCustomerUpdateReceived()", " " + notifyType);
                console.log(notifyData);
                scope.$apply(function() {
                    scope.selectedCustomer = notifyData;
                });                
            } else if (notifyType === MobileCCS.NotifyTypes.ERROR) {
                // handle error in response to update customer details, display error message
            }
        };
        
        scope.showNewCustomerPhone = function showNewCustomerPhone(showNewPhone) {
            scope.showNewPhone = showNewPhone;            
        };
        
        /**
         * If there is a selected interaction on the directive scope, update the customer phone using the CCS API Interaction object.
         * If there is no interaction on the directive scope, then this directive is displaying details of a customer returned 
         * from customer search. Update the customer phone using the static methods on the CCS API Customer object
         */
        scope.updateCustomerPhone = function updateCustomerPhone(phone) {
            if (scope.selectedInteraction !== undefined) {
                scope.selectedInteraction.UpdateCustomerPhone(phone);
            } else {
                MobileCCS.Customer.UpdateCustomerPhone(scope.onCustomerUpdateReceived, scope.selectedCustomer.ID, phone);
            }
        };  
        
        /**
         * If there is a selected interaction on the directive scope, add the customer phone using the CCS API Interaction object.
         * If there is no interaction on the directive scope, then this directive is displaying details of a customer returned 
         * from customer search. Add the customer phone using the static methods on the CCS API Customer object
         */
        scope.addCustomerPhone = function addCustomerPhone() {
            if (scope.selectedInteraction !== undefined) {
                scope.selectedInteraction.AddCustomerPhone(scope.newPhone);
            } else {
                MobileCCS.Customer.AddCustomerPhone(scope.onCustomerUpdateReceived, scope.selectedCustomer.ID, scope.newPhone);
            }
            scope.newPhone = {};
            scope.showNewPhone = false;
        };
        
        /**
         * If there is a selected interaction on the directive scope, remove the customer phone using the CCS API Interaction object.
         * If there is no interaction on the directive scope, then this directive is displaying details of a customer returned 
         * from customer search. Remove the customer phone using the static methods on the CCS API Customer object
         */
        scope.removeCustomerPhone = function removeCustomerPhone(phoneId) {
            if (scope.selectedInteraction !== undefined) {
                scope.selectedInteraction.RemoveCustomerPhone(phoneId);
            } else {
                MobileCCS.Customer.RemoveCustomerPhone(scope.onCustomerUpdateReceived, scope.selectedCustomer.ID, phoneId);
            }
        };        
                
        scope.showNewCustomerEmail = function showNewCustomerEmail(showNewEmail) {
            scope.showNewEmail = showNewEmail;                        
        };
        
        /**
         * If there is a selected interaction on the directive scope, update the customer email using the CCS API Interaction object.
         * If there is no interaction on the directive scope, then this directive is displaying details of a customer returned 
         * from customer search. Update the customer email using the static methods on the CCS API Customer object
         */
        scope.updateCustomerEmail = function updateCustomerEmail(email) {
            if (scope.selectedInteraction !== undefined) {
                scope.selectedInteraction.UpdateCustomerEmail(email);
            } else {
                MobileCCS.Customer.UpdateCustomerEmail(scope.onCustomerUpdateReceived, scope.selectedCustomer.ID, email);
            }
        };

        /**
         * If there is a selected interaction on the directive scope, add the customer email using the CCS API Interaction object.
         * If there is no interaction on the directive scope, then this directive is displaying details of a customer returned 
         * from customer search. Add the customer email using the static methods on the CCS API Customer object
         */
        scope.addCustomerEmail = function addCustomerEmail() {
            if (scope.selectedInteraction !== undefined) {
                scope.selectedInteraction.AddCustomerEmail(scope.newEmailAddress);
            } else {
                MobileCCS.Customer.AddCustomerEmail(scope.onCustomerUpdateReceived, scope.selectedCustomer.ID, scope.newEmailAddress);
            }
            scope.newEmailAddress = {};
            scope.showNewEmail = false;
        };
        
        /**
         * If there is a selected interaction on the directive scope, remove the customer email using the CCS API Interaction object.
         * If there is no interaction on the directive scope, then this directive is displaying details of a customer returned 
         * from customer search. Remove the customer email using the static methods on the CCS API Customer object
         */
        scope.removeCustomerEmail = function removeCustomerEmail(emailId) {
            if (scope.selectedInteraction !== undefined) {
                scope.selectedInteraction.RemoveCustomerEmail(emailId);
            } else {
                MobileCCS.Customer.RemoveCustomerEmail(scope.onCustomerUpdateReceived, scope.selectedCustomer.ID, emailId);
            }
        };
        
        scope.showNewCustomerAddress = function showNewCustomerAddress(showNewAddress) {
            scope.showNewAddress = showNewAddress;                        
        };
              
        /**
         * If there is a selected interaction on the directive scope, update the customer address using the CCS API Interaction object.
         * If there is no interaction on the directive scope, then this directive is displaying details of a customer returned 
         * from customer search. Update the customer address using the static methods on the CCS API Customer object
         */              
        scope.updateCustomerAddress = function updateCustomerAddress(address) {
            if (scope.selectedInteraction !== undefined) {
                scope.selectedInteraction.UpdateCustomerAddress(address);
            } else {
                MobileCCS.Customer.UpdateCustomerAddress(scope.onCustomerUpdateReceived, scope.selectedCustomer.ID, address);
            }
        };

        /**
         * If there is a selected interaction on the directive scope, add the customer address using the CCS API Interaction object.
         * If there is no interaction on the directive scope, then this directive is displaying details of a customer returned 
         * from customer search. Add the customer address using the static methods on the CCS API Customer object
         */ 
        scope.addCustomerAddress = function addCustomerAddress() {
            if (scope.selectedInteraction !== undefined) {
                scope.selectedInteraction.AddCustomerAddress(scope.newPostalAddress);
            } else {
                MobileCCS.Customer.AddCustomerAddress(scope.onCustomerUpdateReceived, scope.selectedCustomer.ID, scope.newPostalAddress);
            }
            scope.newPostalAddress = {};
            scope.showNewAddress = false;
        };
        
        /**
         * If there is a selected interaction on the directive scope, remove the customer address using the CCS API Interaction object.
         * If there is no interaction on the directive scope, then this directive is displaying details of a customer returned 
         * from customer search. Remove the customer address using the static methods on the CCS API Customer object
         */ 
        scope.removeCustomerAddress = function removeCustomerAddress(addressId) {
            if (scope.selectedInteraction !== undefined) {
                scope.selectedInteraction.RemoveCustomerAddress(addressId);
            } else {
                MobileCCS.Customer.RemoveCustomerAddress(scope.onCustomerUpdateReceived, scope.selectedCustomer.ID, addressId);
            }
        };
        
        scope.addToDnc = function addToDnc(phoneId) {
            if (scope.selectedInteraction !== undefined && scope.selectedInteraction.ContactType === MobileCCS.ContactTypes.POMOutbound) {
                scope.selectedInteraction.AddToDNC(phoneId);
            }
        };
    }
    
    return customerDetailsDirective;
    
}]);