/* jshint -W097 */
"use strict";

customerHistory.$inject = ["interactionService", "alertsService"];
angular.module('refClientJS.customerHistory', ['refClientJS.alerts'])
.directive('customerHistory', customerHistory);
     
/**
 * The customer history directive is responsible for display and control of the customer history for both a customer on a live interaction 
 * and a customer returned through customer search.
 */
function customerHistory(interactionService, alertsService) {
            
    /**
     * The directive has an isolated scope which makes it possible to re-use the directive from multiple locations in the client application.
     * For the history of a customer on an interaction, there is a customer history directive per interaction and therefore an isolated scope per 
     * interaction.
     * For the history of a customer returned from a customer search, there is one customer history directive.
     */
    var customerHistoryDirective = {
        restrict: 'E',
        /*
         * Isolated scope for each instance of the customer history directive.
         */
        scope: {
            /**
             * The customer object is injected as an attribute to the customer history directive. The reference to the customer object is therefore 
             * passed in from the parent scope and here it is added to the directives isolated scope. 
             * The customer object is injected when displaying details for a customer returned from a customer search or from a customer 
             * on a live interaction.
             */
            selectedCustomer: '=customer',
            /*
             * The interaction object is injected as an attribute to the customer history directive. The reference to the interaction object is 
             * therefore passed in from the parent scope and here it is added to the directives isolated scope. 
             * The interaction object is injected when displaying details of a customer on a live interaction only. It is used to invoke the methods 
             * on the CCS API interaction object (such as interaction.GetCustomerHistory)
             */ 
            selectedInteraction: '=interaction'
        },
        templateUrl: 'customer/customerHistory/customer-history.tpl.html',
        link: link
    };
    
    function link(scope) {
        
        alertsService.log('customerHistory: link', '');      
        
        scope.customerHistory = undefined;
        scope.customerInteractionMediaList = undefined;
        
        scope.customerHistoryRequested = false;
    
        // angular-ui-grid components. these display and control the content of the customer history table
        scope.customerHistoryGridApi = undefined;
        scope.customerHistoryGridOptions = {       
            enableRowSelection: true,
            enableRowHeaderSelection: false,
            multiSelect: false,
            minRowsToShow: 5,               
            paginationPageSize: 5,
            paginationPageSizes: [5, 10, 20, 40],
            columnDefs: [
                { field: 'ID', enableColumnMenu: false },
                { field: 'ContactType', enableColumnMenu: false },
                { field: 'Skillset', enableColumnMenu: false },
                { field: 'Status', enableColumnMenu: false },
                { field: 'Subject', enableColumnMenu: false },
                { field: 'ArrivalTime', enableColumnMenu: false }            
            ],
            onRegisterApi: function(gridApi) {
                scope.customerHistoryGridApi = gridApi; 
                scope.customerHistoryGridApi.selection.on.rowSelectionChanged(scope, scope.onCustomerHistoryRowSelected);                       
            },
            // customer history contacts list
            data: []
        };       
    
        /*
         * Invoked when a customer interaction was selected from the customer history table
         */
        scope.onCustomerHistoryRowSelected = function onCustomerHistoryRowSelected(selectedRow) {
            alertsService.log('customerHistory: onCustomerHistoryRowSelected()', 'row selected: ' + selectedRow.entity.ID + ' isSelected?' + selectedRow.isSelected);

            if (selectedRow.isSelected === false) {
                // clear customer interaction media list
                scope.customerInteractionMediaList = undefined;
            } else {
                if (scope.customerInteractionMediaList === undefined || (selectedRow.entity.ID !== scope.customerInteractionMediaList.ContactID)) {
                    if (scope.selectedInteraction !== undefined) {
                        interactionService.getCustomerInteractionMedia(selectedRow.entity.ID, scope.selectedInteraction, scope.onCustomerInteractionMediaReceived);
                    } else {
                        MobileCCS.Customer.GetCustomerInteractionMedia(scope.onCustomerInteractionMediaReceived, scope.selectedCustomer.ID, selectedRow.entity.ID);
                    }
                }
            }
        };
        
        /*
         * Updates the directive scope with the customer interaction media list received from the CCS API
         */
        scope.onCustomerInteractionMediaReceived = function onCustomerInteractionMediaReceived(notifyType, notifyData) {
            alertsService.log('customerHistory: onCustomerInteractionMediaReceived()', ' ' + notifyType);
            console.log(notifyData);                
            if (notifyType === MobileCCS.NotifyTypes.RESPONSE) {
                
                scope.$apply(function() {
                    scope.customerInteractionMediaList = notifyData;                        
                });
                
            } else if (notifyType === MobileCCS.NotifyTypes.ERROR) {
                // handle error
            }
        };
        
        /*
         * Updates the customer history grid (table) based on the list of customer history objects returned from the CCS API
         */
        scope.onCustomerHistoryReceived = function onCustomerHistoryReceived(notifyType, notifyData) {
            alertsService.log("customerHistory: onCustomerHistoryReceived()", ' ' + notifyType);

            if (notifyType === MobileCCS.NotifyTypes.RESPONSE) {
                
                // an array of customer history objects for the specified customer     
                scope.$apply(function() {
                    // store the customer history
                    scope.customerHistory = notifyData;
                    // update the customer history grid with the received data so it can be displayed
                    scope.customerHistoryGridOptions.data = notifyData.Contacts;                            
                });

            } else if (notifyType === MobileCCS.NotifyTypes.ERROR) {
                // todo what action to take for an error response to the GetCustomerHistory
            }
        };
        
        /*
         * Callback invoked when the selected customer has changed
         */
        scope.customerChangeCB = function customerChangeCB(newValue, oldValue) {
            if (newValue !== undefined) {     
                if (newValue.ID !== scope.selectedCustomer.ID || scope.customerHistoryRequested === false) {
                    // a different customer has been selected or the initial selection of a customer
                    // clean old history and request customer history from the CCS API
                    scope.customerHistory = undefined;
                    scope.customerInteractionMediaList = undefined;

                    alertsService.log("customerHistory: customerChangeCB()", " customer:" + newValue.ID + " selected, request customer history");
                    MobileCCS.Customer.GetCustomerHistory(scope.onCustomerHistoryReceived, scope.selectedCustomer.ID);   
                    
                    scope.customerHistoryRequested = true;
                } else {
                   alertsService.log("customerHistory: customerChangeCB()", "existing customer:" + scope.selectedCustomer.ID + ", new customer:" + newValue.ID + ", customerHistoryRequested?" + scope.customerHistoryRequested); 
                }                                                            
            }
        };
        
        /*
         * Callback invoked when the selected interaction has changed
         */
        scope.interactionChangeCB = function interactionChangeCB(newValue, oldValue) {
            if (newValue !== undefined) {  
                if (scope.selectedInteraction.CanGetCustomerHistory && scope.selectedInteraction !== undefined && 
                    scope.selectedInteraction.State === 'Active' && scope.customerHistoryRequested === false) { 
                    
                    alertsService.log("customerHistory: interactionChangeCB()", " interaction:" + newValue.ID + " selected, request customer history");
                    interactionService.getCustomerHistory(scope.selectedInteraction, scope.onCustomerHistoryReceived);                     
                    
                    scope.customerHistoryRequested = true;
                }                                                           
            }
        };
        
        /*
         * Setup an angular watch on the selected customer scope variable so their history can be retrieved
         */
        scope.$watch(function() {
                return scope.selectedCustomer;                        
            }, scope.customerChangeCB, true
        );
        
        /*
         * Setup an angular watch on the selected interaction scope variable so the history of the customer associated with this interaction 
         * can be retrieved
         */
        scope.$watch(function() {
                return scope.selectedInteraction;                        
            }, scope.interactionChangeCB, true
        );
    }
    return customerHistoryDirective;
}