/* jshint -W097 */
"use strict";

angular.module('refClientJS.customerName', ['refClientJS.alerts'])
.directive('customerName', ["alertsService", function(alertsService) {

    /**
     * The customer name directive is responsible for display and control of modifications to the customer name for both a customer on a live 
     * interaction and a customer returned through customer search.
     * The directive has an isolated scope which makes it possible to re-use the directive from multiple locations in the client application.
     * For the name of a customer on an interaction, there is a customer name directive per interaction and therefore an isolated scope per 
     * interaction.
     * For the name of a customer returned from customer search, there is one customer name directive.
     */
    var customerNameDirective = {
        restrict: 'E',
        scope: {

            /**
             * The customer object is injected as an attribute to the customer name directive. The reference to the customer object is therefore 
             * passed in from the parent scope and here it is added to the directives isolated scope. 
             * The customer object is injected when displaying the name of a customer returned from a customer search or from a customer 
             * on a live interaction.
             */
            selectedCustomer: "=customer",
            /*
             * The interaction object is injected as an attribute to the customer name directive. The reference to the interaction object is 
             * therefore passed in from the parent scope and here it is added to the directives isolated scope. 
             * The interaction object is injected when displaying the name of a customer on a live interaction only. It is used to invoke the 
             * methods on the CCS API interaction object (such as interaction.UpdateCustomerName)
             */ 
            selectedInteraction: "=interaction"
        },
        templateUrl: 'customer/customerName/customer-name.tpl.html',
        link: link
    };
    
    function link(scope) {
        
        alertsService.log("customerName: link()", "");
        
        /*
         * Store any live edits to the customer name fields on the directive scope.
         * The edits could be stored directly on the CCS API Customer object fields. However if we ever want to allow the user to reset 
         * the name fields back to original values after editing, this would not be possible. Storing an editing copy of the values on 
         * the directive scope will allow this.
         */
        scope.title = "";
        scope.firstName = "";
        scope.lastName = "";                 
        
        /**
         * If there is a selected interaction on the directive scope, update the customer name using the CCS API Interaction object.
         * If there is no interaction on the directive scope, then this directive is displaying details of a customer returned 
         * from customer search. Update the customer name using the static methods on the CCS API Customer object
         */ 
        scope.updateCustomerName = function updateCustomerName() {
            if (scope.selectedInteraction !== undefined) {
                scope.selectedInteraction.UpdateCustomerName(scope.title, scope.firstName, scope.lastName);
            } else {
                MobileCCS.Customer.UpdateCustomerName(scope.onCustomerNameUpdateReceived, scope.selectedCustomer.ID, scope.title, scope.firstName, scope.lastName);
            }
        };    
        
        /*
         * Asynchronous callback method invoked from the CCS API
         */
        scope.onCustomerNameUpdateReceived = function onCustomerNameUpdateReceived(notifyType, notifyData) {
            if (notifyType === MobileCCS.NotifyTypes.RESPONSE) {
                alertsService.log("customerName: onCustomerNameUpdateReceived()", " " + notifyType);
                console.log(notifyData);
                scope.$apply(function() {
                    scope.selectedCustomer = notifyData;
                    scope.updateScopeVariables();
                });                
            } else if (notifyType === MobileCCS.NotifyTypes.ERROR) {
                // handle error in response to update customer name, display error message
            }
        };
        
        /*
         * Update the directives scope variables which store live edits to the customer name
         */
        scope.updateScopeVariables = function updateScopeVariables() {
            if (scope.selectedCustomer !== undefined) {
                scope.title = scope.selectedCustomer.Title;
                scope.firstName = scope.selectedCustomer.FirstName;
                scope.lastName = scope.selectedCustomer.LastName;
            }
        };
        
        /*
         * Callback invoked on a change to the selected customer object
         */
        scope.customerChangeCB = function customerChangeCB() {
            if (scope.selectedCustomer !== undefined) {
                alertsService.log("customerName: customerChangeCB()", "");
                // the customer has changed, update the directives scope variables (that track live edits) with the values from the customer
                scope.updateScopeVariables();
            }
        };
        
        scope.$watch(function() {
                return scope.selectedCustomer;
            }, scope.customerChangeCB, true
        );
    }
    
    return customerNameDirective;    
}]);