/* jshint -W097 */
"use strict";

/**
 * The webchat media directive is responsible for displaying and controlling webchat media for an Interaction.
 */ 
angular.module('refClientJS.webchatMedia', ['refClientJS.alerts'])
.directive('webchatMedia', ["alertsService", "$timeout", function(alertsService, $timeout) {

    var webchatMediaDirective = {
        restrict: 'E',
        /*
         * The interaction object is injected as an attribute to the webchat media directive. The reference to the interaction object is 
         * passed in from the parent scope and here it is added to the directives isolated scope. 
         */  
        scope: {
            selectedInteraction: '=interaction'            
        },
        templateUrl: 'media/webchat-media.tpl.html',
        link: link        
    };
    
    function link(scope) {        
        alertsService.log("webchatMedia: link()", "");
        
        // CCS media object
        scope.media = undefined;
        scope.consultTransferClicked = false;
        scope.finishClicked = false;
        scope.isCompleted = false;
        scope.isCustomerTyping = false;
        scope.isAgentTyping = false;
        scope.sendIsTyping = true;
        scope.shouldDisableChatControls = true;

        scope.transferSkillsets = [];
        scope.autoPhrases = [];
        scope.pagePushUrls = [];
             
        /*
         * Any properties used in ng-model must be stored on an object and not directly on the scope. 
         * This is due to angular scope inheritance
         */
        scope.data = {
            transferTo: "",
            selectedSkillset: undefined,
            selectedUser: undefined,
            selectedPhrase: undefined,
            selectedUrl: undefined,
            transferReason: "",
            closingAgentNote: "",
            agentTypedText: "",
            chatBody: "",
            sendChatTranscript: false,
            agentLabel: ""
        };      
        
        scope.mediaDetailsDisplayed = false;

        scope.toggleShowMediaDetails = function toggleShowMediaDetails() {
            if (scope.mediaDetailsDisplayed) {
                scope.mediaDetailsDisplayed = false;
            } else {
                scope.mediaDetailsDisplayed = true;
            }
        };
        
        /*
         * Reverts the view back to the webchat content view
         */
        scope.cancel = function cancel() {
            scope.finishClicked = false;
            scope.consultTransferClicked = false;
            scope.data.selectedSkillset = undefined;
            scope.data.selectedUser = undefined;
            scope.data.selectedPhrase = undefined;
            scope.data.selectedUrl = undefined;
            scope.data.transferTo = "";
            scope.data.transferReason = "";
        };
        
       /*
        * On click of the consult/transfer button, the transfer view will be displayed. 
        * Retrieves possible transfer destinations.
        */
       scope.displayTransfer = function displayTransfer() {
           alertsService.log('webchatMedia: displayTransfer()', 'retrieve skillsets');
           scope.consultTransferClicked = true;
           if (scope.transferSkillsets.length === 0 && scope.selectedInteraction.CanGetConsultDestinations) {
               scope.selectedInteraction.GetConsultDestinations("Skillset", scope.onConsultDestinationsReceived);
           }
       };

       /*
        * Asynchronous callback function for transfer destinations received from the CCS API
        */
       scope.onConsultDestinationsReceived = function onConsultDestinationsReceived(notifyType, notifyData) {
           alertsService.log('webchatMedia: onConsultDestinationsReceived()', '' + notifyType);
           console.log(notifyData);
           if (notifyType === MobileCCS.NotifyTypes.RESPONSE) {
               for (var i in notifyData.DestinationListValues) {
                   console.log(notifyData.DestinationListValues[i]);
               }
               scope.$apply(function () {
                   if (notifyData.DestinationType === 'Skillset') {
                       scope.transferSkillsets = notifyData.DestinationListValues;
                   }
               });
           }
       };

       /*
        * Asynchronous callback function for frquently used phrases received from the CCS API
        */
       scope.onAutoPhrasesReceived = function onAutoPhrasesReceived(notifyType, notifyData) {
           alertsService.log('webchatMedia: onAutoPhrasesReceived()', '' + notifyType);
           console.log(notifyData);
           if (notifyType === MobileCCS.NotifyTypes.RESPONSE) {
               var phraseList = [];
               for (var i in notifyData.AutoPhrases) {
                   phraseList.push({
                       'Name': notifyData.AutoPhrases[i].Name,
                       'Phrase': notifyData.AutoPhrases[i].PhraseText
                   });
               }
               scope.$apply(function () {
                   scope.autoPhrases = phraseList;
                   console.log(scope.autoPhrases);
               });
           }
       };

       /*
        * Asynchronous callback function for page push URLs received from the CCS API
        */
       scope.onPagePushUrlsReceived = function onPagePushUrlsReceived(notifyType, notifyData) {
           alertsService.log('webchatMedia: onPagePushUrlsReceived()', '' + notifyType);
           console.log(notifyData);
           if (notifyType === MobileCCS.NotifyTypes.RESPONSE) {
               var urlList = [];
               for (var i in notifyData.UrlList) {
                   urlList.push({
                       'Description': notifyData.UrlList[i].Description,
                       'Url': notifyData.UrlList[i].Url
                   });
               }
               scope.$apply(function () {
                   scope.pagePushUrls = urlList;
                   console.log(scope.pagePushUrls);
               });
           }
       };

       /**
        * Requests the auto phrases from the CCS API
        */
       scope.getAutoPhrases = function getAutoPhrases() {
           if (scope.autoPhrases.length === 0) {
               if (scope.selectedInteraction.CanGetAutoPhrases) {
                   alertsService.log('webchatMedia: getAutoPhrases()', 'for media:' + scope.media.ID + ', interaction:' + scope.selectedInteraction.ID);
                   scope.selectedInteraction.GetAutoPhrases(scope.onAutoPhrasesReceived);
               }
           } else {
               alertsService.log('webchatMedia: getAutoPhrases()', 'phrases already retrieved from server');
           }
       };
        
        /**
         * Requests the page push URLs from the CCS API
         */
        scope.getPagePushUrls = function getPagePushUrls() {
            if (scope.pagePushUrls.length === 0) {
                if (scope.selectedInteraction.CanGetPagePushUrls) {
                    alertsService.log('webchatMedia: getPagePushUrls()', 'for media:' + scope.media.ID + ', interaction:' + scope.selectedInteraction.ID);
                    scope.selectedInteraction.GetPagePushUrls(scope.onPagePushUrlsReceived);
                }
            } else {
                alertsService.log('webchatMedia: getPagePushUrls()', 'page push URLs already retrieved from server');
            }
        };
   
        /**
         * The Agent is typing. Sends an isTyping request if sendIsTyping flag is true.
         */
        scope.isTyping = function isTyping() {
            alertsService.log('webchatMedia: isTyping()', 'for media:' + scope.media.ID);
            if (scope.sendIsTyping) {
                scope.media.SendIsTyping();
                scope.isTypingTimer();
            }
        };
        
        /**
         * When an isTyping request is sent, another wont be sent within 3 seconds.
         */
        scope.isTypingTimer = function () {
            var sendIsTypingTimeout = 3000;
            scope.sendIsTyping = false;
            $timeout(function () {
                scope.sendIsTyping = true;
            }, sendIsTypingTimeout);
        };
        
        /**
         * Send a new chat message to participants in the chat.
         */
        scope.sendMessage = function sendMessage(message) {
            if (message !== "") {
                alertsService.log('webchatMedia: sendMessage()', 'for media:' + scope.media.ID);
                scope.media.SendChatMessage(message, false);
                scope.data.chatBody = (scope.data.chatBody + scope.data.agentLabel + ": " + message + "\n");
                scope.data.agentTypedText = "";
            }
        };
        
        /**
         * Send a new page push url to participants in the chat.
         */
        scope.sendPagePushUrl = function sendPagePushUrl(url) {
            if (url !== "") {
                alertsService.log('webchatMedia: sendPagePushUrl()', 'for media:' + scope.media.ID + ' with url:' + url);
                scope.media.SendPagePushUrl(url);
                scope.data.chatBody = (scope.data.chatBody + scope.data.agentLabel + ": " + url + "\n");
            }
        };
        
        /*
         * Transfer button should only be enabled if CanTransferToDestination is true and a skillset has been selected.
         */
        scope.shouldDisableTransferButton = function shouldDisableTransferButton() {
            var shouldDisable = true;
            if (scope.selectedInteraction.CanTransferToDestination) {
                if (scope.data.transferTo === "skillset") {
                    if (scope.data.selectedSkillset !== null && scope.data.selectedSkillset !== undefined) {
                        shouldDisable = false;
                    }
                }
            }
            return shouldDisable;
        };
        
        /*
         * Invoked when the transfer button is clicked. 
         * On a click of the transfer button, the single step transfer operation will be executed. 
         */
        scope.transferToSkillset = function transferToSkillset() {
            if (scope.data.transferTo === "skillset") {
                // invoke transfer to skillset
                alertsService.log('webchatMedia: transfer()', 'to skillset ' + scope.data.selectedSkillset.Name);
                if (scope.data.transferReason.trim().length > 0) {
                    scope.selectedInteraction.TransferToDestination(scope.data.selectedSkillset, scope.data.transferReason);
                } else {
                    scope.selectedInteraction.TransferToDestination(scope.data.selectedSkillset);
                }
                scope.isCompleted = true;
                scope.consultTransferClicked = false;
            }
        };

        /**
         * The Agent has clicked the Finish button.
         * If interaction can be dispositioned the disposition screen will be displayed.
         */
        scope.finish = function finish() {
            // Note: Prototype for future development - ending when wrap-up in not required.
            if (!scope.selectedInteraction.CanDisposition) {
                scope.selectedInteraction.End();
            } else {
                alertsService.log('webchatMedia: finish()', 'for media:' + scope.media.ID);
                scope.finishClicked = true;
            }
        };

        /**
         * The Agent has clicked the Close button so the interaction will be ended.
         */
        scope.close = function close() {
            if (scope.selectedInteraction.CanEnd) {
                alertsService.log('webchatMedia: close()', 'for media:' + scope.media.ID);
                scope.selectedInteraction.End(scope.data.closingAgentNote, scope.data.sendChatTranscript);
            }
        };
        
        /*
         * Asynchronous callback function for media received on the interaction media path.
         */
        scope.onWebchatMediaReceived = function onWebchatMediaReceived(notifyType, notifyData) {
            alertsService.log("webchatMedia: onWebchatMediaReceived()", " " + notifyType);
            console.log(notifyData);
            if (notifyType === MobileCCS.NotifyTypes.NEW && scope.media === undefined) {
                scope.$apply(function () {
                    scope.media = notifyData;
                });
            } else if (notifyType === MobileCCS.NotifyTypes.UPDATE && notifyData.ContactID === scope.media.ContactID) {
                scope.media = notifyData;
                scope.webchatMediaUpdate(notifyData);
            } else if (notifyType === MobileCCS.NotifyTypes.DELETE) {
                // Handle deletion                
            }
        };
        
        /*
         * Handles media if the notify type is UPDATE.
         * The 'Method' property on the notify data describes the type of media update.
         * notifyData.Method will be one of the following: isTyping, newMessage, pagePush, agentJoinRoom, customerDisconnect or exitRoom.
         */
        scope.webchatMediaUpdate = function webchatMediaUpdate(notifyData) {
            alertsService.log("webchatMedia: webchatMediaUpdate()", " ");

            scope.$apply(function () {
                switch (notifyData.Method) {
                    case 'isTyping':
                        scope.isTypingReceived(notifyData.SenderType);
                        break;
                    case 'newMessage':
                        scope.chatMessageReceived(notifyData.DisplayName, notifyData.Message, notifyData.SenderType, notifyData.MessageType);
                        break;
                    case 'pagePush':
                        scope.chatMessageReceived(notifyData.DisplayName, notifyData.URL, notifyData.SenderType, 'normal');
                        break;
                    case 'agentJoinRoom':
                        // informs the agent of the details of the chat room when first joined.
                        scope.data.agentLabel = notifyData.AgentLabel;
                        scope.disableChatControls(false);
                        // if this is a transfered contact, a chat transcript will exist with an array of messages
                        if (notifyData.Transcript.messages.length !== 0) {
                            notifyData.Transcript.messages.forEach(scope.handleTranscriptMessage);
                        }
                        scope.getAutoPhrases();
                        scope.getPagePushUrls();
                        break;
                    case 'customerDisconnect':
                        scope.data.chatBody = (scope.data.chatBody + "The Customer has left the conversation" + "\n");
                        scope.disableChatControls(true);
                        break;
                    case 'exitRoom':
                        //TODO
                        break;
                }
            });
        };
        
        /**
         * Enable or disable chat controls to allow agent interact 
         */
        scope.disableChatControls = function (shouldDisable) {
            scope.shouldDisableChatControls = shouldDisable;
        };
        
        /**
         * Transcript item properties:
         * "type": string - possible values are "message" or "pagePush"
         * "message": string containing the message if the "type" is "message". 
         * "pagePush": string containing the URL if the "type" is "pagePush".
         * "displayname": string containing the label of the participant who sent the message.
         * "sender": object - contains details of the sender
         *   		"sender" properties:    "type": string - possible values are "agent" or "customer".
         *							        "agentId": string - null if sender type is "customer".
         * "messageType": string - possible values are normal, whisper, welcomemessage or comfortmessage. Only exists if type is "message".
         * "timestamp": timestamp of when message was sent.
         */
        scope.handleTranscriptMessage = function (item) {
            if (item.type === "message") {
                scope.chatMessageReceived(item.displayName, item.message, item.sender.type, item.messageType);
            } else if (item.type === "pagePush") {
                scope.chatMessageReceived(item.displayName, item.url, item.sender.type);
            }
        };
        
        /**
         * A new chat message has been received.
         * Parameters:
         * displayname: string containing the label of the participant who sent the message.
         * message: string containing the message body.
         * senderType: string - possible values are "agent" or "customer".
         * messageType: string - possible values are normal, whisper, welcomemessage or comfortmessage..
         */
        scope.chatMessageReceived = function (displayName, message, senderType, messageType) {
            if (messageType === "whisper") {
                scope.data.chatBody = (scope.data.chatBody + displayName.italics() + ": " + message.italics() + "\n");
            } else {
                scope.data.chatBody = (scope.data.chatBody + displayName + ": " + message + "\n");
            }

            if (senderType === "customer") {
                scope.isCustomerTyping = false;
            }

            if (senderType === "agent") {
                scope.isAgentTyping = false;
            }
        };
        
        /**
         * When an isTyping message is received, display the appropriate label for 5 seconds.
         * Parameter
         * senderType: string - possible values are "agent" or "customer".
         */
        scope.isTypingReceived = function (senderType) {
            var isTypingTimeout = 5000;
            if (senderType === "customer") {
                scope.isCustomerTyping = true;
                $timeout(function () {
                    scope.isCustomerTyping = false;
                }, isTypingTimeout);
            } else if (senderType === "agent") {
                scope.isAgentTyping = true;
                $timeout(function () {
                    scope.isAgentTyping = false;
                }, isTypingTimeout);
            }
        };
        
        /**
         * On Interaction change request the Media if it has not already been requested
         */
        scope.onInteractionChangeCB = function onInteractionChangeCB(newValue, oldValue) {
            if (scope.selectedInteraction.CanOpenMedia && scope.selectedInteraction !== undefined &&
                scope.selectedInteraction.State === 'Active' && scope.media === undefined) {

                alertsService.log("webchatMedia: onInteractionChangeCB()", "Request Media");
                if (scope.selectedInteraction.ContactType === "Web_Communications") {
                    scope.selectedInteraction.OpenMedia(scope.onWebchatMediaReceived);
                }
            }
        };
             
        /**
         * Agent has selected an auto phrase. Update UI accordingly.
         */
        scope.onPhraseChangeCB = function onPhraseChangeCB(newValue, oldValue) {
            if (newValue !== undefined) {
                scope.data.agentTypedText += newValue.Phrase;
                scope.data.selectedPhrase = undefined;
            }
        };
        
        /**
         * Agent has selected a page push url. Update UI accordingly.
         */
        scope.onUrlChangeCB = function onUrlChangeCB(newValue, oldValue) {
            if (newValue !== undefined) {
                scope.sendPagePushUrl(newValue.Url);
                scope.data.selectedUrl = undefined;
            }
        };
        
        scope.$watch(function () {
            return scope.selectedInteraction;
        }, scope.onInteractionChangeCB, true);

        scope.$watch(function () {
            return scope.selectedInteraction.ParticipationReason;
        }, scope.onParticipationReasonChangeCB, true);

        scope.$watch(function () {
            return scope.data.selectedPhrase;
        }, scope.onPhraseChangeCB, true);

        scope.$watch(function () {
            return scope.data.selectedUrl;
        }, scope.onUrlChangeCB, true);
    }
    
    return webchatMediaDirective;
}]);