/* jshint -W097 */
"use strict";

/**
 * The email media directive is responsible for displaying and controlling email media for an Interaction.
 */ 
angular.module('refClientJS.emailMedia', ['refClientJS.alerts'])
.directive('emailMedia', ["alertsService", "attachmentService", function(alertsService, attachmentService) {
    /*
     * The directive has an isolated scope which makes it possible to re-use the directive from multiple locations in the client application.
     * There is an instance of the email media directive per interaction and therefore an isolated scope per interaction.
     */
    var emailMediaDirective = {
        restrict: 'E',
        /*
         * The interaction object is injected as an attribute to the email media directive. The reference to the interaction object is 
         * passed in from the parent scope and here it is added to the directives isolated scope. 
         */  
        scope: {
            selectedInteraction: '=interaction'            
        },
        templateUrl: 'media/email-media.tpl.html',
        link: link        
    };
    
    function link(scope) {
        alertsService.log("emailMedia: link()", "");

        // CCS media object
        scope.media = undefined;

        // stores the to, from, cc, bcc, subject and body content the outgoing email
        scope.emailContent = {
            initialized: false,
            isHtml: true
        };

        scope.emailOriginalFormattedMessageContent = "";

        scope.replyClicked = false;
        scope.transferClicked = false;
        scope.finishClicked = false;
        scope.isCompleted = false;

        scope.destinationSkillsets = [];
        scope.consultUsers = [];
        scope.suggestedResponses = [];

        scope.suggestedResponseInserted = false;
        
        scope.isInitiated = (scope.selectedInteraction.State === 'Initiated');
        
        /*
         * Any properties used in ng-model must be stored on an object and not directly on the scope. 
         * This is due to angular scope inheritance
         */
        scope.data = {
            transferTo: "",
            selectedUser: undefined,
            selectedSkillset: undefined,
            selectedSuggestedResponse: undefined,
            agentNote: "",
            closingAgentNote: ""
        };
        
        /*
         * Stores file information for files that have been attached to an email. The file information is for display purposes only. 
         * The actual File object is not stored here (attachmentService). 
         */
        scope.outgoingAttachments = [];

        scope.mediaDetailsDisplayed = false;

        scope.toggleShowMediaDetails = function toggleShowMediaDetails() {
            if (scope.mediaDetailsDisplayed) {
                scope.mediaDetailsDisplayed = false;
            } else {
                scope.mediaDetailsDisplayed = true;
            }
        };
        
        /**
         * Callback function indicates that the email has been sent.
         */
        scope.emailCompleteHandler = function emailCompleteHandler() {
            alertsService.log('emailMedia: emailCompleteHandler()', 'Email sent!');
        };

        /**
         * Callback function indicating the progress of attachment uploads.
         */
        scope.emailProgressHandler = function emailProgressHandler(attachmentIndex, percentage) {
            alertsService.log('emailMedia: emailProgressHandler()', 'Email in progress! attachmentIndex:' + attachmentIndex + ' percentage:' + percentage);
        };
                
        /*
         * Asynchronous callback function for agent/skillset destinations received from the CCS API
         */
        scope.onDestinationsReceived = function onDestinationsReceived(notifyType, notifyData) {
            alertsService.log('emailMedia: onDestinationsReceived()', '' + notifyType);
            console.log(notifyData);

            if (notifyType === MobileCCS.NotifyTypes.RESPONSE) {
                for (var i in notifyData.DestinationListValues) {
                    console.log(notifyData.DestinationListValues[i]);
                }
                scope.$apply(function () {
                    if (notifyData.DestinationType === 'Agent') {
                        scope.consultUsers = notifyData.DestinationListValues;
                    } else if (notifyData.DestinationType === 'Skillset') {
                        scope.destinationSkillsets = notifyData.DestinationListValues;
                    }
                });
            }
        }; 
        
        /*
         * Request suggested responses from the CCS API
         */
        scope.getSuggestedResponses = function getSuggestedResponses() {
            if (scope.suggestedResponses.length === 0) {
                alertsService.log('emailMedia: getSuggestedResponses()', 'for media:' + scope.media.ID);
                scope.media.GetSuggestedResponses(scope.onSuggestedResponsesReceived);
            }
        };

        /*
         * Asynchronous callback function for suggested responses received from the CCS API
         */
        scope.onSuggestedResponsesReceived = function onSuggestedResponsesReceived(notifyType, notifyData) {
            alertsService.log('emailMedia: onSuggestedResponsesReceived()', notifyType + ' ' + notifyData);

            if (notifyType === MobileCCS.NotifyTypes.RESPONSE) {
                for (var i in notifyData.SuggestedResponses) {
                    console.log(notifyData.SuggestedResponses[i]);
                }
                scope.$apply(function () {
                    scope.suggestedResponses = notifyData.SuggestedResponses;
                });
            } else if (notifyType === MobileCCS.NotifyTypes.ERROR) {
                // action to take if an error is returned retrieving suggested responses?
            }
        };    
        
        /*
         * Insert the selected suggested response into the email reply
         */
        scope.insertSuggestedResponse = function insertSuggestedResponse() {
            if (scope.data.selectedSuggestedResponse !== undefined) {
                scope.emailContent.body = scope.data.selectedSuggestedResponse.Body + scope.emailOriginalFormattedMessageContent;
                scope.emailContent.subject = scope.data.selectedSuggestedResponse.Subject;
                scope.suggestedResponseInserted = true;
                scope.downloadSuggestedResponseAttachments(scope.data.selectedSuggestedResponse);
                alertsService.log('emailMedia: insertSuggestedResponse()', 'for media:' + scope.media.ID + ' selectedResponseBody:' + scope.data.selectedSuggestedResponse.Body);
            }
        };
        
        /*
         * Remove a previously inserted suggested response. 
         * Reset the email reply content back to the original email message.
         */
        scope.removeSuggestedResponse = function removeSuggestedResponse() {
            if (scope.suggestedResponseInserted) {
                scope.emailContent.body = scope.emailOriginalFormattedMessageContent;
                scope.emailContent.subject = scope.media.Subject;
                scope.removeAllAttachmentFiles();
                scope.suggestedResponseInserted = false;
            }
        };
        
        /*
         * Download all attachments for a specified suggested response from the CCS API.
         */
        scope.downloadSuggestedResponseAttachments = function downloadSuggestedResponseAttachments(suggestedResponse) {
            angular.forEach(suggestedResponse.Attachments, function (attachment) {
                alertsService.log('emailMedia: downloadSuggestedResponseAttachments()', 'attachment:' + attachment.Address);

                // Request the CCS API to download the content of the attachment from the Contact Center server
                scope.media.GetAttachmentFile(attachment, scope.onAttachmentFileReceived);
            });
        }; 
        
        /*
         * Asynchronous callback handler invoked when download of an attachment through the CCS API has succeeded or failed.
         */
        scope.onAttachmentFileReceived = function onAttachmentFileReceived(notifyType, notifyData) {
            if (notifyType === MobileCCS.NotifyTypes.RESPONSE) {
                // If a RESPONSE is received, the notifyData object is the JavaScript File object that represents the attachment 
                // that has been downloaded successfully

                alertsService.log('emailMedia: onAttachmentFileReceived()', 'File downloaded: ' + notifyData.name);

                var files = [notifyData];

                scope.$apply(function () {
                    scope.addAttachmentFiles(files);
                });
            } else if (notifyType === MobileCCS.NotifyTypes.ERROR) {
                // If an ERROR is received, the File download failed                        
                alertsService.log('emailMedia: onAttachmentFileReceived()', 'File failed to download');
                console.log(notifyData);
            }
        };
        
        /*
         * Add a file or files that were selected from file input (standard attachments) or retrieved from file download.
         * (suggested response attachments)
         */
        scope.addAttachmentFiles = function addAttachmentFiles(files) {
            alertsService.log('emailMedia: addAttachmentFiles()', 'for media:' + scope.media.ID);
            console.log(files);

            /*jshint loopfunc: true */
            for (var i = 0; i < files.length; i++) {

                // check if a file with that name and size is already attached
                var existingFile = scope.outgoingAttachments.find(function (value) {
                    return (value.name === files[i].name);
                });

                if (existingFile !== undefined) {
                    if (existingFile.size === files[i].size) {
                        // identical existing file found, no need to add
                        continue;
                    } else {
                        // file exists but it is a different size to the file being added, remove completely and re-add
                        scope.removeAttachmentFile(existingFile);
                    }
                }

                // create a file info object to store information about the file for display purposes
                var fileInfo = {
                    name: files[i].name,
                    size: files[i].size,
                    type: files[i].type
                };
                alertsService.log('emailMedia: addAttachmentFiles()', 'Attachment:' + fileInfo.name);
                console.log(fileInfo);

                // Save the file info object related to the the attached file on the model so that it can be displayed to the Agent              
                scope.outgoingAttachments.push(fileInfo);
            }

            // Must store the JavaScript File objects outside of the angular scope as the object is read only and is not compatible with the 
            // angular.copy method which is used when the digest is updating the model
            attachmentService.storeAttachmentFilesForMedia(scope.media.ID, files);
        }; 
        
        /*
         * Remove an attachment file that has been added to an outgoing email
         */
        scope.removeAttachmentFile = function removeAttachmentFile(file) {
            if (file !== undefined && file !== null) {
                var indexOfFileToRemove = scope.outgoingAttachments.indexOf(file);
                if (indexOfFileToRemove >= 0) {
                    // remove the file from the attachment array
                    scope.outgoingAttachments.splice(indexOfFileToRemove, 1);

                    attachmentService.removeAttachmentFileForMedia(scope.media.ID, file.name);
                }
                // print remaining files for test
                angular.forEach(scope.outgoingAttachments, function (file) {
                    alertsService.log('emailMedia: removeAttachmentFile()', 'for media:' + scope.media.ID + ' remaining ' + file.name);
                });
            }
        };
        
        /*
         * Remove all attachment files that has been added to an outgoing email
         */
        scope.removeAllAttachmentFiles = function removeAllAttachmentFiles() {
            scope.outgoingAttachments = [];
            attachmentService.clearAttachmentFilesForMedia(scope.media);
        };

        /*
         * Display the reply view and initialize the reply data
         */
        scope.reply = function reply() {
            scope.replyClicked = true;
            scope.initializeReply('reply');
            scope.getSuggestedResponses();
        };

        /*
         * Display the reply view and initialize the reply data
         */
        scope.replyAll = function replyAll() {
            scope.replyClicked = true;
            scope.initializeReply('replyall');
            scope.getSuggestedResponses();
        };

        /*
         * Cancels an in progress reply or transfer. 
         * Reverts the view back to the received email view
         */
        scope.cancel = function cancel() {
            scope.replyClicked = false;
            scope.transferClicked = false;
            scope.finishClicked = false;
            scope.data.selectedSkillset = undefined;
            scope.data.selectedUser = undefined;
            scope.data.transferTo = "";
            scope.data.agentNote = "";

            scope.emailContent = {
                initialized: false,
                isHtml: true
            };

            // clear reference to any stored attachments (the JavaScript File objects representing the attachments)
            attachmentService.clearAttachmentFilesForMedia(scope.media.ID);

            if (scope.isInitiated) {
                // cancel of an adhoc initiated email. This causes a delete of the Email Interaction.
                scope.close();
            }
        };

        /*
         * Send the email reply.
         */
        scope.sendEmailReply = function sendEmailReply() {
            var files = attachmentService.getAttachmentFilesForMedia(scope.media.ID);

            alertsService.log('emailMedia: sendEmailReply()', 'reply content: ' + scope.emailContent.body);
            scope.media.Reply(scope.emailContent.to, scope.emailContent.cc, scope.emailContent.bcc, scope.emailContent.subject, scope.emailContent.body, scope.data.agentNote, (files !== undefined ? files : null), scope.emailContent.isHtml, scope.emailCompleteHandler, scope.emailProgressHandler);

            // clear reference to any stored attachments (the JavaScript File objects representing the attachments)
            attachmentService.clearAttachmentFilesForMedia(scope.media.ID);

            scope.replyClicked = false;
            scope.isCompleted = true;
        };
        
        /*
         * Send an adhoc outgoing Email
         */
        scope.sendEmail = function sendEmail() {
            var files = attachmentService.getAttachmentFilesForMedia(scope.media.ID);

            alertsService.log('emailMedia: sendEmail()', 'content: ' + scope.emailContent.body);

            // ccs send email API
            scope.media.Create(scope.emailContent.to, scope.emailContent.cc, scope.emailContent.bcc, scope.emailContent.subject, scope.emailContent.body, scope.data.agentNote, (files !== undefined ? files : null), scope.emailContent.isHtml, scope.data.selectedSkillset.Number, scope.emailCompleteHandler, scope.emailProgressHandler);

            // clear reference to any stored attachments (the JavaScript File objects representing the attachments)
            attachmentService.clearAttachmentFilesForMedia(scope.media.ID);

            scope.isCompleted = true;
        };

        /*
         * Invoked when the transfer button is clicked. 
         * On first click of the transfer button, the transfer view will be displayed. 
         * On second click of the transfer button, the transfer operation will be executed. 
         * If cancel is invoked, the view will be brought back to the received email view (the state will be reinstated to pre transfer invocation) 
         */
        scope.transfer = function transfer() {
            if (scope.data.transferTo === "external") {
                // if the transfer to option is external and the transfer button was clicked, then invoke transfer to external
                alertsService.log('emailMedia: transfer()', 'to external ' + scope.emailContent.to);
                scope.media.Reply(scope.emailContent.to, scope.emailContent.cc, scope.emailContent.bcc, scope.emailContent.subject, scope.emailContent.body, scope.data.agentNote, null, scope.emailContent.isHtml, scope.emailCompleteHandler, scope.emailProgressHandler);
                scope.transferClicked = false;
                scope.isCompleted = true;

            } else if (scope.data.transferTo === "user") {
                // if the transfer to option is user and the transfer button was clicked, then invoke transfer to user
                alertsService.log('emailMedia: transfer()', 'to user ' + scope.data.selectedUser.Name);
                if (scope.data.agentNote.trim().length > 0) {
                    scope.selectedInteraction.TransferToDestination(scope.data.selectedUser, scope.data.agentNote);
                } else {
                    scope.selectedInteraction.TransferToDestination(scope.data.selectedUser);
                }
                scope.isCompleted = true;
                scope.transferClicked = false;

            } else if (scope.data.transferTo === "skillset") {
                // if the transfer to option is skillset and the transfer button was clicked, then invoke transfer to skillset
                alertsService.log('emailMedia: transfer()', 'to skillset ' + scope.data.selectedSkillset.Name);
                if (scope.data.agentNote.trim().length > 0) {
                    scope.selectedInteraction.TransferToDestination(scope.data.selectedSkillset, scope.data.agentNote);
                } else {
                    scope.selectedInteraction.TransferToDestination(scope.data.selectedSkillset);
                }
                scope.isCompleted = true;
                scope.transferClicked = false;

            } else {
                // initial transfer click to open transfer view
                alertsService.log('emailMedia: transfer()', 'retrieve skillsets and users');
                scope.transferClicked = true;
                if (scope.consultUsers.length === 0) {
                    scope.selectedInteraction.GetConsultDestinations("Agent", scope.onDestinationsReceived);
                }
                if (scope.destinationSkillsets.length === 0) {
                    scope.selectedInteraction.GetConsultDestinations("Skillset", scope.onDestinationsReceived);
                }
                scope.initializeReply('forward');
            }
        };

        /*
         * Determine if the transfer button should be disabled.
         */
        scope.shouldDisableTransferButton = function shouldDisableTransferButton() {
            var shouldDisable = false;
            // check if the transfer button has been clicked already to open the transfer view
            if (scope.transferClicked) {
                if (scope.data.transferTo === "external") {
                    if (scope.emailContent.to === '') {
                        // if the external transfer option is chosen and the forward to field is not populated, disable the transfer button                        
                        shouldDisable = true;
                    }
                } else if (scope.data.transferTo === "user") {
                    if (scope.data.selectedUser === undefined) {
                        // if the user transfer option is chosen and a user has not been selected from the user list, disable the transfer button                        
                        shouldDisable = true;
                    }
                } else if (scope.data.transferTo === "skillset") {
                    if (scope.data.selectedSkillset === undefined) {
                        // if the skillset transfer option is chosen and a skillset has not been selected from the skillset list, disable the transfer button
                        shouldDisable = true;
                    }
                } else {
                    // no transfer option has been chosen
                    alertsService.log('emailMedia: shouldDisableTransferButton()', ' no transfer option selected');
                    shouldDisable = true;
                }
            }
            return shouldDisable;
        };

        /*
         * Agent has clicked the finish button so the disposition screen should be displayed.
         */
        scope.finish = function finish() {
            alertsService.log('emailMedia: finish()', 'for media:' + scope.media.ID);
            scope.finishClicked = true;
        };

        /*
         * Agent has clicked the close button so the interaction is ended.
         */
        scope.close = function close() {
            if (scope.selectedInteraction.CanEnd) {
                alertsService.log('emailMedia: close()', 'for media:' + scope.media.ID);
                scope.selectedInteraction.End(scope.data.closingAgentNote);
            }
        };

        /*
         * Display the reply view and initialize the reply data
         */
        scope.initializeReply = function initializeReply(type) {
            // if email reply object is empty (i.e. initially or after a cancel of a reply or a transfer), then reset the email reply object
            if (scope.emailContent.initialized === true) {
                return;
            }

            scope.emailOriginalFormattedMessageContent = "\r\r------------------------------------------\rOriginal Message:\rFrom: " + scope.media.From + "\rSent: " + scope.media.ArrivalTime + "\rTo: " + scope.media.To + "\rSubject: " + scope.media.Subject + "\r\r" + scope.media.Body;

            scope.emailContent.body = scope.emailOriginalFormattedMessageContent;

            if (scope.media.CC !== undefined) {
                scope.emailContent.cc = scope.media.CC;
            }
            if (scope.media.BCC !== undefined) {
                scope.emailContent.bcc = scope.media.BCC;
            }

            if (scope.media.Encoding === 'HTML') {
                scope.emailContent.isHtml = true;
            } else {
                scope.emailContent.isHtml = false;
            }

            if (type === 'reply') {
                scope.emailContent.subject = scope.media.Subject;

                // the sender of this email needs to be added as a participant in the Reply To field
                scope.emailContent.to = scope.media.From;

            } else if (type === 'replyall') {
                scope.emailContent.subject = scope.media.Subject;

                // the sender of this email needs to be added as a participant in the Reply To field
                scope.emailContent.to = scope.media.From + ';';

                // identify whether there are multiple To participants that need to be sent this reply, split each participant from the To field into an array
                var toParticipants = scope.media.To.split(';');
                if (toParticipants.length > 0) {
                    // add each participant to the Reply To field
                    for (var i in toParticipants) {
                        // if the current To participant is the skillset mailbox, then do not include it in the Reply To field
                        if (toParticipants[i] === scope.media.MailboxAddress) {
                            continue;
                        }
                        scope.emailContent.to += (toParticipants[i] + ';');
                    }
                }

            } else if (type === 'forward') {
                // build forward subject for external transfers                    
                var indexOfRePrefix = scope.media.Subject.search(/Re:/i);
                if (indexOfRePrefix > 0) {
                    // re: found, replace with fw:
                    scope.emailContent.subject = scope.media.Subject.replace(/Re:/gi, "Fw:");
                } else {
                    // re: not found, prepend fw:
                    scope.emailContent.subject = "Fw: " + scope.media.Subject;
                }
            }
            scope.emailContent.initialized = true;
        };
                
        /**
         * Asychronous callback function for media received on the interaction media path.
         */
        scope.onMediaReceived = function onMediaReceived(notifyType, notifyData) {
            alertsService.log("emailMedia: onMediaReceived()", " " + notifyType);

            if (notifyType === MobileCCS.NotifyTypes.NEW || notifyType === MobileCCS.NotifyTypes.UPDATE) {
                scope.$apply(function () {
                    console.log(notifyData);
                    scope.media = notifyData;

                    /*
                     * If this is an Interaction for an adhoc initiated email, retrieve the skillset destinations.
                     */
                    if (scope.selectedInteraction.State === 'Initiated') {
                        alertsService.log("emailMedia: onMediaReceived()", "GetSkillsets for Email Initiate");
                        if (scope.selectedInteraction.CanGetSkillsets) {
                            scope.selectedInteraction.GetSkillsets(scope.onDestinationsReceived);
                        }
                        scope.emailContent.to = scope.media.To;
                        // subject and body must be initialized to empty string. otherwise if no value is entered for these fields the properties 
                        // are not sent at all in the Send 
                        scope.emailContent.subject = "";
                        scope.emailContent.body = "";
                    }
                });
            } else if (notifyType === MobileCCS.NotifyTypes.DELETE) {
                // Handle deletion of media if required                
            } else if (notifyType === MobileCCS.NotifyTypes.ERROR) {
                if (notifyData.Command === "Create") {
                    scope.$apply(function () {
                        // if the 'Create' command failed, display the received error and ensure the view goes back to the initiate email screen
                        alertsService.addAlert(notifyData.Message, 'danger');
                        scope.isCompleted = false;
                    });
                }
            }
        };
        
        /**
         * On Interaction change request the Media if it has not already been requested
         */
        scope.onInteractionChangeCB = function onInteractionChangeCB(newValue, oldValue) {
            // Request the media when the interaction has been answered
            if (scope.selectedInteraction.CanOpenMedia && scope.selectedInteraction !== undefined && scope.media === undefined &&
                (scope.selectedInteraction.State === 'Active' || scope.selectedInteraction.State === 'Initiated')) {

                alertsService.log("emailMedia: onInteractionChangeCB()", "Request Media");
                if (scope.selectedInteraction.ContactType === "EMail") {
                    scope.selectedInteraction.OpenMedia(scope.onMediaReceived);
                }
            }
        };
        
        scope.$watch(function() {
                return scope.selectedInteraction;        
            }, scope.onInteractionChangeCB, true
        );        
    }
    
    return emailMediaDirective;
}]);