angular.module('refClientJS.session', ['ui.router', 'refClientJS.user', 'refClientJS.interactions', 'refClientJS.resources', 'refClientJS.alerts'])
    .factory("sessionService", ["sessionData", "interactionData", "resourceData", "userData", "clientData", "userService", "interactionService", "resourceService", "$state", "alertsService", "alertsData", function (sessionData, interactionData, resourceData, userData, clientData, userService, interactionService, resourceService, $state, alertsService, alertsData) {
        /**
        * Angular factory sessionService registered with the main application module. Contains all methods required to interact with the server managing, 
        * updating and processing session related events.
        *   
        *
        * @returns {Object} sessionService object returned by this service is injected into other services and the main controller that has a dependency on the session
        */

        var sessionService = this;
        var ctiClient = {};

        /**
         * Signs the User into the contact center 
         *
         * @param    {String}  server The fully qualified hostname of the AACC/ACCS server
         * @param    {String}  name The Username of the User account used to sign into the server in the format Domain\Username       
         *
         * @param    {String}  password The password of the User account used to log into the AACC/ACCS server
         *
         * @returns {void}
        */
      sessionService.signIn = function (server, name, password) {

            ctiClient = new MobileCCS.Client("JSDemoClient", MobileCCS.ClientProfile.AgentDesktop);

            ctiClient.Server = server;
            ctiClient.SignIn(name, password, this.handleSession, this.handleClient);


        };

        /**
         * Signs the User out of the contact center 
         *
         *
         * @returns {void}
        */
        sessionService.signOut = function () {

            ctiClient.Disconnect();
        };

        /**
         * Callback function passed to the server during the sign in process. Handles the notifications from the server 
         * about the session. Types of notifications returned are NEW, UPDATE, DELETE, ERROR  
         *
         *
         * @returns {void}
        */
        this.handleSession = function (returnedNotifyType, returnedNotifyData) {
            
            alertsService.log('sessionService: handleSession: ' , returnedNotifyType);

            if (returnedNotifyType === MobileCCS.NotifyTypes.ERROR) {
                alertsService.logerror('sessionService handleSession',returnedNotifyData.Message);
                alertsService.addAlert(returnedNotifyData.Message, 'Error');
                console.log("sessionService ERROR notification: " + returnedNotifyType + ", Message=" + returnedNotifyData.Message);
                return;
            }

            if (returnedNotifyType === MobileCCS.NotifyTypes.DELETE) {
                
                var nullObj = {};
                var nullArray = [];
                sessionData.session = nullObj;
                interactionData.interactions = interactionData.removeAll();
                resourceData.resources = resourceData.removeAll();
                userData.user = nullObj;
                clientData.client = nullObj;
                alertsData.alerts = [];
                $state.go('signin');
            }
            if (returnedNotifyType === MobileCCS.NotifyTypes.NEW) {
                console.log(returnedNotifyData);
                //store the session object in the sessionData factory for use in other services and controllers
                sessionData.session = returnedNotifyData;

                if (returnedNotifyData.NumUsers !== 1) {
                    
                    alertsService.logwarn('sessionService handleSession', "There are " + returnedNotifyData.NumUsers + " Contact Center users associated with this login.  This client is designed to handle a single Contact Center user.  Please login as a different user.");
                    clientData.client.Disconnect();
                    $state.go('signin');
                    return;
                }

                sessionData.session.OpenUsers(userService.handleUser);
                sessionData.session.OpenResources(resourceService.handleResources);
                sessionData.session.OpenInteractions(interactionService.handleInteractions);

                $state.go('signin');

            }
            if (returnedNotifyType === MobileCCS.NotifyTypes.UPDATE) {
                console.log(returnedNotifyData);
                //store the session object in the sessionData factory for use in other services and controllers
                sessionData.session = returnedNotifyData;

                $state.go('signedin');

            }


        };

        /**
         * Callback function passed to the server during the sign in process. Handles the notifications from the server 
         * about the client. Types of notifications returned are NEW, UPDATE, DELETE, ERROR  
         *
         *
         * @returns {void}
        */
        this.handleClient = function (returnedNotifyType, returnedNotifyData) {

            alertsService.log('sessionService: handleClient: ' , returnedNotifyType);
            console.log(returnedNotifyData);


            if (returnedNotifyType === MobileCCS.NotifyTypes.ERROR) {
                alertsService.addAlert(returnedNotifyData.Message, 'Error');
                alertsService.logerror('sessionService: handleClient: ' , returnedNotifyData.Message);
                return;
            }


            if (returnedNotifyType === MobileCCS.NotifyTypes.DELETE) {
                var nullObj = {};
                var nullArray = [];

                sessionData.session = nullObj;
                interactionData.interactions = interactionData.removeAll();
                resourceData.resources = resourceData.removeAll();
                userData.user = nullObj;
                clientData.client = nullObj;
                $state.go('signin');
            }
            if (returnedNotifyType === MobileCCS.NotifyTypes.NEW) {

                clientData.client = returnedNotifyData;
                $state.go('signin');

            }

            if (returnedNotifyType === MobileCCS.NotifyTypes.UPDATE) {
                
                clientData.client = returnedNotifyData;
                if (!clientData.client.IsConnected)
                {
                    if (clientData.client.StateError !== null)
                    {
                        alertsService.addAlert(clientData.client.StateError.Message, 'Error');
                        alertsService.logerror('sessionService: handleClient: ' , clientData.client.StateError.Message);
                    }
                }
                $state.go('signin');
            }

        };

        return sessionService;

    }])
    /**
        * sessionData is a factory which stores the current session information
        *
        *
        * @returns {Object} returns the session object used by dependent services and controllers in the application
    */
    .factory("sessionData", function () {
        var session = {};
        return {
            'session': session
        };
    })
    .factory("clientData", function () {
        var client = {};
        return {
            'client': client
        };
    });