package com.nortel.soa.oi.openq;

import java.io.File;
import java.net.MalformedURLException;
import java.net.URL;
import java.util.HashMap;
import java.util.Iterator;

import javax.xml.namespace.QName;
import javax.xml.ws.Endpoint;
import javax.xml.ws.Holder;

import org.apache.commons.cli.BasicParser;
import org.apache.commons.cli.CommandLine;
import org.apache.commons.cli.HelpFormatter;
import org.apache.commons.cli.Options;
import org.apache.commons.cli.ParseException;

import com.nortel.soa.oi.cct.types.AuthenticationLevel;
import com.nortel.soa.oi.cct.types.Contact;
import com.nortel.soa.oi.cct.types.Intrinsic;
import com.nortel.soa.oi.cct.types.IntrinsicArray;
import com.nortel.soa.oi.cct.types.SsoToken;

public final class OIOpenQ {

	public static final String OPENQ_ADDRESS = "SOAOI/services/OpenQ";
	public static final QName  OPENQ_SERVICE_QNAME = new QName("http://www.nortel.com/soa/oi/OpenQ", "SOAOI_OpenQ");

	private static final String DEFAULT_USERNAME = "!cramU1+gate";
	private static final String DEFAULT_PASSWORD= "_hetUp+SubD95";
	private static final String DEFAULT_DOMAIN = "OpenQ";

	private static final int DEFAULT_CONTACT_ID = 12345;
	private static final String DEFAULT_OUT_OF_PROVIDER_ADDRESS = "$GenericMmOutOfProviderAddress";

	private static final String OPT_HELP = "h";
	private static final String OPT_HELP_LONG = "help";
	private static final String OPT_HELP_DESC = "Print help for this application";

	private static final String OPT_WSDL = "w";
	private static final String OPT_WSDL_LONG = "wsdl";
	private static final String OPT_WSDL_DESC = "Path to wsdl file or wsdl url";

	private static final String OPT_CREATE_CONTACT = "create";
	private static final String OPT_DROP_CONTACT = "drop";
	private static final String OPT_GET_CONTACT = "get";
	
	private static final String OPT_LOGON = "logOn";
	private static final String OPT_LOGOFF = "logOff";
	
	private static final String OPT_ADD_SERVICE_PROVIDER_STATUS_LISTENER = "addServiceProviderStatusListener";
	private static final String OPT_REMOVE_SERVICE_PROVIDER_STATUS_LISTENER = "removeServiceProviderStatusListener";
	private static final String OPT_ADD_FORCE_DISCONNECT_LISTENER = "addForceDisconnectListener";
	private static final String OPT_REMOVE_FORCE_DISCONNECT_LISTENER = "removeForceDisconnectListener";
	private static final String OPT_ADD_DEFAULT_TREATMENT_LISTENER = "addDefaultTreatmentListener";
	private static final String OPT_REMOVE_DEFAULT_TREATMENT_LISTENER = "removeDefaultTreatmentListener";
	
	private static final String OPT_OPERATION = "operation";
	private static final String OPT_OPERATION_DESC =
		"'"+OPT_LOGON+
		"' | '"+OPT_LOGOFF+
		"' | '"+OPT_CREATE_CONTACT+
		"' | '"+OPT_DROP_CONTACT+
		"' | '"+OPT_GET_CONTACT+
		"' | '"+OPT_ADD_SERVICE_PROVIDER_STATUS_LISTENER+
		"' | '"+OPT_REMOVE_SERVICE_PROVIDER_STATUS_LISTENER+
		"' | '"+OPT_ADD_FORCE_DISCONNECT_LISTENER+
		"' | '"+OPT_REMOVE_FORCE_DISCONNECT_LISTENER+
		"' | '"+OPT_ADD_DEFAULT_TREATMENT_LISTENER+
		"' | '"+OPT_REMOVE_DEFAULT_TREATMENT_LISTENER+";";
	
	private static final String OPT_USERNAME = "usr";
	private static final String OPT_USERNAME_LONG = "username";
	private static final String OPT_USERNAME_DESC = "OpenQ username";
	private static final String OPT_PASSWORD = "pwd";
	private static final String OPT_PASSWORD_LONG = "password";
	private static final String OPT_PASSWORD_DESC = "OpenQ password";
	private static final String OPT_DOMAIN = "dmn";
	private static final String OPT_DOMAIN_LONG = "domain";
	private static final String OPT_DOMAIN_DESC = "OpenQ domain";
	private static final String OPT_SSO = "sso";
	private static final String OPT_SSO_LONG = "SsoToken";
	private static final String OPT_SSO_DESC = "OpenQ Sso token returned on successful logOn";
	
	private static final String OPT_CONTACT_ID = "cId";
	private static final String OPT_CONTACT_ID_LONG = "contactExtId";
	private static final String OPT_CONTACT_ID_DESC = "External contact Id";
	private static final String OPT_OUT_OF_PROVIDER_ADDRESS ="oId";
	private static final String OPT_OUT_OF_PROVIDER_ADDRESS_LONG ="outOfProviderAddress";
	private static final String OPT_OUT_OF_PROVIDER_ADDRESS_DESC ="OutOfProviderAddress";
	private static final String OPT_INTRINSICS = "i";
	private static final String OPT_INTRINSICS_LONG = "intrinsics";
	private static final String OPT_INTRINSICS_DESC = "Intrinsics set on this contact";
	
	private static final String OPT_SUBSCRIPTION_ID = "sId";
	private static final String OPT_SUBSCRIPTION_ID_LONG = "subscriptionId";
	private static final String OPT_SUBSCRIPTION_ID_DESC = "Subscription id for the associated listener";
	
	private static final String OPT_LISTENER_IP = "lIp";
	private static final String OPT_LISTENER_IP_LONG = "consumerIp";
	private static final String OPT_LISTENER_IP_DESC = "Consumer ip for listener notifications";
	
	private static final String OPT_LISTENER_PORT = "lPort";
	private static final String OPT_LISTENER_PORT_LONG = "consumerPort";
	private static final String OPT_LISTENER_PORT_DESC = "Consumer port for listener notifications";
	
	private static final String OPT_LOAD_TEST_RATE = "ltR";
	private static final String OPT_LOAD_TEST_RATE_LONG = "loadTestRate";
	private static final String OPT_LOAD_TEST_RATE_DESC = "Load test rate per second";
	
	private static final String OPT_LOAD_TEST_DURATION = "ltD";
	private static final String OPT_LOAD_TEST_DURATION_LONG = "loadTestDuration";
	private static final String OPT_LOAD_TEST_DURATION_DESC = "Load test duration in seconds";

	private enum OPERATION {
		LOGON, 
		LOGOFF,
		CREATE, 
		DROP, 
		GET,
		ADD_SERVICE_PROVIDER_LISTENER, 
		REMOVE_SERVICE_PROVIDER_LISTENER, 
		ADD_FORCE_DISCONNECT_LISTENER, 
		REMOVE_FORCE_DISCONNECT_LISTENER, 
		ADD_DEFAULT_TREATMENT_LISTENER, 
		REMOVE_DEFAULT_TREATMENT_LISTENER
	}

	private static Options _cliOptions = null; 
	private static final AuthenticationLevel _authenticationLevel = new AuthenticationLevel();	
	private static OpenQ _openQPort = null; 
	private static OPERATION _selectedOperation = null;
	private static ContactDetails _contactDetails = null;
	private static String _subscriptionId = null;
	private static String _listenerIp = null;
	private static String _listenerPort = null;
	private static SsoToken _sso = null;
	private static Object _consumerImplementor = new OIOpenQConsumerImpl();
	
	private static boolean _isLoadTestRequired = false;
	private static int _loadTestRatePerSec = 5;
	private static int _loadTestDurationInSec = 30;
	
	private OIOpenQ() {
	}

	private void setupOptions() {
		_cliOptions = new Options();

		_cliOptions.addOption(OPT_HELP, OPT_HELP_LONG, false, OPT_HELP_DESC);

		_cliOptions.addOption(OPT_WSDL, OPT_WSDL_LONG, true, OPT_WSDL_DESC);
		_cliOptions.getOption(OPT_WSDL).setRequired(true);

		_cliOptions.addOption(OPT_USERNAME, OPT_USERNAME_LONG, true, OPT_USERNAME_DESC);
		_cliOptions.addOption(OPT_PASSWORD, OPT_PASSWORD_LONG, true, OPT_PASSWORD_DESC);
		_cliOptions.addOption(OPT_DOMAIN, OPT_DOMAIN_LONG, true, OPT_DOMAIN_DESC);
		_cliOptions.addOption(OPT_SSO, OPT_SSO_LONG, true, OPT_SSO_DESC);

		_cliOptions.addOption(OPT_OPERATION, true, OPT_OPERATION_DESC);
		_cliOptions.getOption(OPT_OPERATION).setRequired(true);

		_cliOptions.addOption(OPT_CONTACT_ID, OPT_CONTACT_ID_LONG, true, OPT_CONTACT_ID_DESC);
		_cliOptions.addOption(OPT_OUT_OF_PROVIDER_ADDRESS, OPT_OUT_OF_PROVIDER_ADDRESS_LONG, true, OPT_OUT_OF_PROVIDER_ADDRESS_DESC);
		_cliOptions.addOption(OPT_INTRINSICS, OPT_INTRINSICS_LONG, true, OPT_INTRINSICS_DESC);
		_cliOptions.addOption(OPT_SUBSCRIPTION_ID, OPT_SUBSCRIPTION_ID_LONG, true, OPT_SUBSCRIPTION_ID_DESC);
		
		_cliOptions.addOption(OPT_LISTENER_IP, OPT_LISTENER_IP_LONG, true, OPT_LISTENER_IP_DESC);
		_cliOptions.addOption(OPT_LISTENER_PORT, OPT_LISTENER_PORT_LONG, true, OPT_LISTENER_PORT_DESC);
		
		_cliOptions.addOption(OPT_LOAD_TEST_RATE, OPT_LOAD_TEST_RATE_LONG, true, OPT_LOAD_TEST_RATE_DESC);
		_cliOptions.addOption(OPT_LOAD_TEST_DURATION, OPT_LOAD_TEST_DURATION_LONG, true, OPT_LOAD_TEST_DURATION_DESC);
	}

	private void setupAuthenticationDetails(CommandLine cl) {
		_authenticationLevel.setUsername(DEFAULT_USERNAME);
		_authenticationLevel.setPassword(DEFAULT_PASSWORD);
		_authenticationLevel.setDomain(DEFAULT_DOMAIN);

		if ( cl.hasOption(OPT_USERNAME) ) {
			_authenticationLevel.setUsername(cl.getOptionValue(OPT_USERNAME));
		}
		if ( cl.hasOption(OPT_PASSWORD) ) {
			_authenticationLevel.setPassword(cl.getOptionValue(OPT_PASSWORD));
		}
		if ( cl.hasOption(OPT_DOMAIN) ) {
			_authenticationLevel.setDomain(cl.getOptionValue(OPT_DOMAIN));
		}
	}

	private CommandLine parseCommandLineOptions(String args[]) {
		CommandLine cl = null;

		try {
			BasicParser parser = new BasicParser();
			cl = parser.parse(_cliOptions, args);

			if ( cl.hasOption(OPT_HELP) ) {
				HelpFormatter formatter = new HelpFormatter();
				formatter.printHelp( "java -jar soa_oi_openq.jar", _cliOptions );
				System.exit(1);
			}

			if ( !cl.hasOption(OPT_WSDL) ) {
				System.err.println("Must specify wsdl");
				HelpFormatter formatter = new HelpFormatter();
				formatter.printHelp( "OIOpenQ", _cliOptions );
				System.exit(1);
			} 
			
			if ( !cl.hasOption(OPT_OPERATION) ) {
				System.err.println("Must specify operation");
				HelpFormatter formatter = new HelpFormatter();
				formatter.printHelp( "OIOpenQ", _cliOptions );
				System.exit(1);
			}
			
			String operation = cl.getOptionValue(OPT_OPERATION);
			if ( operation.equals(OPT_LOGON) ) {
				_selectedOperation = OPERATION.LOGON;
				
			} else if ( operation.equals(OPT_LOGOFF) ) {
				_selectedOperation = OPERATION.LOGOFF;
				
				if ( !cl.hasOption(OPT_SSO) ) {
					System.err.println("Must specify sso");
					HelpFormatter formatter = new HelpFormatter();
					formatter.printHelp( "OIOpenQ", _cliOptions );
					System.exit(1);
				}
			} else if ( operation.equals(OPT_CREATE_CONTACT) ) {
				_selectedOperation = OPERATION.CREATE;
				
				if ( ( cl.hasOption(OPT_LOAD_TEST_RATE) ) || ( cl.hasOption(OPT_LOAD_TEST_DURATION) ) ) {
					_isLoadTestRequired = true;
					String loadTestRate = cl.getOptionValue(OPT_LOAD_TEST_RATE);
					String loadTestDuration = cl.getOptionValue(OPT_LOAD_TEST_DURATION);
					if ( loadTestRate != null ) {
						_loadTestRatePerSec = Integer.valueOf(loadTestRate).intValue();
					}
					if ( loadTestDuration != null ) {
						_loadTestDurationInSec = Integer.valueOf(loadTestDuration).intValue();
					}
				}
				
				_contactDetails = new ContactDetails(
						DEFAULT_CONTACT_ID,
						DEFAULT_OUT_OF_PROVIDER_ADDRESS,
						null );
				
				if (cl.hasOption(OPT_CONTACT_ID) ) {
					_contactDetails.setExternalContactId(Integer.valueOf(cl.getOptionValue(OPT_CONTACT_ID)).intValue());
				}
				if (cl.hasOption(OPT_INTRINSICS) ) {
					HashMap<String, String> intrinsics = new HashMap<String, String>();
					String intrinsicKeyValueListStr = cl.getOptionValue(OPT_INTRINSICS);
					intrinsics = extractIntrinsics(intrinsicKeyValueListStr);
					_contactDetails.setIntrinsics(intrinsics);
				} 
			} else if ( operation.equals(OPT_DROP_CONTACT) ) {
				_selectedOperation = OPERATION.DROP;
				
				if ( ( cl.hasOption(OPT_LOAD_TEST_RATE) ) || ( cl.hasOption(OPT_LOAD_TEST_DURATION) ) ) {
					_isLoadTestRequired = true;
					String loadTestRate = cl.getOptionValue(OPT_LOAD_TEST_RATE);
					String loadTestDuration = cl.getOptionValue(OPT_LOAD_TEST_DURATION);
					if ( loadTestRate != null ) {
						_loadTestRatePerSec = Integer.valueOf(loadTestRate).intValue();
					}
					if ( loadTestDuration != null ) {
						_loadTestDurationInSec = Integer.valueOf(loadTestDuration).intValue();
					}
				}
				
				_contactDetails = new ContactDetails(
						DEFAULT_CONTACT_ID,
						null, 
						null);
				
				if (cl.hasOption(OPT_CONTACT_ID) ) {
					_contactDetails.setExternalContactId(Integer.valueOf(cl.getOptionValue(OPT_CONTACT_ID)).intValue());
				}
			} else if ( operation.equals(OPT_GET_CONTACT) ) {
				_selectedOperation = OPERATION.GET;
				
				_contactDetails = new ContactDetails(
						DEFAULT_CONTACT_ID,
						null,
						null );
				
				if (cl.hasOption(OPT_CONTACT_ID) ) {
					_contactDetails.setExternalContactId(Integer.valueOf(cl.getOptionValue(OPT_CONTACT_ID)).intValue());
				}
			} else if ( operation.equals(OPT_ADD_SERVICE_PROVIDER_STATUS_LISTENER) ) {
				_selectedOperation = OPERATION.ADD_SERVICE_PROVIDER_LISTENER;
				
				if (!cl.hasOption(OPT_LISTENER_IP) ) {
					System.err.println("Must specify valid consumer ip address");
					HelpFormatter formatter = new HelpFormatter();
					formatter.printHelp( "OIOpenQ", _cliOptions );
					System.exit(1);
				} else {
					_listenerIp = cl.getOptionValue(OPT_LISTENER_IP);
				}
				
				if (!cl.hasOption(OPT_LISTENER_PORT) ) {
					System.err.println("Must specify valid consumer port");
					HelpFormatter formatter = new HelpFormatter();
					formatter.printHelp( "OIOpenQ", _cliOptions );
					System.exit(1);
				} else {
					_listenerPort = cl.getOptionValue(OPT_LISTENER_PORT);
				}
			} else if ( operation.equals(OPT_REMOVE_SERVICE_PROVIDER_STATUS_LISTENER) ) {
				_selectedOperation = OPERATION.REMOVE_SERVICE_PROVIDER_LISTENER;
				
				if (!cl.hasOption(OPT_SUBSCRIPTION_ID) ) {
					System.err.println("Must specify valid subscription id");
					HelpFormatter formatter = new HelpFormatter();
					formatter.printHelp( "OIOpenQ", _cliOptions );
					System.exit(1);
				} else {
					_subscriptionId = cl.getOptionValue(OPT_SUBSCRIPTION_ID);
				}
			} else if ( operation.equals(OPT_ADD_FORCE_DISCONNECT_LISTENER) ) {
				_selectedOperation = OPERATION.ADD_FORCE_DISCONNECT_LISTENER;
				
				if (!cl.hasOption(OPT_LISTENER_IP) ) {
					System.err.println("Must specify valid consumer ip address");
					HelpFormatter formatter = new HelpFormatter();
					formatter.printHelp( "OIOpenQ", _cliOptions );
					System.exit(1);
				} else {
					_listenerIp = cl.getOptionValue(OPT_LISTENER_IP);
				}
				
				if (!cl.hasOption(OPT_LISTENER_PORT) ) {
					System.err.println("Must specify valid consumer port");
					HelpFormatter formatter = new HelpFormatter();
					formatter.printHelp( "OIOpenQ", _cliOptions );
					System.exit(1);
				} else {
					_listenerPort = cl.getOptionValue(OPT_LISTENER_PORT);
				}
			} else if ( operation.equals(OPT_REMOVE_FORCE_DISCONNECT_LISTENER) ) {
				_selectedOperation = OPERATION.REMOVE_FORCE_DISCONNECT_LISTENER;
				
				if (!cl.hasOption(OPT_SUBSCRIPTION_ID) ) {
					System.err.println("Must specify valid subscription id");
					HelpFormatter formatter = new HelpFormatter();
					formatter.printHelp( "OIOpenQ", _cliOptions );
					System.exit(1);
				} else {
					_subscriptionId = cl.getOptionValue(OPT_SUBSCRIPTION_ID);
				}
			} else if ( operation.equals(OPT_ADD_DEFAULT_TREATMENT_LISTENER) ) {
				_selectedOperation = OPERATION.ADD_DEFAULT_TREATMENT_LISTENER;
				
				if (!cl.hasOption(OPT_LISTENER_IP) ) {
					System.err.println("Must specify valid consumer ip address");
					HelpFormatter formatter = new HelpFormatter();
					formatter.printHelp( "OIOpenQ", _cliOptions );
					System.exit(1);
				} else {
					_listenerIp = cl.getOptionValue(OPT_LISTENER_IP);
				}
				
				if (!cl.hasOption(OPT_LISTENER_PORT) ) {
					System.err.println("Must specify valid consumer port");
					HelpFormatter formatter = new HelpFormatter();
					formatter.printHelp( "OIOpenQ", _cliOptions );
					System.exit(1);
				} else {
					_listenerPort = cl.getOptionValue(OPT_LISTENER_PORT);
				}
			} else if ( operation.equals(OPT_REMOVE_DEFAULT_TREATMENT_LISTENER) ) {
				_selectedOperation = OPERATION.REMOVE_DEFAULT_TREATMENT_LISTENER;
				
				if (!cl.hasOption(OPT_SUBSCRIPTION_ID) ) {
					System.err.println("Must specify valid subscription id");
					HelpFormatter formatter = new HelpFormatter();
					formatter.printHelp( "OIOpenQ", _cliOptions );
					System.exit(1);
				} else {
					_subscriptionId = cl.getOptionValue(OPT_SUBSCRIPTION_ID);
				}
			} else {
				System.err.println("Must specify valid operation");
				HelpFormatter formatter = new HelpFormatter();
				formatter.printHelp( "OIOpenQ", _cliOptions );
				System.exit(1);
			}

		} catch (ParseException pEx) {
			pEx.printStackTrace();
			HelpFormatter formatter = new HelpFormatter();
			formatter.printHelp( "OIOpenQ", _cliOptions );
			System.exit(1);
		}

		return cl;
	}

	private OpenQ setupConnection( String wsdlLocation ) {

		OpenQ port = null;
		URL wsdlURL = null;
		File wsdlFile = null;
		try {
			wsdlFile = new File(wsdlLocation);
			if (wsdlFile.exists()) {
				wsdlURL = wsdlFile.toURL();
			} else {
				wsdlURL = new URL(wsdlLocation);
			}
		} catch (MalformedURLException e) {
			e.printStackTrace();
		}

		System.out.println("OIOpenQ connecting to : " + wsdlURL.toString());

		try { 
			SOAOIOpenQ ss = new SOAOIOpenQ(wsdlURL, OPENQ_SERVICE_QNAME);
			port = ss.getOpenQ();
			System.out.println("OIOpenQ connected to : " + wsdlURL.toString());
		} catch ( Exception ex ) {
			ex.printStackTrace();
		}

		return port;  
	}

	private static String printUserDetails( AuthenticationLevel auth ) {
		return "User:"+auth.getUsername() + " @ " + auth.getDomain(); 
	}

	
	class ShutdownHook extends Thread {
	    public void run() {
	        System.out.println("Shutting down");
	    }
	}
	
	public static void main(String args[]) throws InterruptedException {

		OIOpenQ openQ = new OIOpenQ();

		openQ.setupOptions();
		CommandLine cl = openQ.parseCommandLineOptions(args);
		openQ.setupAuthenticationDetails(cl);

		_openQPort = openQ.setupConnection(cl.getOptionValue(OPT_WSDL));
		if ( _openQPort == null ) {
			System.err.println("Failed to setup connection to : " + cl.getOptionValue(OPT_WSDL) );
			System.exit(1);
		}
		
		boolean isAutoLogOnOffRequired = true;
		if ( cl.hasOption(OPT_SSO) ){
			// No need to issue logOn request...
			isAutoLogOnOffRequired = false;
			String ssoId = cl.getOptionValue(OPT_SSO);
			_sso = new SsoToken();
			_sso.setToken(ssoId);
		}
		
		Contact contact = null;
		
		switch (_selectedOperation) {
		
		case LOGON:
			System.out.println("Invoking logOn...");
			logOn();
			isAutoLogOnOffRequired = false;
			break;
		case LOGOFF: 
			System.out.println("Invoking logOff...");
			if ( issueLogOffRequest(_openQPort, _sso ) ) {
				System.out.println("Logged off succesfully...");
			} else {
				System.out.println("Log off failed...");
			}
			break;
		case CREATE:
			if ( isAutoLogOnOffRequired ) {
				logOn();
			} 
			if ( _isLoadTestRequired ) {
				System.out.println("Invoking createContact @ " + _loadTestRatePerSec + " contacts per second for " + _loadTestDurationInSec + " seconds");
				
				long startTime = System.currentTimeMillis();
				for ( int testDuration = 0; testDuration < _loadTestDurationInSec; testDuration++ ) {
					for ( int i = 0; i < _loadTestRatePerSec; i ++ ) {
						contact = issueCreateContact(new CreateContactDetails(_contactDetails), _sso);
						_contactDetails.setExternalContactId(_contactDetails.getExternalContactId() + 1);
//						Integer outOfProviderAddressExtId = new Integer(_contactDetails.getOutOfProviderAddress());
//						outOfProviderAddressExtId++; 
//						_contactDetails.setOutOfProviderAddress(outOfProviderAddressExtId.toString());
					}
				}
				
				long endTime = System.currentTimeMillis();
				long totalTime = endTime - startTime;

				int numberOfContactsCreated = _loadTestRatePerSec * _loadTestDurationInSec;
				long averageTimePerContact = totalTime/numberOfContactsCreated;
				System.out.println("Created " + numberOfContactsCreated + " contacts in " + totalTime + "ms");
				System.out.println("Average time to create contact " + averageTimePerContact + "ms");
				
			} else {
				System.out.println("Invoking createContact...");
				System.out.println(_contactDetails.toString());
				contact = issueCreateContact(new CreateContactDetails(_contactDetails), _sso);
				if ( contact != null ) {
					System.out.println("Created contact : externalContactId["+contact.getExternalContactId()+"]");
				} else {
					System.out.println("Contact creation failed for : externalContactId["+_contactDetails.getExternalContactId()+"]");
				}
			}
			
			if ( isAutoLogOnOffRequired ) {
				logOff();
			}
			break;
		case DROP:
			if ( isAutoLogOnOffRequired ) {
				logOn();
			}
			System.out.println("Invoking dropContact...");
			System.out.println(_contactDetails.toString());
			boolean isContactDropped = issueDropContact(new DropContactDetails(_contactDetails), _sso);
			if ( isContactDropped ) {
				System.out.println("Dropped contact : externalContactId["+_contactDetails.getExternalContactId()+"]");
			} else {
				System.out.println("Failed to drop Contact : externalContactId["+_contactDetails.getExternalContactId()+"]");
			}
			if ( isAutoLogOnOffRequired ) {
				logOff();
			}
			break;
		case GET:
			if ( isAutoLogOnOffRequired ) {
				logOn();
			}
			System.out.println("Invoking getContact...");
			System.out.println(_contactDetails.toString());
			contact = issueGetContact(new GetContactDetails(_contactDetails), _sso);
			if ( contact != null ) {
				System.out.println("Retrieved contact : externalContactId["+contact.getExternalContactId()+"]");
			} else {
				System.out.println("Failed to retrieve contact : externalContactId["+_contactDetails.getExternalContactId()+"]");
			}
			if ( isAutoLogOnOffRequired ) {
				logOff();
			}
			break;
		case ADD_SERVICE_PROVIDER_LISTENER: 
			System.out.println("Invoking addServiceProviderListener...");
			issueAddServiceProviderListenerRequest( _openQPort, _sso );
			break;
		case REMOVE_SERVICE_PROVIDER_LISTENER:
			System.out.println("Invoking removeServiceProviderListener...");
			issueRemoveServiceProviderListenerRequest( _openQPort, _subscriptionId, _sso );
			break;
		case ADD_FORCE_DISCONNECT_LISTENER:
			System.out.println("Invoking addForceDisconnectListener...");
			issueAddForceDisconnectListenerRequest( _openQPort, _sso );
			break;
		case REMOVE_FORCE_DISCONNECT_LISTENER:
			System.out.println("Invoking removeForceDisconnectListener...");
			issueRemoveForceDisconnectListenerRequest( _openQPort, _subscriptionId, _sso );
			break;
		case ADD_DEFAULT_TREATMENT_LISTENER:
			System.out.println("Invoking addDefaultTreatmentListener...");
			issueAddDefaultTreatmentListenerRequest( _openQPort, _sso );
			break;
		case REMOVE_DEFAULT_TREATMENT_LISTENER:
			System.out.println("Invoking removeDefaultTreatmentListener...");
			issueRemoveDefaultTreatmentListenerRequest( _openQPort, _subscriptionId, _sso );
			break;
		default:
			System.err.println("Unknown operation : " + _selectedOperation.toString() );
			System.exit(1);
			break;
		}
	}
	
	private static void logOn() {
		_sso = issueLogOnRequest(_openQPort, _authenticationLevel);
		if ( _sso == null ) {
			System.err.println("LogOn failed for : " + printUserDetails(_authenticationLevel) );
			System.exit(1);
		} else { 
			System.out.println(printUserDetails(_authenticationLevel) + " logOn succeeded, " + _sso.toString());
		}
	}

	private static SsoToken issueLogOnRequest( OpenQ port, AuthenticationLevel auth ) {
		SsoToken sso = null;

		if ( auth != null ) {
			if ( auth.getUsername() == null ) {
				System.err.println("Username not specified");
			} else if ( auth.getPassword() == null ) {
				System.err.println("Password not specified");
			} else {
				try {
					sso = port.logIn(auth);
				} catch (LogInFailedFault ex) {
					System.err.println("Cause : " + (ex.getCause() != null ? ex.getCause() : "unknown"));
					System.err.println("Message : " + (ex.getMessage() != null ? ex.getMessage() : "unknown"));
					ex.printStackTrace();
					sso = null;
				}
			}
		} else {
			System.err.println("No authentication details specified");
		}

		return sso;
	}
	
	private static void logOff() {
		boolean isLoggedOff = issueLogOffRequest(_openQPort, _sso);
		if ( isLoggedOff ) {
			System.out.println("Logged off succeeded with sso : " + _sso.toString());
		} else {
			System.err.println("LogOff failed ");
		}
	}

	private static boolean issueLogOffRequest( OpenQ port, SsoToken sso ) {
		boolean isLogOffSuccessful = false;
		if ( sso != null ) {
			try {
				isLogOffSuccessful = port.logOff(sso);
			} catch (LogOffFailedFault ex) {
				ex.printStackTrace();
				sso = null;
			}
		} else {
			System.err.println("No SSO specified");
		}
		return isLogOffSuccessful;
	}

	private static Contact issueCreateContact( CreateContactInterface contact, SsoToken sso ) {
		
		Contact newContact = null;

		try {
			newContact = issueCreateContactRequest(
					_openQPort, 
					contact.getExternalContactId(), 
					contact.getOutOfProviderAddress(),
					contact.getIntrinsics(),
					sso);
		} catch (CreateOQContactFailedFault ccEx) {
			ccEx.printStackTrace();
		} catch (Exception ex) {
			ex.printStackTrace();
		}
		
		return newContact;
	}
	
	private static Contact issueCreateContactRequest( 
			OpenQ port,
			String externalContactId, 
			String outOfProviderAddressName,
			HashMap<String, String> intrinsicsMap,
			SsoToken sso) 
	throws CreateOQContactFailedFault {

		IntrinsicArray intrinsics = new IntrinsicArray();
		if ( intrinsicsMap != null ) {
			Iterator<String> specifiedIntrinsicKeysIter = intrinsicsMap.keySet().iterator();
			while ( specifiedIntrinsicKeysIter.hasNext() ) {
				String intrinsicKeyStr = specifiedIntrinsicKeysIter.next();
				String intrinsicValueStr = intrinsicsMap.get(intrinsicKeyStr);
	
				Intrinsic intrinsic = WsFactoryManager.wsbt().createIntrinsic();
				intrinsic.setKey(intrinsicKeyStr);
				intrinsic.setValue(intrinsicValueStr);
				intrinsic.setImmutable(true);
	
				intrinsics.getItem().add(intrinsic);
			}
		}

		Holder<Contact> contactHolder = new Holder<Contact>();
		contactHolder.value = WsFactoryManager.wsbt().createContact();

		return port.createOQContact(
				externalContactId, 
				outOfProviderAddressName, 
				intrinsics, 
				sso);
	}
	
	private static boolean issueDropContact( DropContactInterface contact, SsoToken sso ) {
		
		boolean isContactDropped = false;

		try {
			isContactDropped = issueDropContactRequest(
						_openQPort, 
						contact.getExternalContactId(), 
						sso);
		} catch (DropOQContactFailedFault dcEx) {
			dcEx.printStackTrace();
		} catch (Exception ex) {
			ex.printStackTrace();
		}
		
		return isContactDropped;
	}
	
	private static boolean issueDropContactRequest( 
			OpenQ port,
			String contactId, 
			SsoToken sso) throws DropOQContactFailedFault {

		return port.dropOQContact(contactId, sso ); 
	}

	private static Contact issueGetContact( GetContactInterface contact, SsoToken sso ) {
		
		Contact theContact = null;

		try {
			theContact = issueGetContactRequest(
						_openQPort, 
						contact.getExternalContactId(), 
						sso);
		} catch (GetOQContactFailedFault gcEx) {
			gcEx.printStackTrace();
		} catch (Exception ex) {
			ex.printStackTrace();
		}
		
		return theContact;
	}

	private static Contact issueGetContactRequest( 
			OpenQ port,
			String externalContactId, 
			SsoToken sso) 
	throws GetOQContactFailedFault {

		return port.getOQContact(externalContactId, sso);
	}
	

	private static void issueAddServiceProviderListenerRequest( OpenQ port, SsoToken sso ) {
		
		String _consumerAddress;

		_consumerAddress = "http://"+_listenerIp+":"+_listenerPort+"/" + Definitions.CONSUMER_ADDRESS;
		System.out.println("Starting Consumer Server @ " + _consumerAddress);
		
//		if (System.getProperty("test.consumer.tls.on").equals("true")) {
//			_consumerAddress = "https://localhost:9099/" + Definitions.CONSUMER_ADDRESS;
//		} 

		Endpoint.publish(_consumerAddress, _consumerImplementor);
		System.out.println("Consumer Server ready");
		
		String subscriptionReference = null;
		try {
			subscriptionReference = port.addServiceProviderStatusListener(_consumerAddress, sso);
			System.out.println("Subscription reference : " + subscriptionReference);
		} catch (AddServiceProviderStatusListenerFailedFault ex) {
			ex.printStackTrace();
		}

		while (true) {
			System.out.print(".");
			try {
				Thread.sleep(1000);
			} catch (InterruptedException ex) {
				ex.printStackTrace();
			}
		}
	}
	
	private static void issueRemoveServiceProviderListenerRequest( OpenQ port, String subscriptionId, SsoToken sso ) {
		
		try {
			boolean removedSubscription = port.removeServiceProviderStatusListener(subscriptionId, sso);
			if ( removedSubscription ) {
				System.out.println("Removed ServiceProviderStatus listener subscription");
			} else {
				System.err.println("Failed to remove ServiceProviderStatus listener subscription");
			}
		} catch (RemoveServiceProviderStatusListenerFailedFault ex) {
			ex.printStackTrace();
		}

		while (true) {
			System.out.print(".");
			try {
				Thread.sleep(1000);
			} catch (InterruptedException ex) {
				ex.printStackTrace();
			}
		}
	}
	
	private static void issueAddForceDisconnectListenerRequest( OpenQ port, SsoToken sso ) {
		
		String _consumerAddress;

		_consumerAddress = "http://"+_listenerIp+":"+_listenerPort+"/" + Definitions.CONSUMER_ADDRESS;
		System.out.println("Starting Consumer Server @ " + _consumerAddress);
		
//		if (System.getProperty("test.consumer.tls.on").equals("true")) {
//			_consumerAddress = "https://localhost:9099/" + Definitions.CONSUMER_ADDRESS;
//		} 

		Endpoint.publish(_consumerAddress, _consumerImplementor);
		System.out.println("Consumer Server ready");
		
		String subscriptionReference = null;
		try {
			subscriptionReference = port.addForceDisconnectListener(_consumerAddress, sso);
			System.out.println("Subscription reference : " + subscriptionReference);
		} catch (AddForceDisconnectListenerFailedFault ex) {
			ex.printStackTrace();
		}

		while (true) {
			System.out.print(".");
			try {
				Thread.sleep(1000);
			} catch (InterruptedException ex) {
				ex.printStackTrace();
			}
		}
	}
	
	private static void issueRemoveForceDisconnectListenerRequest( OpenQ port, String subscriptionId, SsoToken sso ) {
		
		try {
			boolean removedSubscription = port.removeForceDisconnectListener(subscriptionId, sso);
			if ( removedSubscription ) {
				System.out.println("Removed ForceDisconnect listener subscription");
			} else {
				System.err.println("Failed to remove ForceDisconnect listener subscription");
			}
		} catch (RemoveForceDisconnectListenerFailedFault ex) {
			ex.printStackTrace();
		}

		while (true) {
			System.out.print(".");
			try {
				Thread.sleep(1000);
			} catch (InterruptedException ex) {
				ex.printStackTrace();
			}
		}
	}
	
	private static void issueAddDefaultTreatmentListenerRequest( OpenQ port, SsoToken sso ) {
		
		String _consumerAddress;

		_consumerAddress = "http://"+_listenerIp+":"+_listenerPort+"/" + Definitions.CONSUMER_ADDRESS;
		System.out.println("Starting Consumer Server @ " + _consumerAddress);

//		if (System.getProperty("test.consumer.tls.on").equals("true")) {
//			_consumerAddress = "https://localhost:9099/" + Definitions.CONSUMER_ADDRESS;
//		} 

		Endpoint.publish(_consumerAddress, _consumerImplementor);
		System.out.println("Consumer Server ready");
		
		String subscriptionReference = null;
		try {
			subscriptionReference = port.addDefaultTreatmentListener(_consumerAddress, sso);
			System.out.println("Subscription reference : " + subscriptionReference);
		} catch (AddDefaultTreatmentListenerFailedFault ex) {
			ex.printStackTrace();
		}

		while (true) {
			System.out.print(".");
			try {
				Thread.sleep(1000);
			} catch (InterruptedException ex) {
				ex.printStackTrace();
			}
		}
	}
	
	private static void issueRemoveDefaultTreatmentListenerRequest( OpenQ port, String subscriptionId, SsoToken sso ) {
		
		try {
			boolean removedSubscription = port.removeDefaultTreatmentListener(subscriptionId, sso);
			if ( removedSubscription ) {
				System.out.println("Removed DefaultTreatment listener subscription");
			} else {
				System.err.println("Failed to remove DefaultTreatment listener subscription");
			}
		} catch (RemoveDefaultTreatmentListenerFailedFault ex) {
			ex.printStackTrace();
		}

		while (true) {
			System.out.print(".");
			try {
				Thread.sleep(1000);
			} catch (InterruptedException ex) {
				ex.printStackTrace();
			}
		}
	}
	
	/**
	 * Intrinsic key-value pairs detailed in space delimited string, with
	 * the key value pair delimited using comma; 
	 * e.g. "Key1,Value1 Key2,Value2".
	 * 
	 * @param contactTypeStr
	 * @return
	 */
	private static HashMap<String, String> extractIntrinsics(String intrinsicsStr) {
		HashMap<String, String> intrinsics = null;

		if ( intrinsicsStr != null ) {
			String[] intrinsicKeyValuePairArray = intrinsicsStr.split(" ");
			if ( ( intrinsicKeyValuePairArray != null ) && ( intrinsicKeyValuePairArray.length > 0 ) ) {
				intrinsics = new HashMap<String, String>();
				for (String keyValuePair : intrinsicKeyValuePairArray) {
					String[] keyValueArray = keyValuePair.split(",");
					if ( keyValueArray.length == 2) {
						intrinsics.put(keyValueArray[0], keyValueArray[1]);
					}
				}
			}
		}
		return intrinsics;
	}
}

