package com.nortel.rc.dto;

import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Map.Entry;
import java.util.Vector;

public class UserContextDO {

	private List<TerminalTO> terminals = new ArrayList<TerminalTO>();
	private List<AddressTO> unmappedAddresses = new ArrayList<AddressTO>();
	private List<ContactTO> contacts = new ArrayList<ContactTO>();
	private List<ContactTO> routePointContacts = new ArrayList<ContactTO>();
	
	/**
	 * @return the terminals
	 */
	public List<TerminalTO> getTerminals() {
		return terminals;
	}
	/**
	 * @param terminals the terminals to set
	 */
	public void setTerminals(List<TerminalTO> terminals) {
		this.terminals = terminals;
	}
	/**
	 * @return the unmappedAddresses
	 */
	public List<AddressTO> getUnmappedAddresses() {
		return unmappedAddresses;
	}
	/**
	 * @param unmappedAddresses the unmappedAddresses to set
	 */
	public void setUnmappedAddresses(List<AddressTO> unmappedAddresses) {
		this.unmappedAddresses = unmappedAddresses;
	}
	
	public Vector<CapabilityTO> findCapabilitySet(String parent, String child)
	{		
		//Check if it's an address
		for(TerminalTO terminal : terminals)
		{
			if(terminal.getTerminalName().equals(parent))
			{
				for(AddressTO address : terminal.getAddresses())
				{
					if(address.getAddressName().equals(child))
					{
						return new Vector<CapabilityTO>(address.getCapabilities().getCapabilityList());
					}
				}
			}
		}
		
		//Check it it's a terminal
		for(TerminalTO terminal : terminals)
		{
			if(terminal.getTerminalName().equals(child))
			{
				Vector<CapabilityTO> capabilities = new Vector<CapabilityTO>(terminal.getTerminalCapabilities().getCapabilityList());
				capabilities.addAll(terminal.getAgentTerminalCapabilities().getCapabilityList());
				
				return capabilities;
			}
		}

		//Finally check if it's an unmapped address
		if(unmappedAddresses != null)
		{
			for(AddressTO address : unmappedAddresses)
			{
				if(address.getAddressName().equals(child))
				{
					return new Vector<CapabilityTO>(address.getCapabilities().getCapabilityList());
				}
			}
		}
		
		return null;
	}
	
	public TerminalTO getTerminal(String terminalName)
	{
		if(terminals != null)
		{
			for(TerminalTO terminal : terminals)
			{
				if(terminal.getTerminalName().equals(terminalName))
				{
					return terminal;
				}
			}
		}
		return null;
	}
	
	public void updateContact(ContactTO contact)
	{			
		if(contacts.size() == 0)
		{
			contacts.add(contact);
		}
		else
		{
			boolean found = false;
			for(int i=0; i<contacts.size(); i++)
			{
				if(contacts.get(i).getContactId().equals(contact.getContactId()))
				{
					ContactTO contactToUpdate = contacts.get(i);
					contactToUpdate.setCalledAddress(contact.getCalledAddress());
					contactToUpdate.setCallingAddress(contact.getCallingAddress());
					contactToUpdate.setCallingTerminal(contact.getCallingTerminal());
					contactToUpdate.setLastRedirectedAddress(contact.getLastRedirectedAddress());
					contactToUpdate.setOriginalDestinationAddress(contact.getOriginalDestinationAddress());
					contactToUpdate.setAttachedData(contact.getAttachedData());
					contactToUpdate.setUuiData(contact.getUuiData());
					contactToUpdate.setContactTypes(contact.getContactTypes());
					contactToUpdate.setConnections(contact.getConnections());
					contactToUpdate.setExternalContactId(contact.getExternalContactId());
					contacts.set(i, contactToUpdate);
					found = true;
					break;
				}
			}
			if(!found)
			{
				contacts.add(contact);
			}
		}
	}
	
//	public void updateRoutePointAddress(AddressTO address)
//	{			
//		if(unmappedAddresses.size() == 0)
//		{
//			unmappedAddresses.add(address);
//		}
//		else
//		{
//			boolean found = false;
//			for(int i=0; i<unmappedAddresses.size(); i++)
//			{
//				if(unmappedAddresses.get(i).getAddressName().equals(address.getAddressName()))
//				{
//					unmappedAddresses.get(i).setControlled(address.isControlled());
//					found = true;
//					break;
//				}
//			}
//			if(!found)
//			{
//				unmappedAddresses.add(address);
//			}
//		}
//	}
	
	public void updateContactCapabilities(ContactTO contact)
	{
		boolean found = false;
		int indexToUpdate = 0;
		for(int i=0; i<contacts.size(); i++)
		{
			if(contacts.get(i).getContactId().equals(contact.getContactId()))
			{
				indexToUpdate = i;
				found = true;
				break;
			}
		}
		if(found)
		{
			//Update the Capabilities only
			contacts.get(indexToUpdate).setCapabilities(contact.getCapabilities());
			//Update each of the Connection Capabilities
			for(ConnectionTO updatedConnection : contact.getConnections())
			{
				ConnectionTO foundConnection = findConnection(updatedConnection.getConnectionId()); 
				if (foundConnection != null)
					foundConnection.setCapabilities(updatedConnection.getCapabilities());
				
				//Update each of the TerminalConnection Capabilities
				TerminalConnectionTO foundTerminalConnection = null;
				for(TerminalConnectionTO updatedTermConn : updatedConnection.getTerminalConnections())
				{
					foundTerminalConnection = findTerminalConnection(updatedTermConn.getTerminalConnectionId());
					if (foundTerminalConnection != null)		
						foundTerminalConnection.setCapabilities(updatedTermConn.getCapabilities());
				}
			}
		}
		else
		{
			contacts.add(contact);
		}
	}
	
	//Finds a contact that matches the (external)contactId passed
	public ContactTO findContact(String id)
	{		
		for(ContactTO contact : contacts)
		{
			if(contact.getContactId().equals(id) || contact.getExternalContactId().equals(id))
			{
				return contact;
			}
		}
		return null;
	}
	
	public ContactTO findRoutePointContact(String id)
	{		
		for(ContactTO contact : routePointContacts)
		{
			if(contact.getContactId().equals(id) || contact.getExternalContactId().equals(id))
			{
				return contact;
			}
		}
		return null;
	}
	
	public ConnectionTO findConnection(String id)
	{
		ConnectionTO result = null;
		for(ContactTO contact : contacts)
		{			
			ConnectionTO connection = contact.getConnectionById(id);
			if(connection != null)
			{
				result = connection;
				break;
			}		
		}
		return result;		
	}
	
	public TerminalConnectionTO findTerminalConnection(String id)
	{
		TerminalConnectionTO result = null;
		for(ContactTO contact : contacts)
		{			
			for(ConnectionTO connection : contact.getConnections())
			{
				TerminalConnectionTO termConn = connection.getTerminalConnection(id);
				if(termConn != null)
				{
					result = termConn;
				}	
			}		
		}
		return result;
	}
	
	public AddressTO getAddress(String addressName)
	{
		//Check if it's an address
		for(TerminalTO terminal : terminals)
		{
			for(AddressTO address : terminal.getAddresses())
			{
				if(address.getAddressName().equals(addressName))
				{
					return address;
				}
			}
			
		}

		//Finally check if it's an unmapped address
		if(unmappedAddresses != null)
		{
			for(AddressTO address : unmappedAddresses)
			{
				if(address.getAddressName().equals(addressName))
				{
					return address;
				}
			}
		}
		return null;
	}
	
	public TerminalTO deriveTerminal(String addressName)
	{
		//Check if it's an address
		for(TerminalTO terminal : terminals)
		{
			for(AddressTO address : terminal.getAddresses())
			{
				if(address.getAddressName().equals(addressName))
				{
					return terminal;
				}
			}
			
		}		
		return null;
	}
	
	public String deriveTerminalName(String addressName){
		TerminalTO terminal = deriveTerminal(addressName);
		
		return terminal == null ? "" : terminal.getTerminalName(); 
	}
	
	public void removeContact(String id)
	{
		int contactToRemove = -1;
		for(int i=0; i<contacts.size(); i++)
		{
			if(contacts.get(i).getContactId().equals(id))
			{
				contactToRemove = i;
			}
		}
		if(contactToRemove > -1)
		{
			contacts.remove(contactToRemove);
		}
	}
	
	public void removeRoutePointContact(String id)
	{
		int contactToRemove = -1;
		for(int i=0; i<routePointContacts.size(); i++)
		{
			if(routePointContacts.get(i).getContactId().equals(id))
			{
				contactToRemove = i;
			}
		}
		if(contactToRemove > -1)
		{
			routePointContacts.remove(contactToRemove);
		}
	}
	
	public void addContact(ContactTO contact)
	{
		contacts.add(contact);
	}
	
	public void addRoutePointContact(ContactTO contact)
	{
		routePointContacts.add(contact);
	}
	
	public void removeTerminal(String terminalName)
	{
		int indexToRemove = -1;
		int currentIndex = 0;
		for(TerminalTO terminal : terminals)
		{
			if(terminal.getTerminalName().equals(terminalName))			
			{
				indexToRemove = currentIndex;
				break;
			}
			currentIndex++;
		}	
		if(indexToRemove != -1)
		{
			terminals.remove(indexToRemove);
		}
	}
	
	public void removeAddress(String addressName)
	{
		boolean removed = false;
		
		for (TerminalTO terminal : terminals) {
			AddressTO address = terminal.getAddress(addressName);
			if (address != null) {
				terminal.getAddresses().remove(address);
				removed = true;
				break;
			}
		}
		
		if(!removed)
		{
			//Look for it in unmappedAddresses i.e. probably routepoint
			for(AddressTO address : unmappedAddresses)
			{
				if(address.getAddressName().equals(addressName))
				{
					unmappedAddresses.remove(address);
					break;
				}
			}
		}
	}
	
	public void addTerminal(TerminalTO terminal)
	{
		terminals.add(terminal);
	}
	
	public void addAddress(List<TerminalTO> terminals, AddressTO address)
	{
		if(terminals != null)
		{
			for(TerminalTO terminal : terminals)
			{ 
				TerminalTO term = getTerminal(terminal.getTerminalName()); 
				if(term != null)
				{
					boolean haveAlready = false;
					
					//Need to check if the address is already present before adding
					//This happens when Address State notification is received before
					//its parent Terminal's State notification
					for(AddressTO addr : term.getAddresses())
					{
						if(addr.getAddressName().equals(address.getAddressName()))
						{
							haveAlready = true;
							break;
						}
					}
					if(!haveAlready)
					{
						getTerminal(terminal.getTerminalName()).getAddresses().add(address);
					}
				}
			}
		}
		else
		{
			unmappedAddresses.add(address);
		}
	}
	
	//Method generates contact remote state to be displayed in contact list. e.g. "OFFERED,ESTABLISHED(2)"
	//Implementation is in sync with .Net RefClient 
	public String getContactRemoteState(StateEventTO stateEvent){
		StringBuilder result = new StringBuilder("");
		
		if (stateEvent != null){
					
			String contactId = stateEvent.getContactId();
			TerminalTO terminal = null;
			if (!stateEvent.isRemote() && stateEvent.getNotifiedAddress() != null){
				terminal = deriveTerminal(stateEvent.getNotifiedAddress());
			}
			else if (!stateEvent.getTerminal().equals(""))
				terminal = getTerminal(stateEvent.getTerminal());
			else
				terminal = deriveTerminal(stateEvent.getAddress());
								
			for(int i=0; i<contacts.size(); i++)
			{
				if(contacts.get(i).getContactId().equals(contactId))
				{	
					Map<String,Integer> map = new HashMap<String,Integer>();
					String connState = null;
					
					ContactTO contact = contacts.get(i);
					List<ConnectionTO> connections = contact.getConnections();
									
					for (ConnectionTO connection : connections){
										
						connState = connection.getStateSummary().getCurrent();
						if ((terminal == null && getAddress(connection.getAddressName()) == null)
							||(terminal != null && !terminal.getAddresses().contains(getAddress(connection.getAddressName())))){
						
							if (!connState.equalsIgnoreCase("DISCONNECTED")) 
								map.put(connState, ((map.get(connState) == null)?1:map.get(connState)+1));						
						}
					}
					
					for (Entry<String,Integer> entry : map.entrySet()){
						result.append(entry.getKey());
						
						if (entry.getValue() > 1)
							result.append('(').append(entry.getValue()).append(')');
						
						result.append(",");
					}				
					
					break;
				}
			}
		}
		
		if (!result.toString().equals("")){
			return result.substring(0,result.length()-1);
		}else{
			return "<n/a>";
		}		
	}
	
	public void addConnection(StateEventTO stateEvent, StateSummaryTO stateSummary) {
		if (stateEvent != null) {			
			ContactTO contact = findContact(stateEvent.getContactId());
			if (contact != null) {

				ConnectionTO newConnection = new ConnectionTO();
				newConnection.setAddressName(stateEvent.getAddress());
				newConnection.setConnectionId(stateEvent.getConnection().getConnectionId());
				newConnection.setStateSummary(stateSummary);

				contact.getConnections().add(newConnection);
			}
		}		
	}
	
	public void reset() {
		if (terminals != null) {
			terminals.clear();
		}
		if (contacts != null) {
			contacts.clear();
		}
		if (unmappedAddresses != null) {
			unmappedAddresses.clear();
		}
		if (routePointContacts != null) {
			routePointContacts.clear();
		}
	}
}
