package com.nortel.rc.event;

import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.concurrent.BlockingQueue;
import java.util.concurrent.LinkedBlockingQueue;

import org.springframework.core.task.TaskExecutor;

import org.apache.log4j.Level;
import org.apache.log4j.Logger;
import org.apache.log4j.PatternLayout;
import org.apache.log4j.RollingFileAppender;

import com.nortel.rc.dto.AddressCapabilitiesTO;
import com.nortel.rc.dto.AddressPropertyEventTO;
import com.nortel.rc.dto.AddressStateEventTO;
import com.nortel.rc.dto.AgentCapabilitiesTO;
import com.nortel.rc.dto.AgentPropertyEventTO;
import com.nortel.rc.dto.AgentStateEventTO;
import com.nortel.rc.dto.AttachedDataTO;
import com.nortel.rc.dto.ConnectionCapabilitiesTO;
import com.nortel.rc.dto.ConnectionTO;
import com.nortel.rc.dto.ContactCapabilitiesTO;
import com.nortel.rc.dto.ContactPropertyEventTO;
import com.nortel.rc.dto.IntrinsicTO;
import com.nortel.rc.dto.KeyValuePairTO;
import com.nortel.rc.dto.StateEventTO;
import com.nortel.rc.dto.TerminalCapabilitiesTO;
import com.nortel.rc.dto.TerminalConnectionCapabilitiesTO;
import com.nortel.rc.dto.TerminalConnectionTO;
import com.nortel.rc.dto.TerminalPropertyEventTO;
import com.nortel.rc.dto.TerminalStateEventTO;
import com.nortel.rc.dto.AddressPropertyEventTO.AddressProperty;
import com.nortel.rc.dto.AddressStateEventTO.AddressState;
import com.nortel.rc.dto.AgentPropertyEventTO.AgentProperty;
import com.nortel.rc.dto.AgentStateEventTO.AgentState;
import com.nortel.rc.dto.AttachedDataTO.AttachedDataType;
import com.nortel.rc.dto.ContactPropertyEventTO.ContactProperty;
import com.nortel.rc.dto.TerminalPropertyEventTO.TerminalProperty;
import com.nortel.rc.dto.TerminalStateEventTO.TerminalState;
import com.nortel.rc.gui.SOARefClient;

import com.nortel.soa.oi.cct.basenotification.AddressPropertyEventType;
import com.nortel.soa.oi.cct.basenotification.AddressStateEventType;
import com.nortel.soa.oi.cct.basenotification.AgentPropertyEventType;
import com.nortel.soa.oi.cct.basenotification.AgentStateEventType;
import com.nortel.soa.oi.cct.basenotification.ConnectionStateEventType;
import com.nortel.soa.oi.cct.basenotification.ContactPropertyEventType;
import com.nortel.soa.oi.cct.basenotification.NotificationEventHolderType;
import com.nortel.soa.oi.cct.basenotification.NotificationMessageHolderType;
import com.nortel.soa.oi.cct.basenotification.TerminalConnectionStateEventType;
import com.nortel.soa.oi.cct.basenotification.TerminalPropertyEventType;
import com.nortel.soa.oi.cct.basenotification.TerminalStateEventType;
import com.nortel.soa.oi.cct.notification.SystemEventType;
import com.nortel.soa.oi.cct.types.AddressCapabilities;
import com.nortel.soa.oi.cct.types.AddressPropertyAcquireStatus;
import com.nortel.soa.oi.cct.types.AddressPropertyDoNotDisturb;
import com.nortel.soa.oi.cct.types.AddressPropertyForwarding;
import com.nortel.soa.oi.cct.types.AddressPropertyForwardingArray;
import com.nortel.soa.oi.cct.types.AddressType;
import com.nortel.soa.oi.cct.types.AgentCapabilities;
import com.nortel.soa.oi.cct.types.AgentLoginStatus;
import com.nortel.soa.oi.cct.types.AttachedData;
import com.nortel.soa.oi.cct.types.ConnectionCapabilities;
import com.nortel.soa.oi.cct.types.ContactCapabilities;
import com.nortel.soa.oi.cct.types.Intrinsic;
import com.nortel.soa.oi.cct.types.KeyValuePair;
import com.nortel.soa.oi.cct.types.TerminalCapabilities;
import com.nortel.soa.oi.cct.types.TerminalConnectionCapabilities;
import com.nortel.soa.oi.cct.types.TerminalPropertyDoNotDisturb;
import com.nortel.soa.oi.cct.types.TerminalPropertyForwarding;
import com.nortel.soa.oi.cct.types.TerminalPropertyForwardingArray;

public final class NotificationManager
{
	public static String EVENT_LOG_FILE = "";

	private final Logger logger = Logger.getLogger(NotificationManager.class.getName());

	private final BlockingQueue<List<NotificationMessageHolderType>> incomingMessageQueue;

	private SOARefClient refClient;
	private final TaskExecutor taskExecutor;

	public NotificationManager(final TaskExecutor taskExecutor)
	{
		incomingMessageQueue = new LinkedBlockingQueue<List<NotificationMessageHolderType>>();

		initLogger();

		this.taskExecutor = taskExecutor;
		this.taskExecutor.execute(new NotiticationMessageConsumer(incomingMessageQueue));
	}

	/**
	 * Puts incoming notification messages into notification queue for subsequent processing.
	 * @param notificationMessageHolderList
	 */
	public void handleNotify(final List<NotificationMessageHolderType> notificationMessageHolderList)
	{
		incomingMessageQueue.add(notificationMessageHolderList);
	}

	public SOARefClient getRefClient()
	{
		return refClient;
	}

	public void setRefClient(final SOARefClient refClient)
	{
		this.refClient = refClient;
	}

	/**
	 * Creates event log file where all business logic events are written.
	 * File format is %timeofcreation_ms%_Event.log. 
	 */
	private void initLogger()
	{
		try
		{
			PatternLayout layout = new PatternLayout("%d - %m%n");
			EVENT_LOG_FILE = "./logs/" + java.lang.System.currentTimeMillis() + "_Event.log";
			RollingFileAppender appender = new RollingFileAppender(layout, EVENT_LOG_FILE, true);
			appender.setMaxBackupIndex(3);
			appender.setMaximumFileSize(100000);
			logger.addAppender(appender);
			logger.setAdditivity(false);
		}
		catch (final Exception e)
		{
			logger.error("Exception while setting up new event log file!", e);
		}
	}

	private class NotiticationMessageConsumer implements Runnable
	{
		private final BlockingQueue<List<NotificationMessageHolderType>> messageQueue;

		NotiticationMessageConsumer(final BlockingQueue<List<NotificationMessageHolderType>> messageQueue)
		{
			this.messageQueue = messageQueue;
		}

		@Override
		public void run()
		{
			while (!Thread.currentThread().isInterrupted())
			{
				try
				{
					List<NotificationMessageHolderType> messageList = messageQueue.take();
					processMessage(messageList);
				}
				catch (final InterruptedException e)
				{
					logger.error("Message processing thread has been interrupted!", e);
					Thread.currentThread().interrupt(); // should cause the loop to exit.
				}
				catch (final Exception e)
				{
					logger.error("Exception while processing message", e);
				}
			}
		}

		private void processMessage(final List<NotificationMessageHolderType> notificationMessageHolderList) throws Exception
		{
			for (NotificationMessageHolderType notificationMessageHolder : notificationMessageHolderList)
			{
				NotificationEventHolderType notificationEventHolder = notificationMessageHolder.getMessage();

				// Ensure that the subscription Id matches our latest subscription
				// Unsubscribe all others
				if (!notificationMessageHolder.getSubscriptionId().equals(NotificationHandler.SUBSCRIPTION_ID))
				{
					Map<String, Object> params = new HashMap<String, Object>();
					params.put("subscriptionId", notificationMessageHolder.getSubscriptionId());
					logger.log(Level.INFO, "Unsubscribing incorrect subscription: " + notificationMessageHolder.getSubscriptionId());
					getRefClient().getRequestHandler().handle("Unsubscribe", params);
					break;
				}

				if (notificationEventHolder.getAddressPropertyEvent() != null)
				{
					processAddressPropertyEvent(notificationEventHolder.getAddressPropertyEvent());
				}
				else if (notificationEventHolder.getTerminalPropertyEvent() != null)
				{
					processTerminalPropertyEvent(notificationEventHolder.getTerminalPropertyEvent());
				}
				else if (notificationEventHolder.getAgentPropertyEvent() != null)
				{
					processAgentPropertyEvent(notificationEventHolder.getAgentPropertyEvent());
				}
				else if (notificationEventHolder.getContactPropertyEvent() != null)
				{
					processContactPropertyEvent(notificationEventHolder.getContactPropertyEvent());
				}
				else if (notificationEventHolder.getConnectionStateEvent() != null)
				{
					processConnectionStateEvent(notificationEventHolder.getConnectionStateEvent());
				}
				else if (notificationEventHolder.getTerminalConnectionStateEvent() != null)
				{
					processTerminalConnectionStateEvent(notificationEventHolder.getTerminalConnectionStateEvent());
				}
				else if (notificationEventHolder.getAddressStateEvent() != null)
				{
					processAddressStateEvent(notificationEventHolder.getAddressStateEvent());
				}
				else if (notificationEventHolder.getTerminalStateEvent() != null)
				{
					processTerminalStateEvent(notificationEventHolder.getTerminalStateEvent());
				}
				else if (notificationEventHolder.getAgentStateEvent() != null)
				{
					processAgentStateEvent(notificationEventHolder.getAgentStateEvent());
				}
				else if (notificationEventHolder.getSystemEvent() != null)
				{
					processSystemEvent(notificationEventHolder.getSystemEvent());
				}
				else
				{
					logger.log(Level.WARN, "Unknown event received! Ignoring...");
				}
			}
		}

		private void processAddressPropertyEvent(final AddressPropertyEventType addressPropertyEvent)
		{
			logger.log(Level.INFO, "Address Property Event: AddressName [" + addressPropertyEvent.getAddress().getAddressName()
					+ "]\n AddressProperty [" + addressPropertyEvent.getAddressProperty() + "]");
			AddressPropertyEventTO addressPropertyEventTO = new AddressPropertyEventTO();

			addressPropertyEventTO.setRoutePoint(AddressType.ROUTE_POINT.equals(addressPropertyEvent.getAddress().getAddressType()));
			addressPropertyEventTO.setAddressName(addressPropertyEvent.getAddress().getAddressName());

			AddressCapabilities cctCaps = addressPropertyEvent.getAddressCapabilities();
			AddressCapabilitiesTO caps = new AddressCapabilitiesTO();
			caps.setCanConference(cctCaps.isCanConference());
			caps.setCanConference(cctCaps.isCanConsult());
			caps.setCanDoNotDisturb(cctCaps.isCanDoNotDisturb());
			caps.setCanForward(cctCaps.isCanForward());
			caps.setCanGetMessageWaiting(cctCaps.isCanGetMessageWaiting());
			caps.setCanOriginate(cctCaps.isCanOriginate());
			caps.setCanSetMessageWaiting(cctCaps.isCanSetMessageWaiting());
			caps.setCanTransfer(cctCaps.isCanTransfer());
			addressPropertyEventTO.setCapabilities(caps);

			switch (addressPropertyEvent.getAddressProperty())
			{
				case DO_NOT_DISTURB:
					addressPropertyEventTO.setPropertyType(AddressProperty.DO_NOT_DISTURB);
					AddressPropertyDoNotDisturb doNotDisturb = addressPropertyEvent.getAddressPropertyDoNotDisturb();
					addressPropertyEventTO.setActive(doNotDisturb.isDoNotDisturbSet());
					logger.log(Level.INFO, "Address Property: " + AddressProperty.DO_NOT_DISTURB.toString() + " "
							+ addressPropertyEvent.getAddress().getAddressName() + " " + addressPropertyEvent.getAddressProperty().value() + ": "
							+ doNotDisturb.isDoNotDisturbSet());
					break;

				case ACQUIRE_STATUS:
					addressPropertyEventTO.setPropertyType(AddressProperty.ACQUIRE_STATUS);
					AddressPropertyAcquireStatus acquireStatus = addressPropertyEvent.getAddressPropertyAcquireStatus();
					addressPropertyEventTO.setActive(acquireStatus.isAcquired());
					logger.log(Level.INFO, "Address Property: " + AddressProperty.ACQUIRE_STATUS.toString() + " "
							+ addressPropertyEvent.getAddress().getAddressName() + " " + addressPropertyEvent.getAddressProperty().value() + ": "
							+ acquireStatus.isAcquired());
					break;

				case CONTROLLED:
					// Since CONTROLLED event is ambiguous as to whether the RoutePoint has been acquired or
					// deacquired
					// This event is to act as an indicator that acquired status needs updating for this
					// address
					addressPropertyEventTO.setPropertyType(AddressProperty.CONTROLLED);
					logger.log(Level.INFO, "Address Property: " + AddressProperty.CONTROLLED.toString() + " "
							+ addressPropertyEvent.getAddress().getAddressName() + " " + addressPropertyEvent.getAddressProperty().value());
					break;

				case FORWARDING:
					addressPropertyEventTO.setPropertyType(AddressProperty.FORWARDING);
					AddressPropertyForwardingArray forwardingArr = addressPropertyEvent.getAddressPropertyForwardingArray();
					if (forwardingArr != null)
					{
						List<AddressPropertyForwarding> forwardingList = forwardingArr.getItem();
						if (forwardingList.size() > 0)
						{
							addressPropertyEventTO.setActive(true);
						}
					}
					logger.log(Level.INFO, "Address Property: " + AddressProperty.FORWARDING.toString() + " "
							+ addressPropertyEvent.getAddress().getAddressName() + " " + addressPropertyEvent.getAddressProperty().value() + ": "
							+ addressPropertyEventTO.isActive());
					break;

				case SPECIFIC_CAPABILITY:
					addressPropertyEventTO.setPropertyType(AddressProperty.SPECIFIC_CAPABILITY);
					logger.log(Level.INFO, "Address Property: " + AddressProperty.SPECIFIC_CAPABILITY.toString() + " "
							+ addressPropertyEvent.getAddress().getAddressName() + " " + addressPropertyEvent.getAddressProperty().value());
					break;
					
				default:
					logger.log(Level.INFO, "Address Property: " + addressPropertyEvent.getAddressProperty().value() + " is ingored");
					return;
			}
			getRefClient().updateAddressProperty(addressPropertyEventTO);
		}

		private void processTerminalPropertyEvent(final TerminalPropertyEventType terminalPropertyEvent)
		{
			logger.log(Level.INFO, "Terminal Property Event: TerminalName [" + terminalPropertyEvent.getTerminal().getTerminalName()
					+ "]\n TerminalProperty [" + terminalPropertyEvent.getTerminalProperty() + "]");
			TerminalPropertyEventTO terminalPropertyEventTO = new TerminalPropertyEventTO();

			terminalPropertyEventTO.setTerminalName(terminalPropertyEvent.getTerminal().getTerminalName());

			TerminalCapabilities cctCaps = terminalPropertyEvent.getTerminalCapabilities();
			TerminalCapabilitiesTO caps = new TerminalCapabilitiesTO();
			caps.setCanConference(cctCaps.isCanConference());
			caps.setCanConsult(cctCaps.isCanConsult());
			caps.setCanDoNotDisturb(cctCaps.isCanDoNotDisturb());
			caps.setCanForward(cctCaps.isCanForward());
			caps.setCanOriginate(cctCaps.isCanOriginate());
			caps.setCanPickup(cctCaps.isCanPickup());
			caps.setCanPickupFromGroup(cctCaps.isCanPickupFromGroup());
			caps.setCanTransfer(cctCaps.isCanTransfer());

			terminalPropertyEventTO.setCapabilities(caps);

			switch (terminalPropertyEvent.getTerminalProperty())
			{
				case DO_NOT_DISTURB:
					terminalPropertyEventTO.setPropertyType(TerminalProperty.DO_NOT_DISTURB);
					TerminalPropertyDoNotDisturb doNotDisturb = terminalPropertyEvent.getTerminalPropertyDoNotDisturb();
					terminalPropertyEventTO.setActive(doNotDisturb.isDoNotDisturbSet());
					logger.log(Level.INFO, "Terminal Property: " + TerminalProperty.DO_NOT_DISTURB.toString() + " "
							+ terminalPropertyEvent.getTerminal().getTerminalName() + " " + terminalPropertyEvent.getTerminalProperty().value()
							+ ": " + doNotDisturb.isDoNotDisturbSet());
					break;

				case FORWARDING:
					terminalPropertyEventTO.setPropertyType(TerminalProperty.FORWARDING);
					TerminalPropertyForwardingArray forwardingArr = terminalPropertyEvent.getTerminalPropertyForwardingArray();
					if (forwardingArr != null)
					{
						List<TerminalPropertyForwarding> forwardingList = forwardingArr.getItem();
						if (forwardingList.size() > 0)
						{
							terminalPropertyEventTO.setActive(true);
						}
					}
					logger.log(Level.INFO, "Terminal Property: " + AddressProperty.FORWARDING.toString() + " "
							+ terminalPropertyEvent.getTerminal().getTerminalName() + " " + terminalPropertyEvent.getTerminalProperty().value()
							+ ": " + terminalPropertyEventTO.isActive());
					break;

				case READY_STATUS:
					terminalPropertyEventTO.setPropertyType(TerminalProperty.READY_STATUS);
					terminalPropertyEventTO.setActive(terminalPropertyEvent.getAgentTerminalPropertyReadyStatus().isStatus());
					terminalPropertyEventTO.setValue(terminalPropertyEvent.getAgentTerminalPropertyReadyStatus().getReasonCode());
					logger.log(Level.INFO, "Terminal Property: READY_STATUS " + terminalPropertyEvent.getTerminal().getTerminalName() + " "
							+ terminalPropertyEvent.getTerminalProperty().value() + ": " + terminalPropertyEventTO.isActive() + " Reason: "
							+ terminalPropertyEvent.getAgentTerminalPropertyReadyStatus().getReasonCode());
					break;

				case ACTIVITY_CODE:
					terminalPropertyEventTO.setPropertyType(TerminalProperty.ACTIVITY_CODE);
					terminalPropertyEventTO.setValue(terminalPropertyEvent.getAgentTerminalPropertyActivityCode().getActivityCode());
					break;

				case AGENT_LOGIN_SESSION:
					terminalPropertyEventTO.setPropertyType(TerminalProperty.AGENT_LOGIN_SESSION);
					terminalPropertyEventTO.setActive(!terminalPropertyEvent.getAgentTerminalPropertyAgentLoginSession().isLoginEnded());
					logger.log(Level.INFO, "Terminal Property: AGENT_LOGIN_SESSION " + terminalPropertyEvent.getTerminal().getTerminalName() + " "
							+ terminalPropertyEvent.getTerminalProperty().value() + ": " + terminalPropertyEventTO.isActive());
					break;

				case AGENT_LOGIN_SESSION_ENDED:
					terminalPropertyEventTO.setPropertyType(TerminalProperty.AGENT_LOGIN_SESSION_ENDED);
					terminalPropertyEventTO.setActive(terminalPropertyEvent.getAgentTerminalPropertyAgentLoginSessionEnded().isLoginEnded());
					logger.log(Level.INFO, "Terminal Property: AGENT_LOGIN_SESSION_ENDED " + terminalPropertyEvent.getTerminal().getTerminalName()
							+ " " + terminalPropertyEvent.getTerminalProperty().value() + ": " + terminalPropertyEventTO.isActive());
					break;
					
				case SPECIFIC_CAPABILITY:
					terminalPropertyEventTO.setPropertyType(TerminalProperty.SPECIFIC_CAPABILITY);
					break;

				default:
					logger.log(Level.INFO, "Terminal Property: " + terminalPropertyEvent.getTerminalProperty().value() + " is ignored");
					return;
			}
			getRefClient().updateTerminalProperty(terminalPropertyEventTO);
		}

		private void processAgentPropertyEvent(final AgentPropertyEventType agentPropertyEvent)
		{
			logger.log(Level.INFO, "Agent Property Event: AgentLoginID [" + agentPropertyEvent.getAgent().getAgentLoginId() + "]\n AgentName ["
					+ agentPropertyEvent.getAgent().getUserName() + "]\n AgentProperty [" + agentPropertyEvent.getAgentProperty() + "]");
			AgentPropertyEventTO agentPropertyEventTO = new AgentPropertyEventTO();

			agentPropertyEventTO.setAgentLoginId(agentPropertyEvent.getAgent().getAgentLoginId());

			AgentCapabilities cctCaps = agentPropertyEvent.getAgentCapabilities();
			AgentCapabilitiesTO caps = new AgentCapabilitiesTO();
			caps.setCanGetNotReadyReason(cctCaps.isCanGetNotReadyReason());
			caps.setCanGetReadyStatus(cctCaps.isCanGetReadyStatus());
			caps.setCanGetSupervisor(cctCaps.isCanGetSupervisor());
			caps.setCanLogin(cctCaps.isCanLogin());
			caps.setCanLogout(cctCaps.isCanLogout());
			caps.setCanRetrieve(cctCaps.isCanRetrieve());
			caps.setCanSetNotReadyReason(cctCaps.isCanSetNotReadyReason());
			caps.setCanSetReadyStatus(cctCaps.isCanSetReadyStatus());
			caps.setCanSetStaticVoiceTerminal(cctCaps.isCanSetStaticVoiceTerminal());

			agentPropertyEventTO.setCapabilities(caps);

			switch (agentPropertyEvent.getAgentProperty())
			{
				case READY_STATUS:
					agentPropertyEventTO.setPropertyType(AgentProperty.READY_STATUS);
					agentPropertyEventTO.setActive(agentPropertyEvent.getAgentPropertyReadyStatus().isStatus());
					agentPropertyEventTO.setReason(agentPropertyEvent.getAgentPropertyReadyStatus().getReasonCode());
					logger.log(Level.INFO, "Agent Property: " + AgentProperty.READY_STATUS.toString() + " "
							+ agentPropertyEvent.getAgent().getAgentLoginId() + " " + agentPropertyEvent.getAgentProperty().value() + ": "
							+ agentPropertyEventTO.isActive() + " Reason: " + agentPropertyEventTO.getReason());
					break;

				case LOGIN_STATUS:
					agentPropertyEventTO.setPropertyType(AgentProperty.LOGIN_STATUS);
					agentPropertyEventTO.setActive(agentPropertyEvent.getAgentPropertyLoginStatus().getStatus().equals(AgentLoginStatus.LOGGED_IN));
					logger.log(Level.INFO, "Agent Property: " + AgentProperty.LOGIN_STATUS.toString() + " "
							+ agentPropertyEvent.getAgent().getAgentLoginId() + " " + agentPropertyEvent.getAgentProperty().value() + ": "
							+ agentPropertyEventTO.isActive());
					break;

				default:
					logger.log(Level.INFO, "Agent Property: " + agentPropertyEvent.getAgentProperty().value() + " is ignored");
					return;
			}
			getRefClient().updateAgentProperty(agentPropertyEventTO);
		}

		private void processContactPropertyEvent(final ContactPropertyEventType contactPropertyEvent)
		{
			logger.log(Level.INFO, "Contact Property Event: contactID [" + contactPropertyEvent.getContact().getContactId()
					+ "]\n ExternalContactID [" + contactPropertyEvent.getContact().getExternalContactId() + "]\n ContactProperty ["
					+ contactPropertyEvent.getContactProperty() + "]");
			ContactPropertyEventTO contactPropertyEventTO = new ContactPropertyEventTO();

			contactPropertyEventTO.setContactId(contactPropertyEvent.getContact().getContactId());

			ContactCapabilities cctCaps = contactPropertyEvent.getContactCapabilities();
			ContactCapabilitiesTO caps = new ContactCapabilitiesTO();
			caps.setCanAddParty(cctCaps.isCanAddParty());
			caps.setCanBargeIn(cctCaps.isCanBargeIn());
			caps.setCanDrop(cctCaps.isCanDrop());
			caps.setCanGetAttachedData(cctCaps.isCanGetAttachedData());
			caps.setCanGetCallingTerminal(cctCaps.isCanGetCallingTerminal());
			caps.setCanGetIntrinsics(cctCaps.isCanGetIntrinsics());
			caps.setCanGetLastRedirectedAddress(cctCaps.isCanGetLastRedirectedAddress());
			caps.setCanGetOriginalDestination(cctCaps.isCanGetOriginalDestination());
			caps.setCanGetUUI(cctCaps.isCanGetUUI());
			caps.setCanObserve(cctCaps.isCanObserve());
			caps.setCanSetAttachedData(cctCaps.isCanSetAttachedData());
			caps.setCanSetIntrinsics(cctCaps.isCanSetIntrinsics());
			caps.setCanSetUUI(cctCaps.isCanSetUUI());
			caps.setCanWhisper(cctCaps.isCanWhisper());

			contactPropertyEventTO.setCapabilities(caps);

			switch (contactPropertyEvent.getContactProperty())
			{
				case ATTACHED_DATA:
					logger.log(Level.INFO, "ATTACHED DATA");
					contactPropertyEventTO.setPropertyType(ContactProperty.ATTACHED_DATA);
					AttachedDataTO attachedData = new AttachedDataTO();
					AttachedData cctAttachedData = contactPropertyEvent.getContactPropertyAttachedData().getAttachedData();

					if (cctAttachedData != null && cctAttachedData.getType() != null)
					{
						if (cctAttachedData.getType().value().equals("STRING"))
						{
							attachedData.setType(AttachedDataType.STRING);
							attachedData.setStringData(cctAttachedData.getStringData());
							logger.log(Level.INFO, "Contact Property - ATTACHED DATA: " + attachedData.getStringData());
						}
						else if (cctAttachedData.getType().value().equals("BINARY"))
						{
							attachedData.setType(AttachedDataType.BINARY);
							byte[] binaryData = new byte[cctAttachedData.getBinaryData().getItem().size()];
							int byteCount = 0;
							for (Byte b : cctAttachedData.getBinaryData().getItem())
							{
								binaryData[byteCount++] = b.byteValue();
							}
							attachedData.setBinaryData(binaryData);

							logger.log(Level.INFO, "Contact Property - ATTACHED DATA: " + attachedData.getBinaryData().length + " bytes");
						}
						else
						{
							attachedData.setType(AttachedDataType.KVP);
							KeyValuePairTO[] kvps = new KeyValuePairTO[cctAttachedData.getKeyValueData().getItem().size()];
							int counter = 0;
							for (KeyValuePair cctKvp : cctAttachedData.getKeyValueData().getItem())
							{
								KeyValuePairTO kvp = new KeyValuePairTO();
								kvp.setKey(cctKvp.getKey());
								kvp.setValue(cctKvp.getValue());
								kvps[counter] = kvp;
								counter++;
							}
							attachedData.setKeyValueData(kvps);
							logger.log(Level.INFO, "Contact Property - ATTACHED DATA: " + attachedData.getKeyValueData().length + " key-value pair");
						}
					}

					contactPropertyEventTO.setObject(attachedData);
					break;

				case UUI:
					contactPropertyEventTO.setPropertyType(ContactProperty.UUI);
					contactPropertyEventTO.setValue(contactPropertyEvent.getContactPropertyUui().getUui());
					logger.log(Level.INFO, "Contact Property: " + contactPropertyEvent.getContactProperty().value() + " "
							+ contactPropertyEvent.getContactPropertyUui().getUui());
					break;

				case INTRINSICS:
					contactPropertyEventTO.setPropertyType(ContactProperty.INTRINSICS);
					List<IntrinsicTO> intrinsics = new ArrayList<IntrinsicTO>();
					for (Intrinsic intr : contactPropertyEvent.getContactPropertyIntrinsicArray().getItem())
					{
						IntrinsicTO intrinsic = new IntrinsicTO();
						intrinsic.setKey(intr.getKey());
						intrinsic.setValue(intr.getValue());
						intrinsic.setImmutable(intr.isImmutable());
						intrinsics.add(intrinsic);
					}
					contactPropertyEventTO.setObject(intrinsics);
					logger.log(Level.INFO, "Contact Property: " + contactPropertyEvent.getContactProperty().value() + " " + intrinsics.size()
							+ " pieces of Intrinsics");
					break;

				case CALLED_ADDRESS:
					contactPropertyEventTO.setContactId(contactPropertyEvent.getContact().getContactId());
					contactPropertyEventTO.setPropertyType(ContactProperty.CALLED_ADDRESS);
					contactPropertyEventTO.setObject(contactPropertyEvent.getContactPropertyCalledAddress().getCalledAddress());
					logger.log(Level.INFO, "Contact Property: " + contactPropertyEvent.getContactProperty().value() + " CalledAddressName "
							+ contactPropertyEvent.getContactPropertyCalledAddress().getCalledAddress().getAddressName());
					break;

				default:
					logger.log(Level.INFO, "Contact Property: " + contactPropertyEvent.getContactProperty().value() + " is ignored");
					return;
			}
			getRefClient().updateContactProperty(contactPropertyEventTO);
		}

		private void processConnectionStateEvent(final ConnectionStateEventType connectionStateEvent)
		{
			logger.log(Level.INFO, "Connection State Event: AddressName [" + connectionStateEvent.getAddress().getAddressName() + "]\n "
					+ connectionStateEvent.getPreviousState() + " -> " + connectionStateEvent.getNewState() + "\n ContactID ["
					+ connectionStateEvent.getContact().getContactId() + "]\n ExternalContactID ["
					+ connectionStateEvent.getContact().getExternalContactId() + "]\n Reason [" + connectionStateEvent.getReason() + "]");

			if ("IDLE".equalsIgnoreCase(connectionStateEvent.getNewState().value()))
			{
				logger.log(Level.INFO, "Ignoring IDLE TCS Event");
			}
			else if ("ROUTING".equalsIgnoreCase(connectionStateEvent.getNewState().value()))
			{
				//ROUTING can be here only for multimedia conections. We shouldn't ignore it.
				//We need to call alert for current connection here as per multimedia connection logic.
				Map<String, Object> params = new HashMap<String, Object>();
				params.put("connectionId", connectionStateEvent.getConnection().getConnectionId());
				getRefClient().getRequestHandler().handle("AlertConnection", params);
			}
			else if ("FAILED".equalsIgnoreCase(connectionStateEvent.getNewState().value()))
			{
				logger.log(Level.INFO, "Ignoring FAILED TCS Event");
			}
			else
			{
				StateEventTO stateEventTO = new StateEventTO();

				stateEventTO.setRoutePoint(connectionStateEvent.getAddress().getAddressType().equals(AddressType.ROUTE_POINT));
				stateEventTO.setTreatment(connectionStateEvent.getAddress().getAddressType().equals(AddressType.TREATMENT));
				if (stateEventTO.isTreatment())
				{
					stateEventTO.setDefaultTreatment(connectionStateEvent.getSendingAddress().getAddressName());
				}
				stateEventTO.setAddress(connectionStateEvent.getAddress().getAddressName());
				stateEventTO.setRemoteState(connectionStateEvent.getNewState().value());
				stateEventTO.setPrevRemoteState(connectionStateEvent.getPreviousState().value());
				stateEventTO.setReason(connectionStateEvent.getReason() == null ? "" : connectionStateEvent.getReason().value());
				stateEventTO.setContactId(connectionStateEvent.getContact().getContactId());
				stateEventTO.setExternalContactId(connectionStateEvent.getContact().getExternalContactId());

				ConnectionTO connection = new ConnectionTO();
				connection.setConnectionId(connectionStateEvent.getConnection().getConnectionId());

				stateEventTO.setConnection(connection);

				ConnectionCapabilities cctCaps = connectionStateEvent.getConnectionCapabilities();
				ConnectionCapabilitiesTO caps = new ConnectionCapabilitiesTO();
				caps.setCanAccept(cctCaps.isCanAccept());
				caps.setCanDisconnect(cctCaps.isCanDisconnect());
				caps.setCanAlert(cctCaps.isCanAlert());
				caps.setCanPark(cctCaps.isCanPark());
				caps.setCanRedirect(cctCaps.isCanRedirect());
				caps.setCanReject(cctCaps.isCanReject());

				stateEventTO.getConnection().setCapabilities(caps);
				stateEventTO.setNotifiedAddress(connectionStateEvent.getSendingAddress().getAddressName());
				
				if(stateEventTO.getAddress().toString().equals(stateEventTO.getNotifiedAddress().toString())) {
					stateEventTO.setRemote(true);
				} else {
					stateEventTO.setRemote(false);
				}

				getRefClient().updateContactsInfo(stateEventTO);
				logger.log(Level.INFO, "updateContactsInfo: " + stateEventTO.toString());
				if (stateEventTO.isRoutePoint() || stateEventTO.isTreatment())
				{
					getRefClient().updateRoutePointContactsInfo(stateEventTO);
				}
			}
		}

		private void processTerminalConnectionStateEvent(final TerminalConnectionStateEventType terminalConnectionStateEvent)
		{
			logger.log(Level.INFO, "Terminal Connection State Event: TerminalName [" + terminalConnectionStateEvent.getTerminal().getTerminalName()
					+ "]\n " + terminalConnectionStateEvent.getPreviousState() + " -> " + terminalConnectionStateEvent.getNewState()
					+ "\n ContactID [" + terminalConnectionStateEvent.getContact().getContactId() + "]\n ExternalContactID ["
					+ terminalConnectionStateEvent.getContact().getExternalContactId() + "]\n Reason [" + terminalConnectionStateEvent.getReason()
					+ "]");

			if ("IDLE".equalsIgnoreCase(terminalConnectionStateEvent.getNewState().value()))
			{
				logger.log(Level.INFO, "Ignoring IDLE TCS Event");
			}
			else if ("ROUTING".equalsIgnoreCase(terminalConnectionStateEvent.getNewState().value()))
			{
				logger.log(Level.INFO, "Ignoring ROUTING TCS Event");
			}
			else
			{
				StateEventTO stateEventTO = new StateEventTO();

				stateEventTO.setTerminal(terminalConnectionStateEvent.getTerminal().getTerminalName());
				stateEventTO.setLocalState(terminalConnectionStateEvent.getNewState().value());
				stateEventTO.setPrevLocalState(terminalConnectionStateEvent.getPreviousState().value());
				stateEventTO.setReason(terminalConnectionStateEvent.getReason() == null ? "" : terminalConnectionStateEvent.getReason().value());
				stateEventTO.setContactId(terminalConnectionStateEvent.getContact().getContactId());
				stateEventTO.setExternalContactId(terminalConnectionStateEvent.getContact().getExternalContactId());

				TerminalConnectionTO termConn = new TerminalConnectionTO();
				termConn.setTerminalConnectionId(terminalConnectionStateEvent.getTerminalConnection().getTerminalConnectionId());

				stateEventTO.setTerminalConnection(termConn);

				TerminalConnectionCapabilities cctCaps = terminalConnectionStateEvent.getTerminalConnectionCapabilities();
				TerminalConnectionCapabilitiesTO caps = new TerminalConnectionCapabilitiesTO();
				caps.setCanAnswer(cctCaps.isCanAnswer());
				caps.setCanBlindTransfer(cctCaps.isCanBlindTransfer());
				caps.setCanCompleteConference(cctCaps.isCanCompleteConference());
				caps.setCanCompleteTransfer(cctCaps.isCanCompleteTransfer());
				caps.setCanConsult(cctCaps.isCanConsult());
				caps.setCanGenerateDTMF(cctCaps.isCanGenerateDTMF());
				caps.setCanHold(cctCaps.isCanHold());
				caps.setHeldReason(cctCaps.isCanHold() ? "" : stateEventTO.getReason());
				caps.setCanInitiateConference(cctCaps.isCanInitiateConference());
				caps.setCanInitiateTransfer(cctCaps.isCanInitiateTransfer());
				caps.setCanUnhold(cctCaps.isCanUnhold());

				stateEventTO.getTerminalConnection().setCapabilities(caps);
				stateEventTO.setRemote(true);

				getRefClient().updateContactsInfo(stateEventTO);
				logger.log(Level.INFO, "updateContactsInfo: " + stateEventTO.toString());
				if (stateEventTO.isRoutePoint() || stateEventTO.isTreatment())
				{
					getRefClient().updateRoutePointContactsInfo(stateEventTO);
				}
			}
		}

		private void processAddressStateEvent(final AddressStateEventType addressStateEvent)
		{
			logger.log(Level.INFO, "Address State Event: AddressName [" + addressStateEvent.getAddress().getAddressName() + "]\n State ["
					+ addressStateEvent.getState().value() + "]\n Reason [" + addressStateEvent.getReason() + "]");
			AddressStateEventTO addressStateEventTO = new AddressStateEventTO();

			addressStateEventTO.setRoutePoint(addressStateEvent.getAddress().getAddressType().equals(AddressType.ROUTE_POINT));
			addressStateEventTO.setAddressName(addressStateEvent.getAddress().getAddressName());
			addressStateEventTO.setAddressType(addressStateEvent.getAddress().getAddressType().value());
			addressStateEventTO.setProviderName(addressStateEvent.getAddress().getProvider().getProviderName());

			AddressCapabilities cctCaps = addressStateEvent.getAddressCapabilities();
			AddressCapabilitiesTO caps = new AddressCapabilitiesTO();
			if (cctCaps != null)
			{
				caps.setCanConference(cctCaps.isCanConference());
				caps.setCanConference(cctCaps.isCanConsult());
				caps.setCanDoNotDisturb(cctCaps.isCanDoNotDisturb());
				caps.setCanForward(cctCaps.isCanForward());
				caps.setCanGetMessageWaiting(cctCaps.isCanGetMessageWaiting());
				caps.setCanOriginate(cctCaps.isCanOriginate());
				caps.setCanSetMessageWaiting(cctCaps.isCanSetMessageWaiting());
				caps.setCanTransfer(cctCaps.isCanTransfer());
			}

			addressStateEventTO.setCapabilities(caps);

			switch (addressStateEvent.getState())
			{
				case IN_SERVICE:
					addressStateEventTO.setStateType(AddressState.IN_SERVICE);
					break;

				case OUT_OF_SERVICE:
					addressStateEventTO.setStateType(AddressState.OUT_OF_SERVICE);
					break;

				case INVALID:
					addressStateEventTO.setStateType(AddressState.INVALID);
					break;

				default:
					logger.log(Level.INFO, "Address State Event: " + addressStateEvent.getState().value() + " is ignored");
					return;
			}
			getRefClient().updateAddressState(addressStateEventTO);
		}

		private void processTerminalStateEvent(final TerminalStateEventType terminalStateEvent)
		{
			logger.log(Level.INFO, "Terminal State Event: TerminalName [" + terminalStateEvent.getTerminal().getTerminalName() + "]\n State ["
					+ terminalStateEvent.getState().value() + "]\n Reason [" + terminalStateEvent.getReason() + "]");
			TerminalStateEventTO terminalStateEventTO = new TerminalStateEventTO();

			terminalStateEventTO.setTerminalName(terminalStateEvent.getTerminal().getTerminalName());
			terminalStateEventTO.setTerminalType(terminalStateEvent.getTerminal().getTerminalType().value());
			terminalStateEventTO.setProviderName(terminalStateEvent.getTerminal().getProvider().getProviderName());

			TerminalCapabilities cctCaps = terminalStateEvent.getTerminalCapabilities();
			TerminalCapabilitiesTO caps = new TerminalCapabilitiesTO();
			if (cctCaps != null)
			{
				caps.setCanConference(cctCaps.isCanConference());
				caps.setCanConsult(cctCaps.isCanConsult());
				caps.setCanDoNotDisturb(cctCaps.isCanDoNotDisturb());
				caps.setCanForward(cctCaps.isCanForward());
				caps.setCanOriginate(cctCaps.isCanOriginate());
				caps.setCanPickup(cctCaps.isCanPickup());
				caps.setCanPickupFromGroup(cctCaps.isCanPickupFromGroup());
				caps.setCanTransfer(cctCaps.isCanTransfer());
			}

			terminalStateEventTO.setCapabilities(caps);

			switch (terminalStateEvent.getState())
			{
				case IN_SERVICE:
					terminalStateEventTO.setStateType(TerminalState.IN_SERVICE);
					break;

				case OUT_OF_SERVICE:
					terminalStateEventTO.setStateType(TerminalState.OUT_OF_SERVICE);
					break;

				case INVALID:
					terminalStateEventTO.setStateType(TerminalState.INVALID);
					break;

				default:
					logger.log(Level.INFO, "Terminal State Event: " + terminalStateEvent.getState().value() + " is ignored");
					return;
			}
			getRefClient().updateTerminalState(terminalStateEventTO);
		}

		private void processAgentStateEvent(final AgentStateEventType agentStateEvent)
		{
			logger.log(Level.INFO, "Agent State Event: AgentLoginID [" + agentStateEvent.getAgent().getAgentLoginId() + "]\n UserName ["
					+ agentStateEvent.getAgent().getUserName() + "]\n State [" + agentStateEvent.getState());
			AgentStateEventTO agentStateEventTO = new AgentStateEventTO();

			agentStateEventTO.setUserId(agentStateEvent.getAgent().getAgentLoginId());
			agentStateEventTO.setUserName(agentStateEvent.getAgent().getUserName());

			agentStateEventTO.setCapabilities(new AgentCapabilitiesTO());

			switch (agentStateEvent.getState())
			{
				case MAPPED:
					agentStateEventTO.setStateType(AgentState.MAPPED);
					break;

				case UNMAPPED:
					agentStateEventTO.setStateType(AgentState.UNMAPPED);
					break;

				default:
					logger.log(Level.INFO, "Agent State Event: " + agentStateEvent.getState().value() + " is ignored");
					return;
			}
			getRefClient().updateAgentState(agentStateEventTO);
		}

		private void processSystemEvent(final SystemEventType systemEvent) throws Exception
		{
			logger.log(Level.INFO, "System Event: " + systemEvent.getSystemEvent().value());

			switch (systemEvent.getSystemEvent())
			{
				case CONNECTION_FAILURE:
					logger.log(Level.INFO, "CONNECTION_FAILURE");
					break;

				case SUBSCRIPTION_TERMINATION:
					logger.log(Level.INFO, "SUBSCRIPTION_TERMINATION");
					break;

				case SUBSCRIPTION_TERMINATION_IMMINENT:
					logger.log(Level.INFO, "SUBSCRIPTION_TERMINATION_IMMINENT");

					Map<String, Object> params = new HashMap<String, Object>();
					params.put("subscriptionId", NotificationHandler.SUBSCRIPTION_ID);
					logger.log(Level.INFO, "Renewing subscription for id: " + NotificationHandler.SUBSCRIPTION_ID);
					getRefClient().getRequestHandler().handle("RenewSubscription", params);
					break;
				
				case SESSION_TERMINATION_IMMINENT:
					logger.log(Level.INFO, "SessionTerminationImminent event received, renewing session");
					getRefClient().getRequestHandler().handle("SwitchOverHandler", new HashMap<String, Object>());
					break;

				default:
					logger.log(Level.INFO, "System Event: " + systemEvent.getSystemEvent().value() + " is unrecognized");
					break;
			}
		}
	}
}
