package com.nortel.rc.dao;

import java.net.URL;
import java.util.ArrayList;
import java.util.List;

import javax.xml.ws.BindingProvider;

import org.apache.cxf.jaxws.JaxWsProxyFactoryBean;
import org.apache.log4j.Level;
import org.apache.log4j.Logger;

import com.nortel.rc.dto.CallResult;
import com.nortel.rc.dto.ForwardInstructionTO;
import com.nortel.rc.dto.TerminalTO;
import com.nortel.rc.exception.DAOException;
import com.nortel.soa.oi.cct.faults.GetCapabilitiesException;
import com.nortel.soa.oi.cct.faults.GetContactTypesException;
import com.nortel.soa.oi.cct.faults.GetForwardingException;
import com.nortel.soa.oi.cct.sessionservice.SessionService;
import com.nortel.soa.oi.cct.terminalservice.SOAOICCTTerminalService;
import com.nortel.soa.oi.cct.terminalservice.TerminalService;
import com.nortel.soa.oi.cct.types.Forwarding;
import com.nortel.soa.oi.cct.types.SsoToken;
import com.nortel.soa.oi.cct.types.Terminal;
import com.nortel.soa.oi.cct.types.TerminalCapabilities;
import com.nortel.soa.oi.cct.types.TerminalType;
import com.nortel.soa.oi.cct.types.terminalservice.GetContactTypesResponse;
import com.nortel.soa.oi.cct.types.terminalservice.GetDoNotDisturbResponse;
import com.nortel.soa.oi.cct.types.terminalservice.GetForwardingResponse;
import com.nortel.soa.oi.cct.types.terminalservice.GetTerminalCapabilitiesResponse;
import com.nortel.soa.oi.cct.types.terminalservice.IsForwardedResponse;
import com.nortel.soa.oi.cct.types.terminalservice.SetDoNotDisturbTerminalRequest;
import com.nortel.soa.oi.cct.types.terminalservice.SetForwardingTerminalRequest;
import com.nortel.soa.oi.cct.types.terminalservice.TerminalRequest;

public class TerminalServiceDAO extends ServiceDAO implements ITerminalServiceDAO
{	
	private static Logger logger = Logger.getLogger("rc.dao.TerminalServiceDAO");
	private String endPoint = null;
    
    public CallResult getCapabilities(TerminalTO term, String sso) throws DAOException
    {
    	String methodName = "getCapabilities():"; 
		TerminalRequest terminalRequest = new TerminalRequest();
        TerminalCapabilities capabilities = null;
	    SsoToken ssoToken = new SsoToken();
	    Terminal terminal = new Terminal();
	    
	    terminal.setTerminalName(term.getTerminalName());
	    terminal.setTerminalType(TerminalType.valueOf(term.getTerminalType()));
	    ssoToken.setToken(sso);
	    
	    terminalRequest.setTerminal(terminal);
	    terminalRequest.setSsoToken(ssoToken);

	    try {     
	        GetTerminalCapabilitiesResponse response = getPort().getCapabilities(terminalRequest);
	        
	        if (response == null)
	        	throw new GetCapabilitiesException("NULL response received");
	        
	        capabilities = response.getTerminalCapabilities();
	    } catch (Exception e) {
	    	logExecption(logger, methodName, e);
	        throw new DAOException(e);
	    }finally{
	    	logger.log(Level.INFO, methodName+" sso["+sso+"] endpoint : "+ endPoint);
	    }   
	    
	    com.nortel.rc.dto.TerminalCapabilitiesTO caps = new com.nortel.rc.dto.TerminalCapabilitiesTO();
	    caps.setCanConference(capabilities.isCanConference());
	    caps.setCanDoNotDisturb(capabilities.isCanDoNotDisturb());
	    caps.setCanOriginate(capabilities.isCanOriginate());
	    caps.setCanTransfer(capabilities.isCanTransfer());
	    caps.setCanForward(capabilities.isCanForward());
	    
	    CallResult result = new CallResult();
	    result.setObject(caps);
        return result;
    }
    
    public CallResult getDoNotDisturb(TerminalTO term, String sso) throws DAOException
    {    	
    	String methodName = "getDoNotDisturb():"; 
		TerminalRequest request = new TerminalRequest();
	    SsoToken ssoToken = new SsoToken();
	    Terminal terminal = new Terminal();
	    
	    terminal.setTerminalName(term.getTerminalName());
	    terminal.setTerminalType(TerminalType.valueOf(term.getTerminalType()));
	    ssoToken.setToken(sso);
	    
	    request.setTerminal(terminal);
	    request.setSsoToken(ssoToken);

	    GetDoNotDisturbResponse response;
	    
	    try {        	
	    	response = getPort().getDoNotDisturb(request);
	    } catch (Exception e) {
	    	logExecption(logger, methodName, e);
	        throw new DAOException(e);
	    }finally{
	    	logger.log(Level.INFO, methodName+" sso["+sso+"] endpoint : "+ endPoint);
	    }
	    
	    CallResult result = new CallResult();
	    result.setObject(response != null ? response.isDoNotDisturb() : false);
        return result;
    }
    
    public CallResult setDoNotDisturb(TerminalTO term, String sso) throws DAOException
    {    	
    	String methodName = "setDoNotDisturb():"; 
		SetDoNotDisturbTerminalRequest request = new SetDoNotDisturbTerminalRequest();
	    SsoToken ssoToken = new SsoToken();
	    Terminal terminal = new Terminal();
	    
	    terminal.setTerminalName(term.getTerminalName());
	    terminal.setTerminalType(TerminalType.valueOf(term.getTerminalType()));
	    ssoToken.setToken(sso);
	    
	    request.setTerminal(terminal);
	    request.setSsoToken(ssoToken);
	    request.setDoNotDisturb(term.isDoNotDisturb());

	    try {        	
	        getPort().setDoNotDisturb(request);
	    } catch (Exception e) {
	    	logExecption(logger, methodName, e);
	        throw new DAOException(e);
	    }finally{
	    	logger.log(Level.INFO, methodName+" sso["+sso+"] endpoint : "+ endPoint);
	    }
	    
	    CallResult result = new CallResult();
        return result;
    }
    
    public CallResult setForwarding(TerminalTO term, List<ForwardInstructionTO> instructions, String sso) throws DAOException
    {
    	String methodName = "setForwarding():"; 
		SetForwardingTerminalRequest request = new SetForwardingTerminalRequest();
	    SsoToken ssoToken = new SsoToken();
	    Terminal terminal = new Terminal();
	    
	    terminal.setTerminalName(term.getTerminalName());
	    terminal.setTerminalType(TerminalType.valueOf(term.getTerminalType()));
	    ssoToken.setToken(sso);
	    
	    request.setTerminal(terminal);
	    request.setSsoToken(ssoToken);
	    
	    for(ForwardInstructionTO forward : instructions)
	    {
	    	Forwarding forwarding = new Forwarding();
	    	forwarding.setCaller(forward.getFrom());
	    	forwarding.setDestinationAddress(forward.getForwardTo());
	    	forwarding.setType(1);
	    	forwarding.setWhichCalls(1);
	    	request.getForwarding().add(forwarding);
	    }

	    try {        	
	        getPort().setForwarding(request);
	    } catch (Exception e) {
	    	logExecption(logger, methodName, e);
	        throw new DAOException(e);
	    }finally{
	    	logger.log(Level.INFO, methodName+" sso["+sso+"] endpoint : "+ endPoint);
	    }
	    
	    CallResult result = new CallResult();
        return result;
    }
    
    public CallResult getForwarding(TerminalTO term, String sso) throws DAOException
    {
    	String methodName = "getForwarding():"; 
		TerminalRequest request = new TerminalRequest();
        GetForwardingResponse response;
        List<ForwardInstructionTO> forwards = new ArrayList<ForwardInstructionTO>();
	    SsoToken ssoToken = new SsoToken();
	    Terminal terminal = new Terminal();
	    
	    terminal.setTerminalName(term.getTerminalName());
	    terminal.setTerminalType(TerminalType.valueOf(term.getTerminalType()));
	    ssoToken.setToken(sso);
	    
	    request.setTerminal(terminal);
	    request.setSsoToken(ssoToken);

	    try {        	
	    	response = getPort().getForwarding(request);
	    	
	        if (response == null)
	        	throw new GetForwardingException("NULL response received");
	    } catch (Exception e) {
	    	logExecption(logger, methodName, e);
	        throw new DAOException(e);
	    }finally{
	    	logger.log(Level.INFO, methodName+" sso["+sso+"] endpoint : "+ endPoint);
	    }
	    
	    for(Forwarding cctForward : response.getForwarding())
	    {
	    	ForwardInstructionTO forward = new ForwardInstructionTO();
	    	//forward.setCondition(cctForward.getWhichCalls());
	    	//forward.setFilter(cctForward.getType());
	    	forward.setForwardTo(cctForward.getDestinationAddress());
	    	forward.setFrom(cctForward.getCaller());
	    	forwards.add(forward);
	    }
	    
	    CallResult result = new CallResult();
	    result.setObject(forwards);
        return result;
    }
    
    public CallResult isForwarded(TerminalTO term, String sso) throws DAOException
    {
    	String methodName = "isForwarded():"; 
		TerminalRequest request = new TerminalRequest();
        IsForwardedResponse response;
        SsoToken ssoToken = new SsoToken();
	    Terminal terminal = new Terminal();
	    
	    terminal.setTerminalName(term.getTerminalName());
	    terminal.setTerminalType(TerminalType.valueOf(term.getTerminalType()));
	    ssoToken.setToken(sso);
	    
	    request.setTerminal(terminal);
	    request.setSsoToken(ssoToken);

	    try {        	
    	response = getPort().isForwarded(request);
	    } catch (Exception e) {
	    	logExecption(logger, methodName, e);
	        throw new DAOException(e);
	    }finally{
	    	logger.log(Level.INFO, methodName+" sso["+sso+"] endpoint : "+ endPoint);
	    }
	    
	    CallResult result = new CallResult();
	    result.setObject(response != null ? response.isIsForwarded() : false);
	    return result;	
    }
    
    public CallResult cancelForwarding(TerminalTO term, String sso) throws DAOException
    {
    	String methodName = "cancelForwarding():"; 
		TerminalRequest request = new TerminalRequest();
	    SsoToken ssoToken = new SsoToken();
	    Terminal terminal = new Terminal();
	    
	    terminal.setTerminalName(term.getTerminalName());
	    terminal.setTerminalType(TerminalType.valueOf(term.getTerminalType()));
	    ssoToken.setToken(sso);
	    
	    request.setTerminal(terminal);
	    request.setSsoToken(ssoToken);

	    try {        	
	        getPort().cancelForwarding(request);
	    } catch (Exception e) {
	    	logExecption(logger, methodName, e);
	        throw new DAOException(e);
	    }finally{
	    	logger.log(Level.INFO, methodName+" sso["+sso+"] endpoint : "+ endPoint);
	    }
	    
	    CallResult result = new CallResult();
        return result;
    }
    
    public CallResult getContactTypes(TerminalTO term, String sso) throws DAOException
    {
    	String methodName = "getContactTypes():"; 
		TerminalRequest terminalRequest = new TerminalRequest();
        List<String> contactTypes = null;
	    SsoToken ssoToken = new SsoToken();
	    Terminal terminal = new Terminal();
	    
	    terminal.setTerminalName(term.getTerminalName());
	    terminal.setTerminalType(TerminalType.valueOf(term.getTerminalType()));
	    ssoToken.setToken(sso);
	    
	    terminalRequest.setTerminal(terminal);
	    terminalRequest.setSsoToken(ssoToken);

	    try {        	
	        GetContactTypesResponse response = getPort().getContactTypes(terminalRequest);
	        
	        if (response == null)
	        	throw new GetContactTypesException("NULL response received");
	        
	        contactTypes = response.getContactTypes();
	    } catch (Exception e) {
	    	logExecption(logger, methodName, e);
	        throw new DAOException(e);
	    }finally{
	    	logger.log(Level.INFO, methodName+" sso["+sso+"] endpoint : "+ endPoint);
	    }
	    
	    CallResult result = new CallResult();
	    result.setObject(contactTypes);
        return result;
    }
    
    /**
	 * @return {@link SessionService}
	 * Retrieves interface to be used by service consumers
	 */
	public TerminalService getPort() throws DAOException {
		if(endPoint == null)
		{	      
			SOAOICCTTerminalService ss = new SOAOICCTTerminalService(getWsdlUrl(), getServiceQName());
			URL wsdlDoc = ss.getWSDLDocumentLocation();
			endPoint = wsdlDoc.getProtocol()+"://"+wsdlDoc.getAuthority()+wsdlDoc.getPath();
		}
		
		JaxWsProxyFactoryBean factory = new JaxWsProxyFactoryBean();
		factory.setServiceClass(TerminalService.class);
		TerminalService service = (TerminalService) factory.create();
		((BindingProvider)service).getRequestContext().put(BindingProvider.ENDPOINT_ADDRESS_PROPERTY, endPoint);

		return service;
	}
}
