﻿using System;
using System.Collections.Generic;
using System.Linq;
using System.Runtime.Serialization;
using System.ServiceModel;
using System.ServiceModel.Web;
using System.Text;
using System.Text.RegularExpressions;
using Translate;

namespace OpenInterfacesProvider
{
    /// <summary>
    /// Sample web service to be used to give example of how to configure a solution that will be compliant with Email Open Interfaces requirements.
    /// Four sample methods are provided. These contain shell implementations and are intended to be used purely for illustrative purposes.
    /// If nulls are returned then these will be disregarded by Email Open Interfaces and as such can be used in cases where exceptions are caught 
    /// or where the the value should be left unchanged in CCMM.
    /// </summary>
    [ServiceBehavior(Namespace = "http://emailopeninterfaces.com/")]
    public class EmailOpenInterfacesService : IEmailOpenInterfacesService
    {
        private const string CreditCardPattern = @"\b(?:\d[ -]*?){13,16}\b";

        #region Correct format methods only

        /// <summary>
        /// Input array:
        ///     [0]: From address
        ///     [1]: Subject
        /// Output array:
        ///     [0]: Subject
        ///     [1]: Priority
        /// </summary>
        /// <param name="inputParameters"></param>
        /// <returns></returns>
        public string[] PrioritiseBasedOnFromAddress(string[] inputParameters)
        {
            string[] outputParameters = new string[2];
            string prioritisedString = "{Prioritised: ";
            string priorityHigh = "1";
            string priorityLow = "5";

            try
            {
                if (inputParameters != null)
                {
                    if (inputParameters[0] != null) //from address not null
                    {
                        string subject;

                        if (inputParameters[1] != null)
                        {
                            subject = inputParameters[1];
                        }
                        else
                        {
                            subject = string.Empty;
                        }

                        if (inputParameters[0].ToLower().Contains("@avaya.com"))
                        {
                            prioritisedString += "High";
                            outputParameters[1] = priorityHigh;
                        }
                        else
                        {
                            prioritisedString += "Low";
                            outputParameters[1] = priorityLow;
                        }

                        outputParameters[0] = subject + " " + prioritisedString + "}";
                    }
                    else // if from address was null then perform no action
                    {
                        outputParameters[0] = null;
                        outputParameters[1] = null;
                    }
                }
                else
                {
                    outputParameters[0] = null;
                    outputParameters[1] = null;
                }
            }
            catch (Exception)
            {
                outputParameters[0] = null;
                outputParameters[1] = null;
            }

            return outputParameters;
        }

        /// <summary>
        /// Input array:
        ///     [0]: Plain text body of email
        ///     [1]: HTML Body of email
        /// Output array:
        ///     [0]: Plain text body of email
        ///     [1]: HTML Body of email
        /// </summary>
        /// <param name="inputParameters"></param>
        /// <returns></returns>
        public string[] DeleteCreditCardNumber(string[] inputParameters)
        {
            string[] outputParameters = new string[2];

            if (inputParameters != null && inputParameters[0] != null && inputParameters[1] != null)
            {
                try
                {
                    outputParameters[0] = DeleteCreditCardNumber(inputParameters[0]);
                    outputParameters[1] = DeleteCreditCardNumber(inputParameters[1]);
                }
                catch (Exception)
                {
                    outputParameters[0] = inputParameters[0];
                    outputParameters[1] = inputParameters[1];
                }
            }

            return outputParameters;
        }

        /// <summary>
        /// Input array:
        ///     [0]: Subject
        /// Output array:
        ///     [0]: Custom field name
        ///     [1]: Custom field value
        /// </summary>
        /// <param name="inputParameters"></param>
        /// <returns></returns>
        public string[] GetTrackingId(string[] inputParameters)
        {
            string[] outputParameters = new string[2];
            outputParameters[0] = null;
            outputParameters[1] = null;

            try
            {
                if (inputParameters != null)
                {
                    string inputParameter = inputParameters[0];
                    if (inputParameter != null)
                    {
                        if (inputParameters[0] != null && inputParameter.ToUpper().Contains("AVAYA")) // subject - looking for AVAYA followed by a number, e.g. AVAYA1234
                        {
                            outputParameters[0] = "Avaya_Tracking_ID";
                            outputParameters[1] = inputParameter.Substring(inputParameter.ToUpper().IndexOf("AVAYA") + 5);
                        }
                    }
                    //else if subject of email was null then perform no action
                }
            }
            catch (Exception) { }

            return outputParameters;
        }

        /// <summary>
        /// Input array:
        ///     [0]: Plain text body of email
        ///     [1]: HTML Body of email
        ///     [2]: Subject
        /// Output array:
        ///     [0]: Plain text body of email
        ///     [1]: HTML Body of email
        ///     [2]: Subject
        /// </summary>
        /// <param name="value"></param>
        /// <returns></returns>
        public string[] Translate(string[] inputParameters)
        {
            string[] outputParameters = new string[3];

            if (inputParameters != null && inputParameters[0] != null)
            {
                try
                {
                    if (inputParameters[1] == null)
                    {
                        inputParameters[1] = inputParameters[0];
                    }

                    string plainTextContent = inputParameters[0];
                    string htmlContent = inputParameters[1];

                    string subject = string.Empty;
                    
                    if (inputParameters[2] != null)
                    {
                        subject = inputParameters[2];
                    }

                    Translator translator = new Translator();
                    string translationText = translator.Translate(plainTextContent);

                    outputParameters[0] = plainTextContent + GetPlainTextToInsert(translationText); // Plain text

                    if (htmlContent.ToLower().Contains("</body>"))
                    {
                        int index = htmlContent.ToLower().IndexOf("</body>");
                        outputParameters[1] = htmlContent.Substring(0, index) + GetHtmlToInsert(translationText) + htmlContent.Substring(index);
                    }
                    else
                    {
                        outputParameters[1] = htmlContent + GetPlainTextToInsert(translationText); // Plain text
                    }

                    outputParameters[2] = subject + GetPlainTextToInsert(translator.Translate(subject));
                }
                catch
                {
                    outputParameters[0] = inputParameters[0];
                    outputParameters[1] = inputParameters[1];
                    outputParameters[2] = inputParameters[2];
                }
            }

            return outputParameters;
        }
        
        #endregion

        #region Private methods

        private string DeleteCreditCardNumber(string input)
        {
            string output = string.Empty;

            try
            {
                Match match = Regex.Match(input, CreditCardPattern, RegexOptions.IgnoreCase);

                if (match.Success)
                {
                    output = ScrubCreditCard(match, input);
                }
                else
                {
                    output = input;
                }
            }
            catch (Exception)
            {
                output = input;
            }


            return output;
        }

        private string ScrubCreditCard(Match match, string input)
        {
            string output = input;
            string uncleancardNumber = match.Groups.SyncRoot.ToString();

            output = Regex.Replace(input, CreditCardPattern, "++++++++++++++++++++");

            return output;
        }

        private string GetPlainTextToInsert(string translatedText)
        {
            string preTranslationText = " [Translated text: ";
            string postTranslationText = "]";

            if (translatedText == null)
            {
                translatedText = "Text could not be translated";
            }

            return preTranslationText + translatedText + postTranslationText;
        }

        private string GetHtmlToInsert(string translatedText)
        {
            return "<DIV><br><br><FONT color=#ff00ff size=2 face=Arial>" + GetPlainTextToInsert(translatedText) + "</FONT></DIV>";
        }

        #endregion
    }
}