﻿//////////////////////////////////////////////////////////////////////////////
// MainWindow.xaml.cs
//
// Copyright © 2008-2014 Avaya Inc. All rights reserved.
// See: www.avaya.com
//////////////////////////////////////////////////////////////////////////////
using EncoderDecoder;
using EncoderDecoder.Notifications;
using EncoderDecoder.Requests;
using Microsoft.Win32;
using Nortel.CCT;
using System;
using System.Collections.Generic;
using System.ComponentModel;
using System.Diagnostics;
using System.Windows;
using System.Windows.Controls;
using System.Windows.Media;
using System.Windows.Media.Animation;
using System.Windows.Threading;
using System.Xml;
using WebSocketSharp;


namespace CustomDesktop
{
    /// <summary>
    /// Interaction logic for MainWindow.xaml
    /// </summary>
    public partial class MainWindow : Window
    {
        private ICCTConnector cctConnector;

        // Create a logger for use in this class
        private static readonly log4net.ILog log = log4net.LogManager.GetLogger(System.Reflection.MethodBase.GetCurrentMethod().DeclaringType);
        private const string REG_KEY = "SOFTWARE\\Avaya\\SMF_Custom_Desktop";
        private const string USERNAME_KEY = "Username";
        private const string DOMAIN_KEY = "Domain";
        private ToolTip ttbtnToggleReady = new ToolTip();
        private ToolTip ttlblCCTConnectionState = new ToolTip();
        //boolean that tracks if the user has been notified of an smf disconnect. ensures the notifier is only shown once per disconnect.
        private bool userNotifiedOfSMFDisconnect = false;

        //WS Things
        public static Dictionary<long, UcChatTab> chatrooms = new Dictionary<long, UcChatTab>();
        public static Dictionary<long, long> consulting = new Dictionary<long, long>();
        public static Dictionary<long, UcChatTab> observing = new Dictionary<long, UcChatTab>();
        public static Dictionary<long, string> observingData = new Dictionary<long, string>();
        public static Dictionary<UcChatTab, TabControl> chatExtraTabsList = new Dictionary<UcChatTab, TabControl>();
        private Dictionary<long, AgentJoinRoomNotification> joinRoomDetails = new Dictionary<long, AgentJoinRoomNotification>();
        private Dictionary<long, string> MissedWelcomeMessage = new Dictionary<long, string>();
        

        /// <summary>
        /// Default constructor. reads config from the XML, sets necessary events, starts the listener thread.
        /// </summary>
        /// 
        public MainWindow()
        {
            log.Info("-----Messaging Application Logging Started-----");
            if (!ReadConfig())
            {
                MessageBox.Show(this, StringResources.ReadConfigErrorMessage);
            }
            InitializeComponent();

            // Remove support for SSL v3.0 and TLS v1.0
            System.Net.ServicePointManager.SecurityProtocol = System.Net.SecurityProtocolType.Tls11 | System.Net.SecurityProtocolType.Tls12;

            ToolTipService.SetToolTip(btnToggleReady, ttbtnToggleReady);
            ToolTipService.SetToolTip(lblCCTConnectionState, ttlblCCTConnectionState);
            ttlblCCTConnectionState.Content = StringResources.CCTDisconnected;

            this.Closing += new CancelEventHandler(CloseWindow);
            cctConnector = new CCTConnector();
            //Set CCT event handlers
            cctConnector.ContactReceived += AlertContact;
            cctConnector.ConsultCreated += AddConsultWindow;
            cctConnector.ConsultContactReceived += AlertReceivedConsultContact;
            cctConnector.ContactDropped += DropContact;
            cctConnector.ConferenceComplete += ConferenceCompleted;
            cctConnector.TransferComplete += TransferCompleted;
            cctConnector.ReadyStateChanged += SetReadyState;
            cctConnector.LoginStateChanged += SetLoginState;
            cctConnector.ConnectionStateChanged += CCTConnectionStateChanged;
            cctConnector.ConsultRejected += ConsultRejected;
            cctConnector.ConsultAccepted += ConsultAccepted;
            


            ucLoginControl.CCTConnection = cctConnector;
            ucLoginControl.MMConnectionStateChanged += MMConnectionStateChanged;
            Wsock.MMConnectionStateChanged += MMConnectionStateChanged;
            //Set login control event handler
            ucLoginControl.LoggingInEventHandler += UcCCTLoginLoggingInEventHandler;
            ucLoginControl.LogInCompletedEventHandler += UcCCTLoginLogInCompletedEventHandler;
            //Set Contact alert event handlers
            UcContactAlert.ContactAcceptEvent += AcceptContact;
            UcContactAlert.ContactRejectEvent += RejectContact;

            sidebar.BlindTransferRequested += BlindTransferCurrentContact;
            sidebar.AgentConsultRequested += ConsultAgentOnCurrentContact;
            sidebar.NewChatHistoryEvent += ChatHistory;
            sidebar.ObserveContactEvent += ObserveContact;
            sidebar.AddNewScreenPop += AddNewScreenPopHandler;
            sidebar.ForceChatExtraOpen += ForceChatExtraOpenHandler;

            //WS event handlers
            WsockEvent.AgentJoinRoomNotification += WsAgentJoinRoomHandler;
            WsockEvent.NewMessageNotification += WsNewMessageHandler;
            WsockEvent.NewPagePushNotification += WsNewPagePushHandler;
            WsockEvent.NewWelcomeMessageNotification += WsNewWelcomeMessageHandler;
            WsockEvent.NewWhisperMessageNotification += WsNewWhisperMessageHandler;
            WsockEvent.NewComfortMessageNotification += WsNewComfortMessageHandler;
            WsockEvent.CustomerDisconnectNotification += WsCustomerDisconnectHandler;
            WsockEvent.CustomerConnectionStatusNotification += WsCustomerConnectionStatusHandler;
            WsockEvent.AgentQuitChatNotification += WsAgentQuitChatHandler;
            WsockEvent.AgentExitRoomNotification += WsAgentExitRoomHandler;
            WsockEvent.AgentChangeConferenceOwnerNotification += WsAgentChangeConferenceOwnerHandler;
            WsockEvent.IsTypingNotification += WsIsTypingHandler;
            WsockEvent.AgentActiveNotification += WsAgentActiveHandler;
            WsockEvent.SupervisorBargeNotification += WsSupervisorBargeHandler;
            WsockEvent.ScreenPopsNotification += WsScreenPopsHandler;

        }

        

        private void ChatHistory(CustomerHistory history)
        {
            AddNewHistoryWindow(history);
        }


        private void WsNewWelcomeMessageHandler(object s, NotificationEvent e)
        {
            var messageEvent = e.Notification as NewMessageNotification;
            if (messageEvent != null)
            {
                try
                {
                    long contactGuid = messageEvent.body.guid;

                    log.Info("Received new welcome message for contact - " + contactGuid);

                    if (chatrooms.ContainsKey(contactGuid))
                    {
                        UcChatTab correctTab = chatrooms[contactGuid];
                        //Write to ChatArea window
                        this.Dispatcher.Invoke((Action)(() =>
                        {
                            if (messageEvent.body.messageType.ToUpper().Equals("WELCOMEMESSAGE"))
                            {
                                log.Info("Writing new welcome message to chat window for contact - " + contactGuid);
                                correctTab.ChatWindow.AppendAsSystemInfo("WelcomeMessage", messageEvent.body.message);
                            }
                        }));
                    }
                    else
                    {
                        log.Info("Could not find chat tab for welcome message in map for contact - " + contactGuid);
                        if (messageEvent.body.messageType.ToUpper().Equals("WELCOMEMESSAGE"))
                        {
                            log.Info("Saving WelcomeMessage to populate later for contact - " + contactGuid);
                            if (MissedWelcomeMessage.ContainsKey(contactGuid))
                            {
                                MissedWelcomeMessage.Remove(contactGuid);
                            }
                            MissedWelcomeMessage.Add(contactGuid, messageEvent.body.message);
                        }
                    }
                }
                catch (Exception ex)
                {
                    log.Error("Exception new welcome message notification handler " + messageEvent.ToString(), ex);
                }

            }
        }

        private void DisplayWhisperObserve(long contactGuid, NewMessageNotification messageEvent)
        {
            if (chatrooms.ContainsKey(contactGuid))
            {
                UcChatTab correctTab = chatrooms[contactGuid];

                if (correctTab.ConsultWindow != null)
                {
                    this.Dispatcher.Invoke((Action)(() =>
                    {
                        correctTab.ConsultWindow.AppendAsOtherWhisperAgent(messageEvent.body.timestamp, messageEvent.body.displayName, messageEvent.body.message);
                    }));
                }
                else
                {
                    this.Dispatcher.Invoke((Action)(() =>
                    {
                        correctTab.ChatWindow.AppendAsOtherWhisperAgent(messageEvent.body.timestamp, messageEvent.body.displayName, messageEvent.body.message);
                    }));
                }

            }
        }

        private void DisplayWhisperBarge(long contactGuid, NewMessageNotification messageEvent)
        {
            UcChatTab correctTab;
            correctTab = chatrooms[contactGuid];

            if (correctTab.ChatWindow != null && correctTab.ChatWindow.Type == UcChatWindow.WindowType.CONSULT_RECEIVER)
            {
                this.Dispatcher.Invoke((Action)(() =>
                {
                    correctTab.ChatWindow.AppendAsOtherWhisperAgent(messageEvent.body.timestamp, messageEvent.body.displayName, messageEvent.body.message);
                }));
            }
        }

        private void WsNewWhisperMessageHandler(object s, NotificationEvent e)
        {
            var messageEvent = e.Notification as NewMessageNotification;
            if (messageEvent != null)
            {
                try
                {
                    long contactGuid = messageEvent.body.guid;

                    log.Info("Received new whisper message for contact - " + contactGuid);

                    if (observing.ContainsKey(contactGuid))
                    {
                        log.Info("Found observe chatroom");

                        DisplayWhisperObserve(contactGuid, messageEvent);
                        return;
                    }
                    //Handle supervisor barge
                    if (chatrooms.ContainsKey(contactGuid))
                    {
                        DisplayWhisperBarge(contactGuid, messageEvent);
                    }
                    else
                    {
                        log.Error("Could not find chat tab for whisper message in consult map for contact - " + contactGuid);
                    }
                    if (consulting.ContainsKey(contactGuid))
                    {
                        UcChatTab correctTab;
                        long consultId = consulting[contactGuid];

                        if (chatrooms.ContainsKey(consultId))
                        {
                            correctTab = chatrooms[consulting[contactGuid]];
                        }
                        else
                        {
                            correctTab = chatrooms[contactGuid];
                        }

                        //Write to ChatArea window
                        this.Dispatcher.Invoke((Action)(() =>
                        {
                            if (messageEvent.body.messageType.ToUpper().Equals("WHISPER"))
                            {
                                
                                if (correctTab.Contact.IsConsultContact())
                                {
                                    log.Info("Writing new whisper message to chat window [consult receiver] for contact - " + contactGuid);
                                    correctTab.ChatWindow.AppendAsOtherWhisperAgent(messageEvent.body.timestamp, messageEvent.body.displayName, messageEvent.body.message);
                                }
                                else
                                {
                                    log.Info("Writing new whisper message to chat window [consult orig / conf] for contact - " + contactGuid);
                                    correctTab.ConsultWindow.AppendAsOtherWhisperAgent(messageEvent.body.timestamp, messageEvent.body.displayName, messageEvent.body.message);
                                }
                            }
                        }));
                    }

                }
                catch (Exception ex)
                {
                    log.Error("Exception in new whisper message notification handler " + messageEvent.ToString(), ex);
                }

            }
        }
        private void WsNewComfortMessageHandler(object s, NotificationEvent e)
        {
            var messageEvent = e.Notification as NewMessageNotification;
            if (messageEvent != null)
            {
                try
                {
                    long contactGuid = messageEvent.body.guid;

                    log.Info("Received new comfort message for contact - " + messageEvent.body.guid);

                    if (chatrooms.ContainsKey(contactGuid))
                    {

                        UcChatTab correctTab = chatrooms[contactGuid];
                        //Write to ChatArea window
                        this.Dispatcher.Invoke((Action)(() =>
                        {
                            if (messageEvent.body.messageType.ToUpper().Equals("COMFORTMESSAGE"))
                            {
                                log.Info("Writing new comfort message to chat window for contact - " + messageEvent.body.guid);
                                correctTab.ChatWindow.AppendAsSystemInfo("ComfortMessage", messageEvent.body.message);
                            }
                        }));
                    }
                    else
                    {
                        log.Error("Could not find chat tab for comfort message in map for contact - " + contactGuid);
                    }
                }
                catch (Exception ex)
                {
                    log.Error("Exception in comfort message notification handler " + messageEvent.ToString(), ex);
                }

            }
        }

        private void WsNewMessageHandler(object s, NotificationEvent e)
        {
            var messageEvent = e.Notification as NewMessageNotification;
            if (messageEvent != null)
            {
                try
                {
                    long contactGuid = messageEvent.body.guid;
                    UcChatTab correctTab = GetCorrectTab(contactGuid);

                    log.Info("Received new chat message for contact - " + messageEvent.body.guid);

                    if (correctTab != null)
                    {
                        this.Dispatcher.Invoke((Action)(() =>
                        {

                            if (messageEvent.body.sender.type.ToUpper().Equals("CUSTOMER"))
                            {
                                log.Info("Writing new customer chat to chat window for contact - " + messageEvent.body.guid);
                                correctTab.ChatWindow.AppendAsCustomer(messageEvent.body.timestamp, messageEvent.body.displayName, messageEvent.body.message);
                            } 
                            else
                            {
                                log.Info("Writing new agent chat to chat window for contact - " + messageEvent.body.guid);
                                correctTab.ChatWindow.AppendAsAgent(messageEvent.body.timestamp, messageEvent.body.displayName, messageEvent.body.message);
                            }

                            CheckIfTabNotificationIsRequired(correctTab);
                            
                        }));
                    } 
                    else
                    {
                        log.Error("Could not find chat tab for new message in map for contact - " + messageEvent.body.guid);
                    }

                    
                }
                catch (Exception ex)
                {
                    log.Error("Exception in new message notification handler " + messageEvent.ToString(), ex);
                }

            }

        }

        private void WsNewPagePushHandler(object s, NotificationEvent e)
        {
            var messageEvent = e.Notification as NewPagePushNotification;
            if (messageEvent != null)
            {
                try
                {
                    long contactGuid = messageEvent.body.guid;
                    UcChatTab correctTab = GetCorrectTab(contactGuid);

                    log.Info("Received new page push message for contact - " + messageEvent.body.guid);

                    if (correctTab != null)
                    {
                        this.Dispatcher.Invoke((Action)(() =>
                        {
                            if (messageEvent.body.sender.type.ToUpper().Equals("CUSTOMER"))
                            {
                                correctTab.ChatWindow.AppendAsCustomerURL(messageEvent.body.timestamp, messageEvent.body.displayName, messageEvent.body.url);
                            }
                            else
                            {
                                correctTab.ChatWindow.AppendAsAgentURL(messageEvent.body.timestamp, messageEvent.body.displayName, messageEvent.body.url);
                            }
                            
                        }));
                    }
                    else
                    {
                        log.Error("Could not find chat tab for new page push in map for contact - " + messageEvent.body.guid);
                    }
                }
                catch (Exception ex)
                {
                    log.Error("Exception in new page push notification handler " + messageEvent.ToString(), ex);
                }

            }

        }
        
        private void WsAgentJoinRoomHandler(object s, NotificationEvent e)
        {

            var messageEvent = e.Notification as AgentJoinRoomNotification;
            if (messageEvent != null)
            {
                try
                {
                    log.Info("Received JoinRoom notification for contact - " + messageEvent.body.guid);
                    joinRoomDetails.Add(messageEvent.body.guid, messageEvent);
                }
                catch (Exception ex)
                {
                    log.Error("Exception in agent join room handler " + messageEvent.ToString(), ex);
                }
            }
        }

        private void WsCustomerDisconnectHandler(object s, NotificationEvent e)
        {

            var messageEvent = e.Notification as CustomerDisconnectNotification;
            if (messageEvent != null)
            {
                try
                {
                    long contactGuid = messageEvent.body.guid;

                    log.Info("Received customer disconnect notification for contact - " + contactGuid);

                    if (consulting.ContainsKey(contactGuid) && !chatrooms.ContainsKey(contactGuid))
                    {
                        log.Info("Customer disconnect for consult destination agent for contact - " + contactGuid);
                        ContactLeavingChat(messageEvent, false);
                    }
                    //Consult originator
                    else if (chatrooms.ContainsKey(contactGuid))
                    {
                        if (chatrooms[contactGuid].ChatWindow.Type == UcChatWindow.WindowType.CONFERENCE &&
                            !chatrooms[contactGuid].ChatWindow.ConferenceOwner)
                        {
                            log.Info("Customer disconnect for passive conference agent - " + contactGuid);
                            ContactLeavingChat(messageEvent, false);
                            return;
                        }

                        log.Info("Customer disconnect for active conference agent / consult originating agent / 1-1 chat - " + contactGuid);
                        ContactLeavingChat(messageEvent, true);
                    }
                    else
                    {
                        log.Error("Could not find chat tab in map " + messageEvent.ToString());
                    }
                }
                catch (Exception ex)
                {
                    log.Error("Exception in Customer Disconnect handler " + messageEvent.ToString(), ex);
                }
            }
        }

        private void WsCustomerConnectionStatusHandler(object s, NotificationEvent e)
        {

            var messageEvent = e.Notification as CustomerConnectionStatusNotification;
            if (messageEvent != null)
            {
                try
                {
                    long contactGuid = messageEvent.body.guid;

                    log.Info("Received customer Connection Status notification for contact - " + contactGuid);

                    if (consulting.ContainsKey(contactGuid) && !chatrooms.ContainsKey(contactGuid))
                    {
                        UpdateCustStatus(contactGuid, messageEvent.body.isConnected);
                    }
                    else if (chatrooms.ContainsKey(contactGuid))
                    {
                        UpdateCustStatus(contactGuid, messageEvent.body.isConnected);
                    }
                    else
                    {
                        log.Error("Could not find chat tab in map " + messageEvent.ToString());
                    }
                }
                catch (Exception ex)
                {
                    log.Error("Exception in Customer Connection Status handler " + messageEvent.ToString(), ex);
                }
            }
        }

        private void UpdateCustStatus(long guid, bool isConnected)
        {
            UcChatTab correctTab = chatrooms[guid];
            if (correctTab.ChatWindow != null)
            {
                log.Info("Found correct tab for customer connection status change");

                this.Dispatcher.Invoke((Action)(() =>
                {
                    if (isConnected)
                    {
                        log.Info("Setting customer connection status to true");
                        correctTab.ChatWindow.UpdateIconColourStatus("Green");
                        correctTab.ChatWindow.AppendAsSystemInfo("Customer Disconnect", "Customers connection has been restored");
                        if (correctTab.ChatWindow.Type == UcChatWindow.WindowType.OBSERVE || correctTab.ChatWindow.IsBarge){
                            log.Info("Not setting send functionality back to true as window type is observe or barged in");
                        }
                        else
                        {
                            correctTab.ChatWindow.tbComment.IsEnabled = true;
                            correctTab.ChatWindow.btnSend.IsEnabled = true;

                        }
                        
                    }
                    else
                    {
                        log.Info("Setting customer connection status to false");
                        correctTab.ChatWindow.UpdateIconColourStatus("Red");
                        correctTab.ChatWindow.tbComment.IsEnabled = false;
                        correctTab.ChatWindow.btnSend.IsEnabled = false;
                        correctTab.ChatWindow.AppendAsSystemInfo("Customer Disconnect", "Customer has temporarily lost connection");
                    }
                }));
            }
        }

        private void WsAgentQuitChatHandler(object s, NotificationEvent e)
        {

            var messageEvent = e.Notification as AgentQuitChatNotification;
            if (messageEvent != null)
            {
                try
                {
                    long contactGuid = messageEvent.body.guid;

                    log.Info("Received agent quit chat notification for contact - " + contactGuid);

                    if (chatrooms.ContainsKey(contactGuid))
                    {
                        AgentLeavingChat(messageEvent);
                    }
                    else
                    {
                        log.Error("Could not find chat tab in map " + messageEvent.ToString());
                    }
                }
                catch (Exception ex)
                {
                    log.Error("Could not decode new join room notification " + messageEvent.ToString(), ex);
                }
            }
        }


        private void WsAgentExitRoomHandler(object s, NotificationEvent e)
        {

            var messageEvent = e.Notification as AgentExitRoomNotification;
            if (messageEvent != null)
            {
                log.Info("Recieved new AgentExit notification for contact - " + messageEvent.body.guid);
                try
                {
                    AgentExitRoom(messageEvent);
                }
                catch (Exception ex)
                {
                    log.Error("Exception in wsAgentExitRoomHandler " + messageEvent.ToString(), ex);
                }
            }
        }

        private void WsAgentChangeConferenceOwnerHandler(object s, NotificationEvent e)
        {

            var messageEvent = e.Notification as AgentChangeConferenceOwnerNotification;
            if (messageEvent != null)
            {
                try
                {
                    long contactGuid = messageEvent.body.guid;
                    UcChatTab correctTab = chatrooms[contactGuid];

                    log.Info("Recieved new change conference owner notification for contact - " + contactGuid);

                    //Handle promotes whilst observing a conference
                    if (correctTab.ChatWindow != null && correctTab.ChatWindow.Type == UcChatWindow.WindowType.OBSERVE)
                    {
                        this.Dispatcher.Invoke((Action)(() =>
                        {
                            correctTab.ChatWindow.AppendAsSystemInfo("ConferenceOwner", String.Format("{0} is the new conference owner", messageEvent.body.newOwner));
                        }));
                        return;
                    }
                    if (messageEvent.body.newOwner.Equals(Globals.agentID.ToString()))
                    {
                        this.Dispatcher.Invoke((Action)(() =>
                        {
                            log.Info("Setting as Active conference owner [" + Globals.agentID.ToString() + "] on contact - " + contactGuid);
                            correctTab.SetAsConferenceOwner(true);
                        }));
                        
                    }
                    else
                    {
                        this.Dispatcher.Invoke((Action)(() =>
                        {
                            log.Info("Setting as Passive conference owner [" + Globals.agentID.ToString() + "] on contact - " + contactGuid);
                            correctTab.SetAsConferenceOwner(false);
                        }));
                    }
                                        
                }
                catch (Exception ex)
                {
                    log.Error("Exception in wsAgentChangeConferenceOwnerHandler " + messageEvent.ToString(), ex);
                }
            }
        }

        private void WsIsTypingHandler(object s, NotificationEvent e)
        {

            var messageEvent = e.Notification as IsTypingNotification;
            if (messageEvent != null)
            {
                UcChatTab correctTab = null;
                long contactGuid = messageEvent.body.guid;
                bool isConsultReceiver = false;

                log.Info("Recieved IsTyping notification - " + contactGuid);
                try
                {
                    // Get correctTab for consult recevier
                    if (consulting.ContainsKey(contactGuid) && !chatrooms.ContainsKey(contactGuid))
                    {
                        correctTab = chatrooms[consulting[contactGuid]];
                        isConsultReceiver = true;
                    }
                    // Get correctTab for other types
                    else if (chatrooms.ContainsKey(contactGuid))
                    {
                        correctTab = chatrooms[contactGuid];

                        //Handle supervisor barge
                        if (correctTab.ChatWindow != null &&
                            correctTab.ChatWindow.Type.Equals(UcChatWindow.WindowType.CONSULT_RECEIVER))
                        {
                            isConsultReceiver = true;
                        }
                    }                  

                    if (correctTab != null)
                    {

                        DisplayIsTyping(isConsultReceiver, correctTab, messageEvent);
                    }

                    isConsultReceiver = false;

                }
                catch (Exception ex)
                {
                    log.Error("Exception in wsIsTypingHandler " + messageEvent.ToString(), ex);
                }
            }
        }

        private void DisplayIsTyping(bool isConsultReceiver, UcChatTab correctTab, IsTypingNotification messageEvent)
        {
            // Update chat window
            this.Dispatcher.Invoke((Action)(() =>
            {
                // Handle consult destination receiving normals & whispers
                if (isConsultReceiver)
                {
                    correctTab.ChatWindow.DisplayIsTyping(messageEvent);
                    return;
                }

                if (correctTab.ChatWindow != null && messageEvent.body.messageType.ToUpper().Equals("NORMAL"))
                {
                    correctTab.ChatWindow.DisplayIsTyping(messageEvent);
                }
                if (correctTab.ConsultWindow != null && messageEvent.body.messageType.ToUpper().Equals("WHISPER"))
                {
                    correctTab.ConsultWindow.DisplayIsTyping(messageEvent);
                }

            }));
        }
        private void WsAgentActiveHandler(object s, NotificationEvent e)
        {

            var messageEvent = e.Notification as AgentActiveNotification;
            if (messageEvent != null)
            {
                try
                {
                    log.Info(String.Format("Recieved AgentActive notification - AgentID:{0} GUID:{1} ",messageEvent.body.agentId,messageEvent.body.guid));
                    if (messageEvent.body.agentId != Globals.agentID.ToString())
                    {
                        if (messageEvent.body.userType.ToUpper().Equals("SUPERVISOR_OBSERVE"))
                        {
                            observingData.Add(messageEvent.body.guid, messageEvent.body.agentId);

                            UcChatTab correctTab = GetCorrectTab(messageEvent.body.guid);
                            this.Dispatcher.Invoke((Action)(() =>
                            {
                                correctTab.ChatWindow.AppendAsSystemInfo("Supervisor Observe", String.Format("Supervisor {0} is observing this chat", messageEvent.body.agentId));
                                sidebar.AllowRequestToConsultAnAgent(false, messageEvent.body.guid, messageEvent.body.agentId);
                            }));

                        }
                        if (messageEvent.body.userType.ToUpper().Equals("PASSIVE_PARTICIPANT"))
                        {
                            
                            UcChatTab correctTab = GetCorrectTab(messageEvent.body.guid);
                            if (correctTab.ChatWindow != null && correctTab.ChatWindow.Type == UcChatWindow.WindowType.OBSERVE)
                            {
                                correctTab.ChatWindow.remainingOtherAgentsInChat++;
                                this.Dispatcher.Invoke((Action)(() =>
                                {
                                    correctTab.ChatWindow.AppendAsSystemInfo("Consult", String.Format("Agent {0} has joined this chat", messageEvent.body.agentId));
                                }));
                            }
                            else
                            {
                                correctTab.ChatWindow.remainingOtherAgentsInChat++;
                            }

                        }
                        if (messageEvent.body.userType.ToUpper().Equals("SUPERVISOR_BARGE"))
                        {
                            UcChatTab correctTab = GetCorrectTab(messageEvent.body.guid);
                            try
                            {
                                this.Dispatcher.Invoke((Action)(() =>
                                {
                                    //Remove consult window if it exists
                                    if (correctTab.ConsultWindow != null)
                                    {
                                        correctTab.RemoveConsultWindow();
                                    }

                                    //Change to consult receiver window
                                    correctTab.ChatWindow.SetType(UcChatWindow.WindowType.CONSULT_RECEIVER);
                                    correctTab.ChatWindow.lblTitle.Content = "Superviser barge";
                                    correctTab.ChatWindow.IsBarge = true;
                                    //Disable sidebar consult button

                                    correctTab.ChatWindow.AppendAsSystemInfo("Supervisor Barge", String.Format("Supervisor {0} has barged into this chat", messageEvent.body.agentId));
                                }));

                            }
                            catch(Exception ex)
                            {
                                log.Error("exception in SUPERVISOR_BARGE",ex);
                            }

                        }
                    }
                    else
                    {
                        log.Info("Found myself in agent active notification");
                    }

                }
                catch (Exception ex)
                {
                    log.Error("Exception in wsAgentActiveHandler " + messageEvent.ToString(), ex);
                }
            }
        }

        private void WsSupervisorBargeHandler(object s, NotificationEvent e)
        {
            var messageEvent = e.Notification as SupervisorBargeNotification;
            if (messageEvent != null)
            {
                UcChatTab correctTab = GetCorrectTab(messageEvent.body.guid);
                try
                {
                    this.Dispatcher.Invoke((Action)(() =>
                    {
                        if (correctTab.ChatWindow.Type == UcChatWindow.WindowType.CONSULT_RECEIVER)
                        {
                            correctTab.ChatWindow.consultReceiverBeforeBarge = true;
                        }
                        //Remove consult window if it exists
                        if (correctTab.ConsultWindow != null)
                        {
                            correctTab.RemoveConsultWindow();
                        }

                        //Change to consult receiver window
                        correctTab.ChatWindow.SetType(UcChatWindow.WindowType.CONSULT_RECEIVER);
                        correctTab.ChatWindow.lblTitle.Content = "Superviser barge";
                        correctTab.ChatWindow.IsBarge = true;
                        correctTab.ChatWindow.ConferenceOwner = false;
                        //Disable sidebar consult button

                        correctTab.ChatWindow.AppendAsSystemInfo("Supervisor Barge", String.Format("Supervisor {0} has barged into this chat", messageEvent.body.supervisorId));
                    }));

                }
                catch (Exception ex)
                {
                    log.Error("exception in SUPERVISOR_BARGE", ex);
                }
            }
        }

        private void RequestScreenPops(long contactGuid)
        {
            ScreenPopRequest spr = new ScreenPopRequest();
            spr.apiVersion = BaseRequest.JSapiVersion.one_point_zero;
            spr.body.guid = contactGuid;
            Wsock.Send(spr);
        }

        private void WsScreenPopsHandler(object s, NotificationEvent e)
        {
            var messageEvent = e.Notification as ScreenPopsNotification;
            if (messageEvent != null)
            {
                try
                {
                    this.Dispatcher.Invoke((Action)(() =>
                    {
                        sidebar.AddScreenPops(messageEvent);
                    }));

                }
                catch (Exception ex)
                {
                    log.Error("exception in SUPERVISOR_BARGE", ex);
                }
            }
        }

        /// <summary>
        /// Updates for when a customer explicitly leaves the chat room
        /// </summary>
        /// <param name="notification"></param>
        private void ContactLeavingChat(CustomerDisconnectNotification notification, bool chatOriginater)
        {
            long contactGuid = notification.body.guid;
            UcChatTab correctTab = GetCorrectTab(contactGuid);
            
            if(correctTab != null)
            {
                correctTab.ChatWindow.customerLeftChat = true;

                // Update chat window
                this.Dispatcher.Invoke((Action)(() =>
                {
                    if (chatOriginater)
                    {
                        // Get the closed reasons from the notification
                        log.Info("Populating close reason codes from customer disconnect for contact - " + contactGuid);
                        sidebar.EnableConsult(false);
                        ClosedReasonCodes codes = CreateClosedReasonCodesList(notification);
                        correctTab.ChatWindow.PopulateClosedReasons(codes);
                    }

                    log.Info("Writing customer disconnect status message to chat window for contact - " + contactGuid);
                    if (correctTab.ChatWindow.IsBarge)
                    {
                        correctTab.ChatWindow.SupervisorLeftChat = true;
                    }
                    correctTab.ChatWindow.AppendAsSystemInfo("Customer Exit", "Customer has quit the chat");
                    correctTab.ChatWindow.DisableSendingNewMessages();
                    correctTab.ChatWindow.btnBarge.Visibility = System.Windows.Visibility.Hidden;
                    if (correctTab.ConsultWindow != null)
                    {
                        log.Info("Writing customer disconnect status message to consult window for contact - " + contactGuid);
                        correctTab.ConsultWindow.AppendAsSystemInfo("Customer Exit", "Customer has quit the chat");
                        correctTab.ConsultWindow.DisableSendingNewMessages();
                        //All other agents have left so allow supervisor to close the whisper window
                        correctTab.ConsultWindow.btnClose.Visibility = System.Windows.Visibility.Visible;
                        correctTab.ConsultWindow.btnClose.Click += correctTab.BtnCloseBargeWhisperClick;
                        correctTab.ConsultWindow.btnDrop.Visibility = System.Windows.Visibility.Hidden;
                    }
                    
                }));
            }
            

        }

        /// <summary>
        /// Updates for when a agent explicitly leaves the chat room
        /// </summary>
        /// <param name="notification"></param>
        private void AgentLeavingChat(AgentQuitChatNotification notification)
        {
            // Find the correct chat tab
            UcChatTab correctTab = chatrooms[notification.body.guid];

            // Get the closed reasons from the notification
            ClosedReasonCodes codes = CreateClosedReasonCodesList(notification);

            // Update chat window
            this.Dispatcher.Invoke((Action)(() =>
            {
                sidebar.EnableConsult(false);
                correctTab.ChatWindow.PopulateClosedReasons(codes);
                correctTab.ChatWindow.DisableSendingNewMessages();
                correctTab.ChatWindow.ShowClosedReasonWindow();
            }));

            //Chat is over remove from the map
            chatrooms.Remove(notification.body.guid);
        }

        /// <summary>
        /// Updates for when a agent explicitly leaves the chat room
        /// </summary>
        /// <param name="notification"></param>
        private void AgentExitRoom(AgentExitRoomNotification notification)
        {

            long chatGuid = notification.body.guid;

            log.Info("Received agent exit room for contact - " + chatGuid);

            //Check if supervisor was observing
            if (observingData.ContainsKey(chatGuid) && observingData[chatGuid] == notification.body.agentId)
            {
                log.Info("AgentExitRoom for scenario supervisor observe on contact - " + chatGuid);
                UcChatTab correctTab = chatrooms[chatGuid];
                //Write to ChatArea window
                log.Info("Writing agent exit status message to chat window on contact - " + chatGuid);

                if (correctTab.ConsultWindow != null)
                {
                    this.Dispatcher.Invoke((Action)(() =>
                    {
                        correctTab.ConsultWindow.AppendAsSystemInfo("Supervisor Exit", "Supervisor is no longer observing");  
                    }));
                }

                this.Dispatcher.Invoke((Action)(() =>
                {
                    correctTab.ChatWindow.AppendAsSystemInfo("Supervisor Exit", "Supervisor is no longer observing");
                    sidebar.AllowRequestToConsultAnAgent(true, chatGuid, notification.body.agentId);
                }));

                observingData.Remove(chatGuid);
                return;
            }
            //Receiving Consult
            if (consulting.ContainsKey(chatGuid) && !chatrooms.ContainsKey(chatGuid))
            {
                log.Info("AgentExitRoom for scenario ConsultReceiver on contact - " + chatGuid);
                UcChatTab correctTab = chatrooms[consulting[chatGuid]];
                //Write to ChatArea window
                this.Dispatcher.Invoke((Action)(() =>
                {
                    // Scenario - Passive with own ID - Dropped
                    // Scenario - Passive with own ID - Left
                    // Scenario - Passive with own ID - ConsultChat Over
                    log.Info("Writing agent exit status message to chat window on contact - " + chatGuid);
                    if (notification.body.agentId == Globals.agentID.ToString())
                    {
                        correctTab.ChatWindow.SupervisorLeftChat = true;
                        correctTab.ChatWindow.customerLeftChat = true;
                    }
                    correctTab.ChatWindow.agentLeftChat = true;
                    correctTab.ChatWindow.AppendAsSystemInfo("Agent Exit", "Agent has closed the consult");
                    correctTab.ChatWindow.DisableSendingNewMessages();
                    return;
                }));
            }
            //Consult originator
            else if (chatrooms.ContainsKey(chatGuid))
            {
                
                UcChatTab correctTab = chatrooms[chatGuid];
                
                //Conferencing
                if (correctTab.ChatWindow.Type == UcChatWindow.WindowType.CONFERENCE)
                {
                    this.Dispatcher.Invoke((Action)(() =>
                    {
                        //If passive has left
                        if (correctTab.ChatWindow.ConferenceOwner)
                        {
                            log.Info("AgentExitRoom for scenario ConfActive on contact - " + chatGuid);
                            correctTab.ConsultWindow.agentLeftChat = true;
                            correctTab.ConsultWindow.AppendAsSystemInfo("Agent Exit", "Passive has left the conference");
                            correctTab.ConsultWindow.DisableSendingNewMessages();
                            correctTab.ConsultWindow.btnDrop.Visibility = System.Windows.Visibility.Hidden;
                            correctTab.ConsultWindow.btnPromote.Visibility = System.Windows.Visibility.Hidden;
                            correctTab.ConsultWindow.btnClose.Visibility = System.Windows.Visibility.Visible;
                            correctTab.ConsultWindow.btnClose.Click += correctTab.BtnCloseConfWhisperClick;
                            correctTab.ChatWindow.SetType(UcChatWindow.WindowType.NORMAL);
                        }
                        //If active has kicked passive
                        else
                        {
                            log.Info("AgentExitRoom for scenario ConfPassive on contact - " + chatGuid);
                            correctTab.ConsultWindow.agentLeftChat = true;
                            correctTab.ConsultWindow.AppendAsSystemInfo("Agent Exit", "You have been kicked by the conference owner");
                            correctTab.ConsultWindow.DisableSendingNewMessages();
                            correctTab.ChatWindow.AppendAsSystemInfo("Agent Exit", "You have been kicked by the conference owner");
                            correctTab.ChatWindow.DisableSendingNewMessages();
                        }
                        return;
                    }));
                }
                else if (correctTab.ChatWindow.Type == UcChatWindow.WindowType.OBSERVE)
                {
                    
                    correctTab.ChatWindow.remainingOtherAgentsInChat--;
                    this.Dispatcher.Invoke((Action)(() =>
                    {
                        if (notification.body.agentId.Equals(Globals.agentID.ToString()))
                        {
                            correctTab.ChatWindow.agentLeftChat = true;
                            correctTab.ChatWindow.UpdateIconColourStatus("Red"); 
                            correctTab.ChatWindow.AppendAsSystemInfo("Agent Exit", "You have been kicked from the chat");
                            correctTab.ChatWindow.btnBarge.Visibility = System.Windows.Visibility.Hidden;
                            return;
                        }
                        else
                        {
                            correctTab.ChatWindow.AppendAsSystemInfo("Agent Exit", String.Format("Agent {0} has left the chat", notification.body.agentId));

                        }

                        if (correctTab.ChatWindow.remainingOtherAgentsInChat == 0)
                        {
                            correctTab.ChatWindow.AppendAsSystemInfo("Agent Exit", "All agents have left the chat");
                            correctTab.ChatWindow.agentLeftChat = true;
                            correctTab.ChatWindow.UpdateIconColourStatus("Red"); 
                        }
                    }));
                    return; 
                }
                else if (correctTab.ChatWindow.Type == UcChatWindow.WindowType.BARGE)
                {

                    correctTab.ChatWindow.remainingOtherAgentsInChat--;
                    this.Dispatcher.Invoke((Action)(() =>
                    {
                        if (notification.body.agentId.Equals(Globals.agentID.ToString()))
                        {
                            correctTab.ChatWindow.agentLeftChat = true;
                            correctTab.ChatWindow.UpdateIconColourStatus("Red");
                            correctTab.ChatWindow.AppendAsSystemInfo("Agent Exit", "You have been kicked from the chat");
                            return;

                        }
                        else
                        {
                            correctTab.ChatWindow.AppendAsSystemInfo("Agent Exit", String.Format("Agent {0} has left the chat", notification.body.agentId));
                            correctTab.ConsultWindow.AppendAsSystemInfo("Agent Exit", String.Format("Agent {0} has left the chat", notification.body.agentId));
                        }

                        if (correctTab.ChatWindow.remainingOtherAgentsInChat == 0)
                        {
                            if (correctTab.ConsultWindow != null)
                            {
                                //All other agents have left so allow supervisor to close the whisper window
                                correctTab.ConsultWindow.btnClose.Visibility = System.Windows.Visibility.Visible;
                                correctTab.ConsultWindow.btnClose.Click += correctTab.BtnCloseBargeWhisperClick;
                                correctTab.ConsultWindow.btnDrop.Visibility = System.Windows.Visibility.Hidden;
                                //Go back to a 1-1 chat
                                correctTab.ChatWindow.SetType(UcChatWindow.WindowType.NORMAL);
                            }
                        }
                    }));
                    return;
                }
                //Originating consult
                else
                {
                    log.Info("AgentExitRoom for scenario ConsultOriginator on contact - " + chatGuid);
                    this.Dispatcher.Invoke((Action)(() =>
                    {
                        if (correctTab.ConsultWindow != null && !correctTab.ChatWindow.IsBarge)
                        {
                            log.Info("Writing agent exit status message to chat window on contact - " + chatGuid);
                            correctTab.ConsultWindow.agentLeftChat = true;
                            correctTab.ConsultWindow.AppendAsSystemInfo("Agent Exit", "Agent has closed the consult");
                            correctTab.ConsultWindow.DisableSendingNewMessages();
                        }
                        else
                        {
                            log.Info("Writing agent exit status message to chat window on contact - " + chatGuid);
                            if (correctTab.ChatWindow.IsBarge && notification.body.agentId == Globals.agentID.ToString())
                            {
                                correctTab.ChatWindow.SupervisorLeftChat = true;
                                correctTab.ChatWindow.customerLeftChat = true;
                            }
                            else
                            {
                                correctTab.ChatWindow.agentLeftChat = true;
                            }
                            correctTab.ChatWindow.AppendAsSystemInfo("Agent Exit", "Agent has closed the consult");
                            correctTab.ChatWindow.DisableSendingNewMessages();
                        }
                        return;
                    }));
                }
            }
            else
            {
                log.Error("Could not find chat tab for new agent exit in map for contact - " + chatGuid);
            }

        }

        private UcChatTab GetCorrectTab(long contactGuid)
        {
            UcChatTab correctTab = null;

            //Destination Consult
            if (consulting.ContainsKey(contactGuid) && !chatrooms.ContainsKey(contactGuid))
            {
                correctTab = chatrooms[consulting[contactGuid]];
            }
            //Normal 1-1, Originating consult, Conference
            else if (chatrooms.ContainsKey(contactGuid))
            {
                correctTab = chatrooms[contactGuid];
            }

            return correctTab;
        }

        /// <summary>
        /// Create the closed reason codes list which is used to populate dropdown list
        /// </summary>
        /// <param name="notificationType"></param>
        /// <returns></returns>
        private ClosedReasonCodes CreateClosedReasonCodesList(IBaseNotification notificationType)
        {
            ClosedReasonCodes closingCodes = new ClosedReasonCodes();

            if (notificationType.GetType() == typeof(AgentQuitChatNotification))
            {
                AgentQuitChatNotification temp = notificationType as AgentQuitChatNotification;
                closingCodes.defaultReasonCode = temp.body.closedReasonCodes.defaultReasonCode;
                foreach (Code aCode in temp.body.closedReasonCodes.codes)
                {
                    Code code = new Code();
                    code.id = aCode.id;
                    code.closedReason = aCode.closedReason;
                    closingCodes.codes.Add(code);
                }
            }
            else if (notificationType.GetType() == typeof(CustomerDisconnectNotification))
            {
                CustomerDisconnectNotification temp = notificationType as CustomerDisconnectNotification;
                closingCodes.defaultReasonCode = temp.body.closedReasonCodes.defaultReasonCode;
                foreach (Code aCode in temp.body.closedReasonCodes.codes)
                {
                    Code code = new Code();
                    code.id = aCode.id;
                    code.closedReason = aCode.closedReason;
                    closingCodes.codes.Add(code);
                }
            }

            return closingCodes;

        }

        private void ConsultAccepted(Contact contact)
        {
            if (this.Dispatcher.CheckAccess())
            {
                Dispatcher.BeginInvoke(DispatcherPriority.Normal, (Action)(() =>
                {
                    //find the correct chat tab
                    foreach (UcChatTab tab in tabControl.Items)
                    {

                        if (tab.ConsultContact != null && tab.ConsultContact.ID == contact.ID)
                        {
                            tab.ConsultWindow.EnableConsult();
                            tab.ConsultWindow.UpdateIconColourStatus("Green");
                            tab.PendingConsult = false;
                            tabControl.SelectedItem = tab;                           
                            sidebar.EnableConsult(false);
                            return;
                        }
                    }
                }));
            }
            else
            {
                Action<Contact> a = new Action<Contact>(ConsultAccepted);
                this.Dispatcher.Invoke(a, contact);
            }
        }

        private void MMConnectionStateChanged(bool isConnected)
        {
            try
            {
                log.Info("Updating MMConnectionStateChanged - " + isConnected);

                Dispatcher.BeginInvoke(DispatcherPriority.Normal, (Action)(() =>
                {
                    if (!isConnected)
                    {
                        if (cctConnector.IsConnected())
                        {
                            // Force agent to go not ready so they are not alerted anymore contacts
                            btnToggleReady.IsEnabled = false;
                            if (cctConnector.GetAgentIsReadyState())
                            {
                                //Set agent to not ready
                                cctConnector.ToggleReady();
                                MessageBox.Show(this, "Forcing agent to a not ready state due to loss of connection to MM websocket", "Agent State Change");

                            }
                        }
                    }
                    else
                    {
                        // Reenable ready status button
                        btnToggleReady.IsEnabled = true;

                    }
                }));

                //Stop ability to send messages
                foreach (UcChatTab tab in tabControl.Items)
                {
                    log.Info("Found a chat tab");
                    if (tab.ChatWindow != null || tab.ConsultWindow != null)
                    {
                        Dispatcher.BeginInvoke(DispatcherPriority.Normal, (Action)(() =>
                        {
                            log.Info("Updating send button and chat box to enabled: " + isConnected);
                            if (isConnected)
                            {
                                EnableSendFunc(tab, true);
                            }
                            else
                            {
                                EnableSendFunc(tab, false);
                            }
                        }));
                    }
                }

                //Update GUI label with status
                SMFConnectionStateChanged(isConnected);
            }
            catch (Exception ex)
            {
                log.Error("Exception in MMConnectionStateChanged", ex);
            }
        }

        private void EnableSendFunc(UcChatTab tab, bool enable)
        {
            try {
                log.Info("Setting send functions to - " + enable);
                if (tab.ChatWindow != null)
                {
                    log.Info("Setting send functionality on chat window");
                    if (tab.ChatWindow.Type == UcChatWindow.WindowType.OBSERVE || tab.ChatWindow.IsBarge)
                    {
                        log.Info("Not setting send functionality back to true as window type is observe or barged in");
                    }
                    else
                    {
                        tab.ChatWindow.btnSend.IsEnabled = enable;
                        tab.ChatWindow.tbComment.IsEnabled = enable;
                    }
                }

                if (tab.ConsultWindow != null)
                {
                    log.Info("Setting send functionality on consult window");
                    tab.ConsultWindow.btnSend.IsEnabled = enable;
                    tab.ConsultWindow.tbComment.IsEnabled = enable;
                }
            }
            catch(Exception ex)
            {
                log.Error("Exception in EnableSendFunc", ex);
            }
        }

        private void SMFConnectionStateChanged(bool isConnected)
        {
            log.Info("Updating SMFConnectionStateChanged - " + isConnected);

            if (this.Dispatcher.CheckAccess())
            {
                Dispatcher.BeginInvoke(DispatcherPriority.Normal, (Action)(() =>
                {
                    if (isConnected)
                    {
                        log.Info("SMFConnectionStateChanged isConnected - " + isConnected);
                        lblSMFConnectionState.Background = Brushes.Green;
                        lblSMFConnectionState.Content = StringResources.SMFConnected;
                        userNotifiedOfSMFDisconnect = false;
                    }
                    else
                    {
                        log.Info("SMFConnectionStateChanged isConnected - " + isConnected);
                        lblSMFConnectionState.Background = Brushes.Red;
                        lblSMFConnectionState.Content = StringResources.SMFDisconnected;
                        if (!userNotifiedOfSMFDisconnect)
                        {
                            userNotifiedOfSMFDisconnect = true;
                        }
                    }
                }));
            }
            else
            {
                Action<Boolean> a = new Action<Boolean>(SMFConnectionStateChanged);
                this.Dispatcher.Invoke(a, isConnected);
            }
        }

        private void CCTConnectionStateChanged(CCTConnector.CCTConnectionState state)
        {
            log.Info("Updating CCTConnectionStateChanged - " + state);
            if (this.Dispatcher.CheckAccess())
            {
                Dispatcher.BeginInvoke(DispatcherPriority.Normal, (Action)(() =>
                {
                    switch (state)
                    {
                        case CCTConnector.CCTConnectionState.CONNECTED:
                            lblCCTConnectionState.Content = StringResources.CCTConnected;
                            lblCCTConnectionState.Background = Brushes.Green;
                            ttlblCCTConnectionState.Content = StringResources.MainWindowCCTConnectedTooltip;
                            break;
                        case CCTConnector.CCTConnectionState.DISCONNECTED_BY_APPLICATION:
                            //This application initiated the disconnect
                            lblCCTConnectionState.Content = StringResources.CCTDisconnected;
                            lblCCTConnectionState.Background = Brushes.Transparent;
                            ttlblCCTConnectionState.Content = StringResources.CCTDisconnected;
                            sidebar.EnableConsult(false);
                            break;
                        case CCTConnector.CCTConnectionState.DISCONNECTED_OTHER:
                            sidebar.EnableConsult(false);
                            lblCCTConnectionState.Content = StringResources.CCTDisconnected;
                            lblCCTConnectionState.Background = Brushes.Red;
                            ttlblCCTConnectionState.Content = StringResources.CCTDisconnectWarning;
                            MessageBox.Show(this,StringResources.CCTDisconnectWarning, StringResources.Warning, MessageBoxButton.OK);
                            break;
                        default:
                            log.Error("Invalid case in CCTConnectionStateChanged");
                            break;
                    }                   
                }));
            }
            else
            {
                Action<CCTConnector.CCTConnectionState> a = new Action<CCTConnector.CCTConnectionState>(CCTConnectionStateChanged);
                //Cast to object needed due to array covariance in c#, the dispatcher will try to invoke with just the first shortmessages object,
                //causing a runtime exception
                this.Dispatcher.Invoke(a, state);
            }
        }

        /// <summary>
        /// Eventhandler for when multiple agents are defined in the CCT Session
        /// </summary>
        /// <param name="agents"></param>
        /// <returns></returns>
        private void MultipleAgentsReceived(IAgent[] agents)
        {
            if (this.Dispatcher.CheckAccess())
            {
                Dispatcher.Invoke(DispatcherPriority.Normal, (Action)(() =>
                {
                    MultipleAgentDialog multiAgentDialog = new MultipleAgentDialog(agents);
                    CustomDesktop.MultipleAgentDialog.MultipleAgentDialogResult result = multiAgentDialog.Show();
                    if (result == MultipleAgentDialog.MultipleAgentDialogResult.OK)
                    {
                        ResultCode cctResult = cctConnector.SetAgent(multiAgentDialog.SelectedAgent);
                        if (cctResult.ResultType != ResultCode.ResultCodeType.SMF_SUCCESS)
                        {
                            HandleUnsuccessfulLogin(cctResult);
                        }

                    }
                    else
                    {
                        //user declined to select an agent, so disconnect CCT 
                        cctConnector.Disconnect();
                        DisplayLoggedOutControls();
                    }
                }));
            }
            else
            {
                Action<IAgent[]> a = new Action<IAgent[]>(MultipleAgentsReceived);
                //Cast to object needed due to array covariance in c#, the dispatcher will try to invoke with just the first shortmessages object,
                //causing a runtime exception
                this.Dispatcher.Invoke(a, (object)agents);
            }
        }


        private void CheckMissedWelcomeMessage(UcChatTab tab)
        {
            if (MissedWelcomeMessage.ContainsKey(tab.Contact.ID))
            {
                log.Info("Displaying missed welcome message for contact - " + tab.Contact.ID);
                tab.ChatWindow.AppendAsSystemInfo("WelcomeMessage", MissedWelcomeMessage[tab.Contact.ID]);
                MissedWelcomeMessage.Remove(tab.Contact.ID);
            }
        }



        private void AddNewHistoryWindow(CustomerHistory history)
        {
            log.Info("Adding new history window");
            UcChatTab tab = new UcChatTab(history.guid);
            tabControl.Items.Add(tab);
            tabControl.SelectedItem = tab;
            tab.HistoryClosed += TabHistoryClosed;

            log.Info("Writing chat historic transcript");
            string theTranscript = history.transcript;
            string[] stringSeparators = new string[] { "\n\n" };
            string[] lines = theTranscript.Split(stringSeparators, StringSplitOptions.None);
            foreach (string line in lines)
            {
                tab.ChatWindow.AppendAsHistoricChat(line);

            }
        }
        /// <summary>
        /// Adds a new chat window to the listener thread and the GUI.
        /// </summary>
        /// <param name="contactID"></param>
        private void AddNewChatWindow(long contactID, UcChatWindow.WindowType windowType)
        {
            log.Info(string.Format("Adding chat window for contact ID {0}", contactID));
            try
            {

                Contact contact = cctConnector.GetContact(contactID);

                //For a non-originating consult
                if (contact.IsConsultContact())
                {
                    consulting.Add(contact.ParentID, contact.ID);
                }

                //Check we have received the room data back from websocket
                Stopwatch s = new Stopwatch();
                s.Start();
                while (s.Elapsed < TimeSpan.FromSeconds(10))
                {
                    if (joinRoomDetails.ContainsKey(contactID))
                    {
                        AgentJoinRoomNotification ajrn = joinRoomDetails[contactID];
                        UcChatTab tab = new UcChatTab(contact, ajrn);
                        //Add to chatrooms map
                        chatrooms.Add(contact.ID, tab);
                        log.Info(string.Format("Added chat window to map for contact ID {0}", contactID));
                        CheckMissedWelcomeMessage(tab);
                        tab.CompleteConferenceEvent += CompleteConferenceEventHandler;
                        tab.CompleteTransferEvent += CompleteTransferEventHandler;
                        tab.ChatClosed += TabChatClosed;
                        tab.OriginatingConsultClosed += TabOriginatingConsultClosed;
                        tab.NonOriginatingConsultClosed += TabNonOriginatingConsultClosed;
                        tab.ActiveConfConsultClosed += TabActiveConfConsultClosed;
                        tab.ActiveConfClosed += TabActiveConfClosed;
                        tab.PassiveConfClosed += TabPassiveConfClosed;
                        tab.ClosedReasonState += TabClosedReasonState;
                        tab.WindowTypeChange += TabWindowTypeChange;
                        tabControl.Items.Add(tab);
                        tabControl.SelectedItem = tab;

                        TabControl chatExtraTabs = CreateNewChatExtraForContact();
                        chatExtraTabsList.Add(tab, chatExtraTabs);

                        if (ajrn.body.customFields.Count > 0)
                        {
                            PopulateCustomIntrinsics(ajrn.body.customFields, contact);
                        }
                        //update the sidebar
                        sidebar.SetContactDetails(contactID);

                        //if the contact is not a consult, allow consultation
                        sidebar.EnableConsult(!contact.IsConsultContact());

                        joinRoomDetails.Remove(contactID);

                        if (windowType == UcChatWindow.WindowType.OBSERVE)
                        {
                            observing.Add(contact.ID, tab);
                            tab.ChatWindow.SetType(UcChatWindow.WindowType.OBSERVE);
                            tab.BargeEvent += BargeEventHandler;
                            tab.ObserveClosed += TabObserveClosed;
                        }

                        //Request screen pops
                        RequestScreenPops(contactID);

                        s.Stop();
                        return;
                   }
                }
                s.Stop();

                //Couldn't join the room so revert
                MessageBox.Show("Could not join the requested room");
                DropContact(contactID);
                //Change the sidebar details back
                UcChatTab tab2 = tabControl.SelectedItem as UcChatTab;
                if (tab2 != null)
                {
                    SetActiveContactOnTabSelection(tab2);
                }
            }
            catch (Exception ex)
            {
                log.Error("An exception occured adding a chat window", ex);
            }
        }

        private void PopulateCustomIntrinsics(List<AgentJoinRoomNotification.CustomField> cFields, Contact contact)
        {
            Dictionary<string, string> temp = new Dictionary<string, string>();
            try
            {
                foreach (AgentJoinRoomNotification.CustomField cField in cFields)
                {
                    if (temp.ContainsKey("Custom:" + cField.title))
                    {
                        log.Info("Removing duplicate custom intrinsic");
                        temp.Remove("Custom:" + cField.title);
                    }
                    else
                    {
                        temp.Add("Custom:" + cField.title, cField.value);
                    }
                }
            }
            catch (Exception ex)
            {
                log.Error("Exception thrown in GetCustomIntrinsics", ex);
            }

            contact.AddCustomIntrinsics(temp);
        }


        private void BargeEventHandler(UcChatTab chatTab)
        {
            try
            {
                log.Info("BARGEEVENTHANDLER - START");
                IContact contact = null;

                foreach (SupervisedAgentNew superAgent in CCTEntities.listOfMyAgents.Values)
                {
                    if (superAgent.agentChats.ContainsKey(chatTab.Contact.ID))
                    {
                        log.Info("BARGEEVENTHANDLER - Terminal " + CCTConnector.GetCCTSession().Terminals[0]);
                        log.Info("BARGEEVENTHANDLER - Address " + CCTConnector.observingChats[chatTab.Contact.ID.ToString()].Address);

                        contact = superAgent.agentChats[chatTab.Contact.ID].chatContact;

                        log.Info("BARGEEVENTHANDLER - Attempting to barge");
                        ITerminal correctTerminal = SideBar.GetCorrectWebCommsTerminal();

                        if (correctTerminal == null)
                        {
                            log.Error("Could not find Web_Communications CCT terminal");
                            MessageBox.Show("Could not find Web_Communications CCT terminal");
                            return;
                        }

                        contact.BargeIn(correctTerminal, CCTConnector.observingChats[chatTab.Contact.ID.ToString()].Address);
                        log.Info("BARGEEVENTHANDLER - Barged In");
                    }
                }

                log.Info("BARGEEVENTHANDLER - Setting window type and close event handler");
                chatTab.ChatWindow.SetType(UcChatWindow.WindowType.BARGE);
                chatTab.BargeWhisperClosed += TabBargeWhisperClosed;
            }
            catch (Exception ex)
            {
                log.Error("BARGEEVENTHANDLER ERROR", ex);


            }
        }



        void TabWindowTypeChange(ChatWindowTypeChangeEventArgs e)
        {
            UcChatTab tab = tabControl.SelectedItem as UcChatTab;
            if (tab == null || tab.ChatWindow != e.ChatWindow)
            {
                return;
            }

            if (e.CurrentType == UcChatWindow.WindowType.NORMAL)
            {
                sidebar.EnableConsult(true);
            }
            else
            {
                sidebar.EnableConsult(false);
            }
        }

        void TabHistoryClosed(UcChatTab chatTab)
        {
            sidebar.EnableConsult(true);
            TabChatClosed(chatTab);
        }

        void TabActiveConfConsultClosed(UcChatTab chatTab)
        {
            consulting.Remove(chatTab.Contact.ID);
            cctConnector.SetActiveContact(chatTab.Contact.ID);
            sidebar.EnableConsult(true);
        }
        void TabActiveConfClosed(UcChatTab chatTab)
        {
            //disconnect the consult contact
            cctConnector.DisconnectConnection(chatTab.Contact.ID);
            sidebar.RemoveContactDetails(chatTab.Contact.ID);
            consulting.Remove(chatTab.Contact.ID);
        }
        void TabPassiveConfClosed(UcChatTab chatTab)
        {
            //disconnect the consult contact
            cctConnector.DisconnectConnection(chatTab.Contact.ID);
            sidebar.RemoveContactDetails(chatTab.Contact.ID);
            chatrooms.Remove(chatTab.Contact.ID);
            consulting.Remove(chatTab.Contact.ID);
            TabChatClosed(chatTab);
        }
        void TabNonOriginatingConsultClosed(UcChatTab chatTab)
        {
            //disconnect the consult contact
            cctConnector.DisconnectConnection(chatTab.Contact.ParentID);
            cctConnector.DisconnectConnection(chatTab.Contact.ID);

            sidebar.RemoveContactDetails(chatTab.Contact.ParentID);
            chatrooms.Remove(chatTab.Contact.ID);
            consulting.Remove(chatTab.Contact.ParentID);
            TabChatClosed(chatTab);
        }
        void TabObserveClosed(UcChatTab chatTab)
        {
            //disconnect the consult contact
            cctConnector.RemoveContact(chatTab.Contact.ID);
            cctConnector.DisconnectObserveConnection(chatTab.Contact.ID);
            sidebar.RemoveContactDetails(chatTab.Contact.ID);
            chatrooms.Remove(chatTab.Contact.ID);
            observing.Remove(chatTab.Contact.ID);
            observingData.Remove(chatTab.Contact.ID);
            TabChatClosed(chatTab);
        }
        void TabBargeWhisperClosed(UcChatTab chatTab)
        {
            cctConnector.SetActiveContact(chatTab.Contact.ID);
            sidebar.EnableConsult(true);
        }
        void TabEmailClosed(UcChatTab chatTab)
        {
            cctConnector.DisconnectConnection(chatTab.Contact.ID);
            cctConnector.RemoveConnection(chatTab.Contact.ID);
            sidebar.EnableConsult(true);
            TabChatClosed(chatTab);
        }
        void TabClosedReasonState(UcChatTab chatTab)
        {
            sidebar.EnableConsult(false);
            //As far as the listener thread is concerned, the chat is finished so remove the tab. This will also leave the conversation on the server.
        }

        private void TabOriginatingConsultClosed(UcChatTab chatTab)
        {
            //disconnect the consult contact
            cctConnector.DisconnectConnection(chatTab.ConsultContact.ID);
            cctConnector.SetActiveContact(chatTab.Contact.ID);
            consulting.Remove(chatTab.Contact.ID);
            sidebar.EnableConsult(true);
        }


        /// <summary>
        /// Reads the current configuration from the appsettings xml file.
        /// </summary>
        /// <returns></returns>
        private bool ReadConfig()
        {
            bool success = true;
            try
            {
                XmlDocument xml = new XmlDocument();
                xml.Load("AppSettings.xml");

                XmlNode node = xml.SelectSingleNode("//SMFServer");
                Globals.SMFServer = node.InnerText;
                log.Info("Using SMF Server: " + Globals.SMFServer);

                node = xml.SelectSingleNode("//ThreadSleepTime");
                Globals.ThreadSleepTime = Int32.Parse(node.InnerText);
                log.Info("Using thread sleep time: " + Globals.ThreadSleepTime);

                node = xml.SelectSingleNode("//DefaultCCTUserID");
                Globals.DefaultCCTUser = node.InnerText;
                log.Info("Using default cct user: " + Globals.DefaultCCTUser);

                node = xml.SelectSingleNode("//DefaultCCTServer");
                Globals.DefaultCCTServer = node.InnerText;
                log.Info("Using default cct server: " + Globals.DefaultCCTServer);

                node = xml.SelectSingleNode("//DefaultCCTUserDomain");
                Globals.DefaultCCTDomain = node.InnerText;
                log.Info("Using default cct domain: " + Globals.DefaultCCTDomain);

                node = xml.SelectSingleNode("//WebSocketURL");
                Globals.DefaultWSURL = node.InnerText.ToUri();
                log.Info("Using websocket URL: " + Globals.DefaultWSURL);

                node = xml.SelectSingleNode("//HAEnabled");
                Globals.HAEnabled = node.InnerText.ToUpper().Equals("TRUE");
                log.Info("HAEnabled: " + Globals.HAEnabled);

                if (Globals.HAEnabled)
                {
                    node = xml.SelectSingleNode("//CampusAlternateServer");
                    if (node.InnerText.IsNullOrEmpty())
                    {
                        Globals.DefaultCampusAlternateServer = null;
                    }
                    else
                    {
                        Globals.DefaultCampusAlternateServer = node.InnerText;
                    }
                    log.Info("Using CampusAlternateServer: " + Globals.DefaultCampusAlternateServer);

                    node = xml.SelectSingleNode("//GeographicAlternateServer");
                    if (node.InnerText.IsNullOrEmpty())
                    {
                        Globals.DefaultGeographicAlternateServer = null;
                    }
                    else
                    {
                        Globals.DefaultGeographicAlternateServer = node.InnerText;
                    }
                    log.Info("Using GeographicAlternateServer: " + Globals.DefaultGeographicAlternateServer);

                    node = xml.SelectSingleNode("//RGNWebSocket");
                    Globals.RGNWebSocket = node.InnerText.ToUri();
                    log.Info("RGNWebSocket: " + Globals.RGNWebSocket);

                    node = xml.SelectSingleNode("//RGNMMServer");
                    Globals.RGNMMServer = node.InnerText;
                    log.Info("RGNMMServer: " + Globals.RGNMMServer);

                    node = xml.SelectSingleNode("//AutoReconnect");
                    Globals.DefaultAutoReconnect = node.InnerText.ToUpper().Equals("TRUE");
                    log.Info("AutoReconnect: " + Globals.DefaultAutoReconnect);
                }

                

                node = xml.SelectSingleNode("//IsSecure");
                Globals.IsSecure = node.InnerText.ToUpper().Equals("TRUE");
                log.Info("IsSecure: " + Globals.IsSecure);

                string transportType = "http://";
                if (Globals.IsSecure)
                {
                    transportType = "https://";
                }

                node = xml.SelectSingleNode("//DefaultMMServer");
                Globals.DefaultMMServer = node.InnerText;
                log.Info("Using DefaultMMServer: " + Globals.DefaultMMServer);
                Globals.AgentUtilityWSSoapURL = transportType + node.InnerText + "/ccmmagentwebservices/AgentUtilityWS.asmx";
                Globals.AgentEmailWSSoapURL = transportType + node.InnerText + "/ccmmagentwebservices/AgentEmailWS.asmx";
                Globals.AgentContactWSSoapURL = transportType + node.InnerText + "/ccmmagentwebservices/AgentContactWS.asmx";
                log.Info("Using AgentUtilityWSSoapURL: " + Globals.AgentUtilityWSSoapURL);
                log.Info("Using AgentEmailWSSoapURL: " + Globals.AgentEmailWSSoapURL);
                log.Info("Using AgentContactWSSoapURL: " + Globals.AgentContactWSSoapURL);

                node = xml.SelectSingleNode("//DownloadAttachmentsLocation");
                Globals.DownloadAttachmentsLocation = node.InnerText;
                log.Info("DownloadAttachmentsLocation: " + Globals.DownloadAttachmentsLocation);
                
                node = xml.SelectSingleNode("//Version");
                Globals.VersionNumber = float.Parse(node.InnerText);
                log.Info("Version Number: " + Globals.VersionNumber.ToString());
                log.Info("-----------------------------------------------------");
            }
            catch (FormatException fe)
            {
                //design decision: if the conversion for the thread sleep time fails, proceed with the default 1 sec
                log.Info("Thread sleep time conversion failed, using default 1 sec", fe);
            }
            catch (Exception ex)
            {
                log.Fatal("Fatal Error reading config file:", ex);
                success = false;
            }
            return success;
        }

        private static string GetUserNameFromRegistry()
        {
            string username = null;
            try
            {
                // Opening the registry key
                RegistryKey rk = Registry.LocalMachine;
                // Open a subKey as read-only
                RegistryKey sk1 = rk.OpenSubKey(REG_KEY);
                if (sk1 == null)
                {
                    return "";
                }
                else
                {
                    return (string)sk1.GetValue(USERNAME_KEY);
                }

            }
            catch (Exception ex)
            {
                log.Error("Error occured reading user name from registry", ex);
            }
            return username;
        }

        private static string GetDomainFromRegistry()
        {
            string domain = null;
            try
            {
                // Opening the registry key
                RegistryKey rk = Registry.LocalMachine;
                // Open a subKey as read-only
                RegistryKey sk1 = rk.OpenSubKey(REG_KEY);
                if (sk1 == null)
                {
                    return "";
                }
                else
                {
                    return (string)sk1.GetValue(DOMAIN_KEY);
                }
            }
            catch (Exception ex)
            {
                log.Error("Error occured reading domain from registry", ex);
            }
            return domain;
        }

        public void WriteUserNameToRegistry(string username)
        {
            try
            {
                // Opening the registry key
                RegistryKey rk = Registry.LocalMachine;
                // Open a subKey as read-only
                RegistryKey sk1 = rk.CreateSubKey(REG_KEY);
                sk1.SetValue(USERNAME_KEY, username);
            }
            catch (Exception ex)
            {
                log.Error("Error occured writing user name to registry", ex);
            }
        }

        public void WriteDomainToRegistry(string domain)
        {
            try
            {
                // Opening the registry key
                RegistryKey rk = Registry.LocalMachine;
                // Open a subKey as read-only
                RegistryKey sk1 = rk.CreateSubKey(REG_KEY);
                sk1.SetValue(DOMAIN_KEY, domain);
            }
            catch (Exception ex)
            {
                log.Error("Error occured writing domain to registry", ex);
            }
        }
        #region CCT
        public void BlindTransferCurrentContact(CDEntities.CDContactCentreSkillset skillset)
        {
            if (this.Dispatcher.CheckAccess())
            {
                Dispatcher.BeginInvoke(DispatcherPriority.Normal, (Action)(() =>
                 {
                     UcChatTab currTab = (UcChatTab)tabControl.SelectedItem;
                     UcChatWindow chatWindow = currTab.ChatWindow;
                     ResultCode result = cctConnector.BlindTransfer(chatWindow.contactID.Value, skillset);
                     if (result.ResultType == ResultCode.ResultCodeType.CCT_SUCCESS)
                     {
                         //close the window
                         currTab.LeaveReason = CDEntities.leaveReason.BLIND_TRANSFER;
                         TabChatClosed(currTab);
                         chatrooms.Remove(chatWindow.contactID.Value);
                     }
                 }));
            }
            else
            {
                Action<CDEntities.CDContactCentreSkillset> a = new Action<CDEntities.CDContactCentreSkillset>(BlindTransferCurrentContact);
                this.Dispatcher.Invoke(a, skillset);
            }
        }

        public void ObserveContact(Contact contact)
        {
            cctConnector.AddObserveContact(contact.ID, contact);
            //invoke this on the main thread
            Dispatcher.BeginInvoke(DispatcherPriority.Normal, (Action)(() =>
            {
                log.Info(string.Format("Join conversation invoked for contact ID {0}", contact.ID));

                this.AddNewChatWindow(contact.ID, UcChatWindow.WindowType.OBSERVE);
            }));

        }

        public void ConsultAgentOnCurrentContact(long agentID)
        {
            if (this.Dispatcher.CheckAccess())
            {
                Dispatcher.BeginInvoke(DispatcherPriority.Normal, (Action)(() =>
                {
                    UcChatTab currTab = (UcChatTab)tabControl.SelectedItem;
                    UcChatWindow chatWindow = currTab.ChatWindow;
                    currTab.consultedAgent = agentID;
                    cctConnector.InitiateConference(chatWindow.contactID.Value, agentID);
                    currTab.PendingConsult = true;
                }));
            }
            else
            {
                Action<long> a = new Action<long>(ConsultAgentOnCurrentContact);
                this.Dispatcher.Invoke(a, agentID);
            }
        }

        public void ConferenceCompleted(Contact contact)
        {
            if (this.Dispatcher.CheckAccess())
            {
                //invoke this on the main thread
                Dispatcher.BeginInvoke(DispatcherPriority.Normal, (Action)(() =>
                {
                    foreach (UcChatTab tab in tabControl.Items)
                    {
                        if (tab.Contact.ParentID == contact.ID)
                        {
                            tab.ConferenceComplete(contact);
                            return;
                        }
                    }
                    //Log error as contact not found
                    log.Error(string.Format("Error: tried to ConferenceComplete {0} but the parent contact did not exist in the tab control", contact));
                }));
            }
            else
            {
                Action<Contact> a = new Action<Contact>(ConferenceCompleted);
                this.Dispatcher.Invoke(a, contact);
            }
        }

        private void ConsultRejected(Contact contact)
        {
            if (this.Dispatcher.CheckAccess())
            {
                Dispatcher.BeginInvoke(DispatcherPriority.Normal, (Action)(() =>
               {
                   //find the correct chat tab
                   foreach (UcChatTab tab in tabControl.Items)
                   {

                       if (tab.ConsultContact != null && tab.ConsultContact.ID == contact.ID && tab.PendingConsult)
                       {
                           tab.RemoveConsultWindow();
                           tabControl.SelectedItem = tab;
                           MessageBox.Show(this, StringResources.ConsultRejectedWarningMessage,StringResources.Warning);
                           sidebar.EnableConsult(true);
                           consulting.Remove(tab.ConsultContact.ParentID);
                           return;
                       }
                   }
               }));
            }
            else
            {
                Action<Contact> a = new Action<Contact>(ConsultRejected);
                this.Dispatcher.Invoke(a, contact);
            }
        }

        public void TransferCompleted(Contact contact)
        {
            if (this.Dispatcher.CheckAccess())
            {
                //invoke this on the main thread
                Dispatcher.BeginInvoke(DispatcherPriority.Normal, (Action)(() =>
                {
                    foreach (UcChatTab tab in tabControl.Items)
                    {
                        if (tab.Contact.ParentID == contact.ID)
                        {
                            //remove and replace some maps
                            chatrooms.Remove(tab.Contact.ID);
                            consulting.Remove(contact.ID);

                            tab.TransferComplete(contact);
                            sidebar.EnableConsult(true);

                            chatrooms.Add(contact.ID,tab);

                            return;
                        }
                    }
                    //Log error as contact not found
                    log.Error(string.Format("Error: tried to TransferComplete {0} but the parent contact did not exist in the tab control", contact));
                }));
            }
            else
            {
                Action<Contact> a = new Action<Contact>(TransferCompleted);
                this.Dispatcher.Invoke(a, contact);
            }
        }

        public void AddConsultWindow(Contact consultContact)
        {
            if (this.Dispatcher.CheckAccess())
            {
                //For originating consult
                consulting.Add(consultContact.ParentID, consultContact.ID);
                //invoke this on the main thread
                Dispatcher.BeginInvoke(DispatcherPriority.Normal, (Action)(() =>
                {
                    foreach (UcChatTab tab in tabControl.Items)
                    {
                        if (tab.Contact.ID == consultContact.ParentID)
                        {
                            //tab.ChatWindow.consultedAgent
                            tab.AddConsultWindow(consultContact);
                            sidebar.EnableConsult(false);
                            this.gMain.ColumnDefinitions[1].MinWidth = 600;
                            return;
                        }
                    }
                    //Log error as contact not found
                    log.Error(string.Format("Error: tried to add consult contact {0} but the parent contact did not exist in the tab control", consultContact));
                }));
            }
            else
            {
                Action<Contact> a = new Action<Contact>(AddConsultWindow);
                this.Dispatcher.Invoke(a, consultContact);
            }
        }

        public void AlertReceivedConsultContact(Contact contact)
        {
            if (this.Dispatcher.CheckAccess())
            {
                //invoke this on the main thread
                Dispatcher.BeginInvoke(DispatcherPriority.Normal, (Action)(() =>
                {
                    log.Info(string.Format("AlertConsultContact invoked for contact ID {0}", contact.ID));
                    this.Activate();
                    sidebar.AddContactDetails(contact, contact.ParentID);
                    SidebarVisibility(true);
                    sidebar.SidebarTabs.SelectedIndex = 0;
                    UcContactAlert.Visibility = Visibility.Visible;
                    UcContactAlert.AlertContact(contact);
                }));
            }
            else
            {
                Action<Contact> a = new Action<Contact>(AlertReceivedConsultContact);
                this.Dispatcher.Invoke(a, contact);
            }
        }

        public void AlertContact(Contact contact)
        {
            if (this.Dispatcher.CheckAccess())
            {
                //invoke this on the main thread
                Dispatcher.BeginInvoke(DispatcherPriority.Normal, (Action)(() =>
                {
                    log.Info(string.Format("AlertContact invoked for contact ID {0}", contact.ID));
                    this.Activate();
                    sidebar.AddContactDetails(contact);
                    SidebarVisibility(true);
                    sidebar.SidebarTabs.SelectedIndex = 0;
                    UcContactAlert.Visibility = Visibility.Visible;
                    UcContactAlert.AlertContact(contact);
                }));
            }
            else
            {
                Action<Contact> a = new Action<Contact>(AlertContact);
                this.Dispatcher.Invoke(a, contact);
            }
        }


        public void DropContact(long contactID)
        {
            if (this.Dispatcher.CheckAccess())
            {
                //invoke this on the main thread
                Dispatcher.BeginInvoke(DispatcherPriority.Normal, (Action)(() =>
                {
                    log.Info(string.Format("DropContact invoked for contact ID {0}", contactID));
                    if (tabControl.Items.Count == 0)
                    {
                        sidebar.Reset(true);
                    }
                    else
                    {
                        sidebar.Reset(false);
                    }
                    sidebar.RemoveContactDetails(contactID);
                    UcContactAlert.DropContact(contactID);
                }));
            }
            else
            {
                Action<long> a = new Action<long>(DropContact);
                this.Dispatcher.Invoke(a, contactID);
            }
        }

        private void CreateNewEmailContact(Contact contact)
        {
            EmailManager em = new EmailManager();
            Email newMail = new Email();
            newMail = em.ReadContact(contact.ID.ToString());
            
            Dispatcher.BeginInvoke(DispatcherPriority.Normal, (Action)(() =>
            {
                UcChatTab tab = new UcChatTab(contact);
                tabControl.Items.Add(tab);
                tabControl.SelectedItem = tab;

                TabControl chatExtraTabs = CreateNewChatExtraForContact();
                chatExtraTabsList.Add(tab, chatExtraTabs);

                tab.EmailClose += TabEmailClosed;

                if(newMail!=null){
                    tab.ChatWindow.emailFrom.Text = newMail.mailFrom;
                    tab.ChatWindow.emailTo.Text = newMail.mailTo;
                    tab.ChatWindow.emailCc.Text = newMail.mailCC;
                    tab.ChatWindow.emailBcc.Text = newMail.mailBCC;
                    tab.ChatWindow.emailSummary.Text = newMail.subject;
                    tab.ChatWindow.emailText.AppendText(newMail.text);
                    if (newMail.emailAttachments.Count > 0)
                    {
                        PopulateAttachments(tab, newMail);
                    }
                    else
                    {
                        tab.ChatWindow.lAttachments.Visibility = Visibility.Collapsed;
                        tab.ChatWindow.cbAttachments.Visibility = Visibility.Collapsed;
                    }
                    tab.ChatWindow.receivedEmail = newMail;
                }

                RequestScreenPops(contact.ID);
            }));
            
        }

        private void PopulateAttachments(UcChatTab tab, Email mail){

            log.Info("Populating email attachments");

            Dispatcher.BeginInvoke(DispatcherPriority.Normal, (Action)(() =>
            {
                foreach (EmailAttachment ea in mail.emailAttachments)
                {
                    tab.ChatWindow.cbAttachments.Items.Add(ea);
                }

                EmailAttachment title = new EmailAttachment();
                title.displayFileName = "-- Select an attachment to download --";
                tab.ChatWindow.cbAttachments.Items.Insert(0, title);
                tab.ChatWindow.cbAttachments.SelectedIndex = 0;
            }));
        }

        public void AcceptContact(Contact contact)
        {
            UcContactAlert.Visibility = Visibility.Hidden;
            cctConnector.AcceptContact(contact);

            //Accept an email contact
            if (contact.Intrinsics["CONTACTSUBTYPE"].ToUpper().Equals("EMAIL"))
            {
                CreateNewEmailContact(contact);
                return;
            }
            if (!contact.IsConsultContact())
            {
                JoinConversation(contact.ID);
            }
            else
            {
                //Join the chat with the parent contact ID
                JoinConversation(contact.ParentID);
            }

        }

        public void RejectContact(Contact contact)
        {
            UcContactAlert.Visibility = Visibility.Hidden;
            
            sidebar.RemoveContactDetails(contact);
            cctConnector.RejectContact(contact);

            if(tabControl.Items.Count > 0)
            {
                //Change the sidebar details back to the previous
                UcChatTab tab = tabControl.SelectedItem as UcChatTab;
                if (tab != null)
                {
                    SetActiveContactOnTabSelection(tab);
                }
            }
            else
            {
                //No chats so remove the sidebar
                SidebarVisibility(false);
            }
        }

        public void JoinConversation(long contactID)
        {
            try
            {
                if (this.Dispatcher.CheckAccess())
                {
                    //invoke this on the main thread
                    Dispatcher.BeginInvoke(DispatcherPriority.Normal, (Action)(() =>
                    {
                        log.Info(string.Format("Join conversation invoked for contact ID {0}", contactID));
                        
                        this.AddNewChatWindow(contactID, UcChatWindow.WindowType.NORMAL);
                    }));
                }
                else
                {
                    Action<long> a = new Action<long>(JoinConversation);
                    this.Dispatcher.Invoke(a, contactID);
                }
            }
            catch (Exception ex)
            {
                log.Error("Exception caught joining conversation", ex);
            }
        }

        public void SetReadyState(ReadyStatusAgentProperty newState)
        {

            try
            {
                if (this.Dispatcher.CheckAccess())
                {
                    Dispatcher.BeginInvoke(DispatcherPriority.Normal, (Action)(() =>
                    {
                        log.Info(string.Format("Setting ready state to: {0}", newState.ReadyState));
                        if (newState.IsReady)
                        {
                            btnToggleReady.Content = StringResources.GoNotReadyText;
                            ttbtnToggleReady.Content = StringResources.MainWindowGoNotReadyTooltip;
                            lblReadyState.Background = Brushes.Green;
                            this.lblReadyState.Content = StringResources.ReadyStatePrefix + StringResources.Ready;
                        }
                        else
                        {
                            btnToggleReady.Content = StringResources.GoReadyText;
                            ttbtnToggleReady.Content = StringResources.MainWindowGoReadyTooltip;
                            lblReadyState.Background = Brushes.DarkOrange;
                            this.lblReadyState.Content = StringResources.ReadyStatePrefix + StringResources.NotReady;

                        }
                    }));
                }
                else
                {
                    Action<ReadyStatusAgentProperty> a = new Action<ReadyStatusAgentProperty>(SetReadyState);
                    this.Dispatcher.Invoke(a, newState);
                }
            }
            catch (Exception ex)
            {
                log.Error("Exception caught setting ready state", ex);
            }
        }


        public void SetLoginState(LoginStatusAgentProperty newState)
        {
            try
            {
                if (this.Dispatcher.CheckAccess())
                {
                    //invoke this on the main thread
                    Dispatcher.BeginInvoke(DispatcherPriority.Normal, (Action)(() =>
                    {
                        if (newState.IsLoggedIn)
                        {
                            this.lblLoginState.Content = StringResources.LoginPrefix + StringResources.LoggedInText;
                            lblLoginState.Background = Brushes.Green;
                        }
                        else
                        {
                            this.lblLoginState.Content = StringResources.LoginPrefix + StringResources.LoggedOutText;
                            lblLoginState.Background = Brushes.DarkOrange;
                            log.Info("Received a logout state event. notifying user and logging out.");

                            if (Wsock.CheckConnectionAlive())
                            {
                                Wsock.CloseWebSocket(CloseStatusCode.Normal, "Closing the socket due to forced CCT logout");
                            }
                            MessageBox.Show(this,StringResources.AgentHasBeenLoggedOutMessage,"Information",MessageBoxButton.OK);
                            Environment.Exit(0);
                        }
                    }));
                }
                else
                {
                    Action<LoginStatusAgentProperty> a = new Action<LoginStatusAgentProperty>(SetLoginState);
                    this.Dispatcher.Invoke(a, newState);
                }
            }
            catch (Exception ex)
            {
                log.Error("Exception caught setting login state", ex);
            }

        }

        private bool slidOut = false;
        private void ChatExtraSliderClick(object sender, RoutedEventArgs e)
        {
            if (slidOut)
            {
                this.Width = 650;
                ChatColumn.MaxWidth = 2000;
                ChatExtraColumn.Width = new GridLength(1, GridUnitType.Auto);
                MainChatExtraTabs.Visibility = Visibility.Collapsed;
                slidOut = false;
            }
            else
            {
                this.Width = Width + 350;
                ChatColumn.MaxWidth = 650;
                ChatExtraColumn.Width = new GridLength(1, GridUnitType.Star);
                MainChatExtraTabs.Visibility = Visibility.Visible;
                slidOut = true;
            }
        }

        private void DisplayChatExtraSection(bool toDisplay){
            if (toDisplay)
            {
                this.Width = Width + 15;
                ChatExtraSlider.Visibility = Visibility.Visible;
            }
            else
            {
                this.Width = Width - 15;
                ChatExtraSlider.Visibility = Visibility.Collapsed;
                if (MainChatExtraTabs.Visibility == Visibility.Visible)
                {
                    this.Width = Width - 350;
                    MainChatExtraTabs.Visibility = Visibility.Collapsed;
                    slidOut = false;
                }
            }
        }

        private TabControl CreateNewChatExtraForContact()
        {
            //Create New tab control
            TabControl tabControl = new TabControl();
            tabControl.HorizontalAlignment = System.Windows.HorizontalAlignment.Stretch;

            //AgentNotes
            TabItem tabItem = new TabItem();
            Grid grid = null;
            RichTextBox textbox = null;

            textbox = new RichTextBox();
            textbox.HorizontalAlignment = System.Windows.HorizontalAlignment.Stretch;
            textbox.VerticalAlignment = System.Windows.VerticalAlignment.Stretch;
            textbox.HorizontalScrollBarVisibility = ScrollBarVisibility.Auto;
            textbox.VerticalScrollBarVisibility = ScrollBarVisibility.Auto;

            grid = new Grid();
            grid.Children.Add(textbox);

            tabItem.Header = "Notes";
            tabItem.Content = grid;

            tabControl.Items.Add(tabItem);

            Dispatcher.BeginInvoke(DispatcherPriority.Normal, (Action)(() =>
            {
                MainChatExtraTabs.Items.Add(tabControl);
                MainChatExtraTabs.SelectedItem = tabControl;

                if (ChatExtraSlider.Visibility != Visibility.Visible)
                {
                    DisplayChatExtraSection(true);
                }
            }));

            return tabControl;
        }


        private void AddNewScreenPopHandler(string url, string title, long guid)
        {
            try
            {
                Dispatcher.BeginInvoke(DispatcherPriority.Normal, (Action)(() =>
                {

                    UcChatTab selectedTab = tabControl.SelectedItem as UcChatTab;
                    if (chatExtraTabsList.ContainsKey(selectedTab))
                    {
                        TabControl theTabControl = chatExtraTabsList[selectedTab];

                        TabItem tabItem = new TabItem();
                        Grid grid = null;
                        WebBrowser wb = new WebBrowser();

                        wb.HorizontalAlignment = System.Windows.HorizontalAlignment.Stretch;
                        wb.VerticalAlignment = System.Windows.VerticalAlignment.Stretch;
                        wb.Navigate(url);

                        grid = new Grid();
                        grid.Children.Add(wb);

                        tabItem.Header = "SP: " + title;
                        tabItem.Content = grid;

                        theTabControl.Items.Add(tabItem);
                        theTabControl.SelectedItem = tabItem;
                    }

                }));
            }
            catch (Exception ex)
            {
                log.Error("Exception adding a screenpop tab", ex);
            }
        }

        private void ForceChatExtraOpenHandler()
        {
            try
            {
                Dispatcher.BeginInvoke(DispatcherPriority.Normal, (Action)(() =>
                {
                    //Slide out chat extra if not currently displayed
                    if (!slidOut)
                    {
                        RoutedEventArgs r = new RoutedEventArgs();
                        ChatExtraSliderClick(null, r);
                    }

                }));
            }
            catch (Exception ex)
            {
                log.Error("Exception forcing open chat extra", ex);
            }
        }

        public void BtnToggleReadyClick(object sender, EventArgs e)
        {
            cctConnector.ToggleReady();
        }

        private void CompleteTransferEventHandler(UcChatTab chatTab)
        {
            cctConnector.CompleteTransfer(chatTab.ConsultContact);
            sidebar.RemoveContactDetails(chatTab.Contact);
            //Remove from maps
            chatrooms.Remove(chatTab.Contact.ID);
            consulting.Remove(chatTab.Contact.ID);
            //Close the chat window
            TabChatClosed(chatTab);
        }

        private void CompleteConferenceEventHandler(UcChatTab chatTab)
        {
            cctConnector.CompleteConference(chatTab.ConsultContact);
        }

        #endregion

        private void UpdateSelectTab()
        {
            if (tabControl.SelectedItem == null)
            {
                return;
            }
            UcChatTab selectedTab = tabControl.SelectedItem as UcChatTab;

            Dispatcher.BeginInvoke(DispatcherPriority.Normal, (Action)(() =>
            {
                //find the correct chat tab
                foreach (UcChatTab tab in tabControl.Items)
                {
                    if (tab.Equals(selectedTab))
                    {
                        //Remove animation if exists
                        if (tab.NotificationPending)
                        {
                            ClearNotification(tab);
                        }

                        // Set flag
                        tab.IsCurrentTabInView = true;
                    }
                    else
                    {
                        // Reset flag
                        tab.IsCurrentTabInView = false;
                    }
                }
            }));
        }

        private void CheckIfTabNotificationIsRequired(UcChatTab tab)
        {
            if (!tab.IsCurrentTabInView)
            {
                CreateNotification(tab);
            }
        }

        private void CreateNotification(UcChatTab tab)
        {
            // Set the flag
            tab.NotificationPending = true;

            // Update tab header to alert user to the tab
            DoubleAnimation animation = new DoubleAnimation();
            animation.AutoReverse = true;
            animation.From = 1d;
            animation.To = 0d;
            animation.Duration = new Duration(TimeSpan.FromMilliseconds(800));
            animation.RepeatBehavior = RepeatBehavior.Forever;
            tab.BeginAnimation(OpacityProperty, animation);
        }

        private void ClearNotification(UcChatTab tab)
        {
            // Reset the flag
            tab.NotificationPending = false;

            // Reset the tab colour
            tab.BeginAnimation(OpacityProperty, null);
        }

        #region Event Handlers
        /// <summary>
        /// Event handler for the tab control selection changed event. updates the sidebar.
        /// </summary>
        /// <param name="sender"></param>
        /// <param name="e"></param>
        private void TabControlSelectionChanged(object sender, SelectionChangedEventArgs e)
        {
            if (tabControl.SelectedItem == null)
            {
                return;
            }
            UcChatTab tab = tabControl.SelectedItem as UcChatTab;
            tab.Background = Brushes.DarkSlateGray;
            tab.Foreground = Brushes.Red;

            //Update focus flag
            UpdateSelectTab();


            //Update the chat extra section i.e. Agent notes, screenpops, pagepush etc
            UpdateChatExtraOnTabSelection(tab);

            if (tab.ChatWindow != null && tab.ChatWindow.Type == UcChatWindow.WindowType.HISTORY)
            {
                sidebar.EnableConsult(false);
            }

            if (tab == null || tab.Contact == null)
            {
                return;
            }

            SetActiveContactOnTabSelection(tab);

        }

        private void UpdateChatExtraOnTabSelection(UcChatTab tab)
        {
            if (chatExtraTabsList.ContainsKey(tab))
            {
                if (ChatExtraSlider.Visibility != Visibility.Visible)
                {
                    DisplayChatExtraSection(true);
                }
                TabControl tc = chatExtraTabsList[tab];
                MainChatExtraTabs.SelectedItem = tc;
            }
            else
            {
                if (ChatExtraSlider.Visibility == Visibility.Visible)
                {
                    DisplayChatExtraSection(false);
                }
            }
        }
        private void SetActiveContactOnTabSelection(UcChatTab tab)
        {
            if (tab.Contact.IsConsultContact())
            {
                sidebar.SetContactDetails(tab.Contact.ParentID);
                this.gMain.ColumnDefinitions[1].MinWidth = 600;
                cctConnector.SetActiveContact(tab.Contact.ParentID);
                sidebar.EnableConsult(false);
                SidebarVisibility(true);
            }
            else if (!cctConnector.IsConnected() || tab.ChatWindow.GetState() == UcChatWindow.State.CLOSED_REASON)
            {
                this.gMain.ColumnDefinitions[1].MinWidth = 300;
                sidebar.SetContactDetails(tab.Contact.ID);
                cctConnector.SetActiveContact(tab.Contact.ID);
                sidebar.EnableConsult(true);
                SidebarVisibility(true);
            }
            else
            {
                this.gMain.ColumnDefinitions[1].MinWidth = 300;
                sidebar.SetContactDetails(tab.Contact.ID);
                cctConnector.SetActiveContact(tab.Contact.ID);
                sidebar.EnableConsult(true);
                SidebarVisibility(true);
            }
        }

        private void SidebarVisibility(bool isVisible)
        {
            if(Globals.usertype.Equals(UserType.AgentSupervisor)){
                if (isVisible)
                {
                    sidebar.supervisorTab.Visibility = Visibility.Visible;
                    sidebar.contactTab.Visibility = Visibility.Visible;
                    sidebar.historyTab.Visibility = Visibility.Visible;
                    sidebar.Visibility = Visibility.Visible;
                }
                else
                {
                    sidebar.supervisorTab.Visibility = Visibility.Visible;
                    sidebar.contactTab.Visibility = Visibility.Collapsed;
                    sidebar.historyTab.Visibility = Visibility.Collapsed;
                    sidebar.Visibility = Visibility.Visible;
                    sidebar.SidebarTabs.SelectedIndex = 1;
                }
            }
            else
            {
                if (isVisible)
                {
                    sidebar.supervisorTab.Visibility = Visibility.Collapsed;
                    sidebar.contactTab.Visibility = Visibility.Visible;
                    sidebar.historyTab.Visibility = Visibility.Visible;
                    sidebar.Visibility = Visibility.Visible;
                }
                else
                {
                    sidebar.Visibility = Visibility.Hidden;
                }
            }
        }
        /// <summary>
        /// Event handler for the logout button. logs out the current agent from cct and agent web services.
        /// </summary>
        /// <param name="sender"></param>
        /// <param name="e"></param>
        private void BtnLogoutClick(object sender, RoutedEventArgs e)
        {
            if (tabControl.Items.Count > 0)
            {
                MessageBox.Show(this,StringResources.CloseAllOpenContactsErrorMessage,StringResources.Error);
                return;
            }

            //Logout of email web service
            EmailManager em = new EmailManager();
            em.EmailLogout(Globals.agentID.ToString());
            sidebar.Reset(true);
            sidebar.Visibility = Visibility.Hidden;
            tabControl.Visibility = Visibility.Hidden;
            lblReadyState.Content = StringResources.ReadyStatePrefix;
            lblReadyState.Background = Brushes.Transparent;
            lblLoginState.Content = StringResources.LoginPrefix + StringResources.LoggedOutText;
            lblLoginState.Background = Brushes.DarkOrange;
            btnToggleReady.Visibility = Visibility.Hidden;
            lblCCTConnectionState.Visibility = Visibility.Hidden;
            lblCCTConnectionState.Background = Brushes.Transparent;
            lblSMFConnectionState.Visibility = Visibility.Hidden;
            lblSMFConnectionState.Background = Brushes.Transparent;
            cctConnector.Disconnect();
            //WS logout
            PerformWSLogout();
            ucLoginControl.Visibility = Visibility.Visible;
            imgBackground.Visibility = Visibility.Visible;
            btnLogout.Visibility = Visibility.Hidden;
            this.Title = StringResources.MainWindowLoggedOutTitle;
            Globals.CCMMpassword = null;

            if (Globals.usertype == UserType.AgentSupervisor)
            {
                //Stop requesting agent data
                sidebar.StopAgentDataRequestTimedWork();
            }
        }

        private void PerformWSLogout()
        {
            //Create WS logout request
            AgentLogoutRequest logoutRequest = new AgentLogoutRequest();
            logoutRequest.apiVersion = AgentLogoutRequest.JSapiVersion.one_point_zero;

            //Send WS logout request
            Wsock.Send(logoutRequest);

            //Close websocket
            Wsock.CloseWebSocket(CloseStatusCode.Normal, "Closing the socket due to logout");
        }

        /// <summary>
        /// Event handler for the CCT login control.
        /// Fired when login has completed (may have failed).
        /// Manipulates the interface to show or hide various controls.
        /// </summary>
        /// <param name="result"></param>
        private void UcCCTLoginLogInCompletedEventHandler(ResultCode result)
        {
            if (result.ResultType == ResultCode.ResultCodeType.SMF_SUCCESS)
            {
                DisplayLoggedInControls();
            }
            else
            {
                HandleUnsuccessfulLogin(result);
            }
        }

        private void DisplayLoggedOutControls()
        {
            log.Info("DisplayLoggedOutControls called");
            dpConnecting.Visibility = Visibility.Hidden;
            ucLoginControl.Visibility = Visibility.Visible;
            imgBackground.Visibility = Visibility.Visible;
            lblReadyState.Background = Brushes.Transparent;
            lblLoginState.Background = Brushes.Transparent;
            lblReadyState.Content = StringResources.ReadyStatePrefix;
            lblLoginState.Content = StringResources.LoginPrefix;
            lblCCTConnectionState.Background = Brushes.Transparent;
            lblSMFConnectionState.Background = Brushes.Transparent;
            lblCCTConnectionState.Visibility = Visibility.Hidden;
            lblSMFConnectionState.Visibility = Visibility.Hidden;
            CCTConnector.connectionToActiveDropped = false;
        }

        private void DisplayLoggedInControls()
        {
            log.Info("DisplayLoggedInControls called");
            dpConnecting.Visibility = Visibility.Hidden;
            ucLoginControl.Visibility = Visibility.Hidden;
            imgBackground.Visibility = Visibility.Visible;
            btnToggleReady.IsEnabled = true;
            btnToggleReady.Visibility = Visibility.Visible;
            btnLogout.Visibility = Visibility.Visible;
            lblCCTConnectionState.Visibility = Visibility.Visible;
            lblSMFConnectionState.Visibility = Visibility.Visible;
            gLogin.Visibility = Visibility.Visible;
            tabControl.Visibility = Visibility.Visible;
            this.Title = StringResources.MainWindowLoggedInPrefix + cctConnector.GetAgentString();
            lblCCTConnectionState.Background = Brushes.Green;
            lblSMFConnectionState.Background = Brushes.Green;
            lblCCTConnectionState.Content = StringResources.CCTConnected;
            lblSMFConnectionState.Content = StringResources.SMFConnected;

            if(Globals.usertype.Equals(UserType.AgentSupervisor)){
                SidebarVisibility(false);
            }
        }

        private void HandleUnsuccessfulLogin(ResultCode result)
        {
            if (this.Dispatcher.CheckAccess())
            {
                switch (result.ResultType)
                {
                    case ResultCode.ResultCodeType.CCT_NOT_REACHABLE:
                        MessageBox.Show(this,StringResources.CCTServerCommunicationFailure, StringResources.Error, MessageBoxButton.OK);
                        DisplayLoggedOutControls();
                        break;
                    case ResultCode.ResultCodeType.CCT_EXCEPTION:
                        MessageBox.Show(this,StringResources.CCTLoginGenericError + result.Message, StringResources.Error, MessageBoxButton.OK);
                        DisplayLoggedOutControls();
                        break;
                    case ResultCode.ResultCodeType.CCT_NO_AGENT_FOUND:
                        MessageBox.Show(this,StringResources.CCTNoAgentFound, StringResources.Error, MessageBoxButton.OK);
                        DisplayLoggedOutControls();
                        break;
                    case ResultCode.ResultCodeType.CCT_MULTIPLE_AGENTS_REGISTERED:
                        MultipleAgentsReceived(cctConnector.GetReceivedAgents());
                        break;
                    case ResultCode.ResultCodeType.CCT_AGENT_ALREADY_LOGGED_IN:
                        MessageBox.Show(this, StringResources.CCTAgentAlreadyLoggedInMessage, StringResources.Warning);
                        cctConnector.Disconnect();
                        DisplayLoggedOutControls();
                        break;
                    case ResultCode.ResultCodeType.CCT_CREDENTIALS_ERROR:
                        MessageBox.Show(this,StringResources.CCTLoginIncorrectCredentials, StringResources.Error, MessageBoxButton.OK);
                        DisplayLoggedOutControls();
                        break;
                    case ResultCode.ResultCodeType.SMF_AGENT_ALREADY_LOGGED_IN:
                        if (MessageBox.Show(this, StringResources.SMFAlreadyLoggedIn, StringResources.Error, MessageBoxButton.YesNo) != MessageBoxResult.Yes)
                        {
                            cctConnector.Disconnect();
                            DisplayLoggedOutControls();
                            lblReadyState.Background = Brushes.Transparent;
                        }
                        break;
                    case ResultCode.ResultCodeType.SMF_EXCEPTION:
                        MessageBox.Show(this,StringResources.SMFLoginGeneralErrorMessage + result.Message, StringResources.Error, MessageBoxButton.OK);
                        cctConnector.Disconnect();
                        DisplayLoggedOutControls();
                        break;
                    case ResultCode.ResultCodeType.SMF_NOT_REACHABLE:
                        MessageBox.Show(this, String.Format(StringResources.SMFNameResolutionError ,result.Message), StringResources.Error, MessageBoxButton.OK);
                        cctConnector.Disconnect();
                        DisplayLoggedOutControls();
                        break;
                    case ResultCode.ResultCodeType.SMF_WEB_ERROR:
                        MessageBox.Show(this, StringResources.SMFGeneralWebErrorMessage + result.Message, StringResources.Error, MessageBoxButton.OK);
                        cctConnector.Disconnect();
                        DisplayLoggedOutControls();
                        break;
                    case ResultCode.ResultCodeType.GENERAL_ERROR:
                        MessageBox.Show(this, StringResources.GeneralErrorMessage + result.Message, StringResources.Error, MessageBoxButton.OK);
                        cctConnector.Disconnect();
                        DisplayLoggedOutControls();
                        break;
                    case ResultCode.ResultCodeType.MM_COULD_NOT_OPEN_WEBSOCKET:
                        MessageBox.Show(this, "Could not open websocket", StringResources.Error, MessageBoxButton.OK);
                        cctConnector.Disconnect();
                        DisplayLoggedOutControls();
                        break;
                    case ResultCode.ResultCodeType.MM_LOGIN_ERROR:
                        MessageBox.Show(this, "Could not log in to CCMM", StringResources.Error, MessageBoxButton.OK);
                        cctConnector.Disconnect();
                        DisplayLoggedOutControls();
                        break;
                    default:
                        log.Error("Invalid case in HandleUnsuccessfulLogin");
                        break;
                }
            }
            else
            {
                Action<ResultCode> a = new Action<ResultCode>(HandleUnsuccessfulLogin);
                this.Dispatcher.BeginInvoke(a, result);
            }
        }

        /// <summary>
        /// Event handler for the CCT login interface.
        /// Fired when the log in is being carried out.
        /// </summary>
        private void UcCCTLoginLoggingInEventHandler()
        {
            try
            {
                if (this.Dispatcher.CheckAccess())
                {
                    //invoke this on the main thread
                    Dispatcher.BeginInvoke(DispatcherPriority.Normal, (Action)(() =>
                    {
                        dpConnecting.Visibility = Visibility.Visible;
                    }));
                }
                else
                {
                    Action a = new Action(UcCCTLoginLoggingInEventHandler);
                    this.Dispatcher.Invoke(a);
                }
            }
            catch (Exception ex)
            {
                log.Error("Exception caught logging in to CCT", ex);
            }
        }

        /// <summary>
        /// Event handler for the window closing. 
        /// Prevents closing the window unless all contacts are closed.
        /// </summary>
        /// <param name="sender"></param>
        /// <param name="e"></param>
        public void CloseWindow(object sender, CancelEventArgs e)
        {
            if (tabControl.Items.Count > 0)
            {
                MessageBox.Show("You must close all open contacts before closing the application");
                e.Cancel = true;
                return;
            }
            cctConnector.Disconnect();

            log.Info("MainWindowClosed");
        }

        /// <summary>
        /// Handles a chatTab close event. Fired when a chat has ended.
        /// </summary>
        /// <param name="chatTab"></param>
        private void TabChatClosed(UcChatTab chatTab)
        {
            if (chatTab != null)
            {
                //remove the chatWindow
                tabControl.Items.Remove(chatTab);               
                if (chatTab.ChatWindow.Type == UcChatWindow.WindowType.CONFERENCE)
                {
                    long contactID = chatTab.ChatWindow.contactID.Value;
                    cctConnector.RemoveContact(contactID);
                    sidebar.RemoveContactDetails(contactID);
                }
                if (tabControl.Items.Count == 0)
                {
                    sidebar.Reset(true);

                    //Remove chatextra
                    DisplayChatExtraSection(false);

                    //Reset sizes
                    ChatColumn.MaxWidth = 2000;
                    ChatExtraColumn.Width = new GridLength(1, GridUnitType.Auto);
                }
                else
                {
                    sidebar.Reset(false);
                }
            }
        }
        #endregion


    }
}
