﻿//////////////////////////////////////////////////////////////////////////////
// EmailManager.cs
//
// Copyright 2015 Avaya Inc. All Rights Reserved.
//
// Usage of this source is bound to the terms described in
// Licenses/License.txt
//
// Avaya - Confidential & Proprietary. Use pursuant to your signed agreement or
// Avaya Policy
//////////////////////////////////////////////////////////////////////////////

using CustomDesktop.CCMMAgentContactWS;
using CustomDesktop.CCMMAgentEmailWS;
using CustomDesktop.CCMMAgentUtilityWS;
using System;
using System.Collections.Generic;
using System.IO;
using System.Net;
using System.Net.Security;
using System.Security.Cryptography.X509Certificates;
using System.Text.RegularExpressions;
using System.Windows.Forms;

namespace CustomDesktop
{
    class EmailManager
    {
        // Create a logger for use in this class
        private static readonly log4net.ILog log = log4net.LogManager.GetLogger(System.Reflection.MethodBase.GetCurrentMethod().DeclaringType);
        private CCMMAgentUtilityWS.AgentUtilityWSSoapClient wsUtiliy;
        private CCMMAgentContactWS.AgentContactWSSoapClient contactWS;
        private CCMMAgentEmailWS.AgentEmailWSSoapClient emailWS;
        static string session = string.Empty;
        private AWContact contact;
        private static Dictionary<string,CCMMAgentEmailWS.ArrayOfAWAttachment> DictionaryAttachmentsList = new Dictionary<string,ArrayOfAWAttachment>();


        public EmailManager()
        {
            if (Globals.IsSecure)
            {
                ServicePointManager.ServerCertificateValidationCallback += new RemoteCertificateValidationCallback(ValidateRemoteCertificate);
                wsUtiliy = new AgentUtilityWSSoapClient("Secure_AgentUtilityWSSoap", Globals.AgentUtilityWSSoapURL);
                contactWS = new AgentContactWSSoapClient("Secure_AgentContactWSSoap", Globals.AgentContactWSSoapURL);
                emailWS = new AgentEmailWSSoapClient("Secure_AgentEmailWSSoap", Globals.AgentEmailWSSoapURL);
            }
            else
            {
                wsUtiliy = new AgentUtilityWSSoapClient("Unsecure_AgentUtilityWSSoap", Globals.AgentUtilityWSSoapURL);
                contactWS = new AgentContactWSSoapClient("Unsecure_AgentContactWSSoap", Globals.AgentContactWSSoapURL);
                emailWS = new AgentEmailWSSoapClient("Unsecure_AgentEmailWSSoap", Globals.AgentEmailWSSoapURL);
            }
        }

        private bool ValidateRemoteCertificate(object sender, X509Certificate certificate, X509Chain chain, SslPolicyErrors sslPolicyErrors)
        {
            bool result = false;
            log.Info("Cert server name " + certificate.Subject);
            if (certificate.Subject.ToUpper().Contains("YourServerName"))
            {
                result = true;
            }

            return true; //Change if using proper certs
        }

        public void EmailLogon(string username, string password)
        {
            if (!TryLogon(username, password))
            {
                log.Info("Retrying email login");
                TryLogon(username, password);
            }

        }

        private bool TryLogon(string username, string password)
        {
            try
            {
                session = wsUtiliy.AgentDesktopLogin(username, password);
                contactWS.Open();
                log.Info("Logged into email with session - " + session);
                return true;
            }
            catch (Exception ex)
            {
                log.Error("Could not log in to email service", ex);
                log.Error("Email endpoint - " + wsUtiliy.Endpoint.Address.ToString());
                EmailLogout(username);
                return false;
            }
        }

        public void EmailLogout(string username)
        {
            try
            {
                if (session != null)
                {
                    wsUtiliy.AgentDesktopLogoff(username);
                    session = null;
                    log.Info("Logged out of email service");
                }

            }
            catch (Exception ex)
            {
                log.Error("Could not logout of email service", ex);
            }
        }

        public void CloseContact(string guid, string closeComment, long closeReasonId, bool closeReasonCodeSpecified)
        {
            try
            {
                contactWS.CloseContact(guid, closeComment, closeReasonId, closeReasonCodeSpecified, session);
                log.Info("Closed email contact - GUID:" + guid);
            }
            catch (Exception ex)
            {
                log.Error("Could not close email contact - GUID:" + guid, ex);
                MessageBox.Show(ex.ToString());
            }
        }

        public enum Recipients
        {
            CUSTOMER,
            ALL

        }

        public void SendReply(Recipients recipients, string guid, string subject, string body)
        {
            try
            {
                if (recipients == Recipients.CUSTOMER)
                {
                    if (DictionaryAttachmentsList.ContainsKey(guid))
                    {
                        log.Info("Sending email reply with attachments to sender - GUID:" + guid);

                        string url = string.Format("http://{0}/agentdesktop/UploadFile.aspx?AgentID={1}&ContactID={2}&OutboundLocation=G:%5CAvaya%5CContact%20Center%5CEmail%20Attachments%5COutbound",
                            Globals.DefaultMMServer, Globals.agentID, guid);
                        log.Info("Upload attachment URL: " + url);

                        CCMMAgentEmailWS.ArrayOfAWAttachment attachments = DictionaryAttachmentsList[guid];
                        foreach (CCMMAgentEmailWS.AWAttachment aw in attachments)
                        {
                            UploadAttachmentToShare(aw.fileContents, aw.displayFileName, "application/octet-stream", url);
                        }
                        emailWS.ReplyWithAttachments(guid, subject, body, session, DictionaryAttachmentsList[guid]);
                    }
                    else
                    {
                        log.Info("Sending email reply to sender - GUID:" + guid);
                        emailWS.Reply(guid, subject, body, session);
                    }
                }
                else if (recipients == Recipients.ALL)
                {
                    log.Info("Sending email reply to all - GUID:" + guid);
                    emailWS.ReplyAll(guid, subject, body, session);
                }
            }
            catch (Exception ex)
            {
                log.Error("Could not send email reply - GUID:" + guid, ex);
            }
        }

        public void Forward(string guid, string subject, string body, string to, string cc, string bcc)
        {
            try
            {
                log.Info("Forwarding email");
                emailWS.Forward(guid, subject, body, to, cc, bcc, session);
            }
            catch (Exception ex)
            {
                log.Error("Could not forward email - GUID:" + guid, ex);
            }
        }

        public void UploadAttachmentToShare(byte[] file, string filename, string contentType, string url){

            try
            {
                log.Info("Uploading attachment to share - " + filename);

                var webClient = new WebClient();
                string boundary = "------------------------" + DateTime.Now.Ticks.ToString("x");
                webClient.Headers.Add("Content-Type", "multipart/form-data; boundary=" + boundary);
                var fileData = webClient.Encoding.GetString(file);
                var package = string.Format("--{0}\r\nContent-Disposition: form-data; name=\"file\"; filename=\"{1}\"\r\nContent-Type: {2}\r\n\r\n{3}\r\n--{0}--\r\n",
                    boundary, filename, contentType, fileData);

                var nfile = webClient.Encoding.GetBytes(package);

                webClient.UploadData(url, "POST", nfile);
            }
            catch (Exception ex)
            {
                log.Error("Could not uplaod attachment to share", ex);
            }

        }

        public CCMMAgentEmailWS.ArrayOfAWAttachment AddToAttachmentsList(Microsoft.Win32.OpenFileDialog openFileDialog, string guid)
        {
            try
            {
                CCMMAgentEmailWS.ArrayOfAWAttachment AttachmentsList;

                if (DictionaryAttachmentsList.ContainsKey(guid))
                {
                    AttachmentsList = DictionaryAttachmentsList[guid];
                }
                else
                {
                    AttachmentsList = new CCMMAgentEmailWS.ArrayOfAWAttachment();
                }
                

                for (int inx = 0; inx < openFileDialog.FileNames.Length; inx++)
                {
                    string sFileName = openFileDialog.FileNames[inx];
                    FileInfo tempFI = new FileInfo(sFileName);
                    byte[] fileInBytes = File.ReadAllBytes(sFileName);

                    long tempFileSize = tempFI.Length;

                    if (tempFileSize > 1048576) // 1MB, (arbitrary) limit for test client 
                    {
                        MessageBox.Show(sFileName + " is too large.\n  For this test client, file sizes are limited to 1MB", "FILE IS TOO LARGE", MessageBoxButtons.OK, MessageBoxIcon.Error);
                    }
                    else
                    {
                        CCMMAgentEmailWS.AWAttachment latestAttachment = new CCMMAgentEmailWS.AWAttachment();

                        latestAttachment.displayFileName = GetFileName(sFileName);
                        latestAttachment.internalFileName = "\\" + Globals.agentID + "\\" + guid + "\\" + latestAttachment.displayFileName;
                        latestAttachment.fileContents = fileInBytes;

                        CCMMAgentEmailWS.AWCodeMapping cmDirection = new CCMMAgentEmailWS.AWCodeMapping();
                        cmDirection.fieldName = "AttachmentDirection";
                        cmDirection.textValue = "OutAttachShare"; // Inbound attachments use "InAttachShare"
                        cmDirection.numericValue = 2; // Inbound use 2
                        cmDirection.numericValueSpecified = true;
                        latestAttachment.direction = cmDirection;

                        AttachmentsList.Add(latestAttachment);

                        if (DictionaryAttachmentsList.ContainsKey(guid))
                        {
                            DictionaryAttachmentsList.Remove(guid);
                        }

                        DictionaryAttachmentsList.Add(guid, AttachmentsList);
                    }
                }

                return AttachmentsList;
            }
            catch (Exception Ex)
            {
                MessageBox.Show("Error attaching file: " + Ex);
                return null;
            }
        }

        public CCMMAgentEmailWS.ArrayOfAWAttachment RemoveFromAttachmentsList(CCMMAgentEmailWS.AWAttachment attachment, string guid)
        {
            try
            {
                // Get the attachment list
                CCMMAgentEmailWS.ArrayOfAWAttachment AttachmentsList;
                if (DictionaryAttachmentsList.ContainsKey(guid))
                {
                    AttachmentsList = DictionaryAttachmentsList[guid];
                }
                else
                {
                    AttachmentsList = new CCMMAgentEmailWS.ArrayOfAWAttachment();
                }

                // Remove the attachment
                AttachmentsList.Remove(attachment);

                // Update the attachment list
                if (DictionaryAttachmentsList.ContainsKey(guid))
                {
                    DictionaryAttachmentsList.Remove(guid);
                }
                DictionaryAttachmentsList.Add(guid, AttachmentsList);

                return AttachmentsList;
            }
            catch (Exception Ex)
            {
                MessageBox.Show("Error removing attachment: " + Ex);
                return null;
            }
        }

        private string GetFileName(string strFullPath)
        {
            string strResult = strFullPath;

            string[] strPathArray = Regex.Split(strFullPath, @"\\");
            strResult = strPathArray[strPathArray.GetUpperBound(0)];

            return strResult;
        }

        public Email ReadContact(string guid)
        {
            try
            {
                Email newMail = new Email();
                contact = contactWS.ReadContact(guid, session);


                if (contact.actionList != null && contact.actionList.Length > 0)
                {
                    log.Info("Reading customer email");
                    newMail.mailFrom = contact.actionList[0].mailFrom;
                    newMail.mailTo = contact.actionList[0].mailTo;
                    newMail.mailCC = contact.actionList[0].mailCC;
                    newMail.mailBCC = contact.actionList[0].mailBCC;
                    newMail.subject = contact.actionList[0].subject;
                    newMail.text = contact.actionList[0].text;

                    if (contact.actionList[0].attachmentList != null)
                    {
                        CCMMAgentContactWS.AWAttachment[] attachmentList = contact.actionList[0].attachmentList;


                        foreach (CCMMAgentContactWS.AWAttachment tempAttach in attachmentList)
                        {
                            log.Info("Found email attachments");
                            EmailAttachment attachment = new EmailAttachment();

                            attachment.displayFileName = tempAttach.displayFileName;
                            attachment.fileContents = tempAttach.fileContents;
                            attachment.id = tempAttach.id;
                            attachment.idSpecified = tempAttach.idSpecified;
                            attachment.internalFilename = tempAttach.internalFileName;
                            attachment.ResolveURL(tempAttach.internalFileName);

                            newMail.emailAttachments.Add(attachment);

                        }

                    }
                }

                return newMail;
            }
            catch (Exception ex)
            {
                MessageBox.Show(ex.Message, "Error", MessageBoxButtons.OK, MessageBoxIcon.Exclamation);
                return null;

            }
        }


    }
}
