﻿//////////////////////////////////////////////////////////////////////////////
// UcContactAlert.xaml.cs
//
// Copyright © 2008-2014 Avaya Inc. All rights reserved.
// See: www.avaya.com
//////////////////////////////////////////////////////////////////////////////
using EncoderDecoder;
using EncoderDecoder.Requests;
using System;
using System.Windows;
using System.Windows.Controls;
using System.Windows.Media;
using System.Windows.Media.Animation;

namespace CustomDesktop
{
    /// <summary>
    /// Interaction logic for UcContactAlert.xaml
    /// </summary>
    public partial class UcContactAlert : UserControl
    {
        // Create a logger for use in this class
        private static readonly log4net.ILog log = log4net.LogManager.GetLogger(System.Reflection.MethodBase.GetCurrentMethod().DeclaringType);
        public delegate void ContactAccepted(Contact contact);
        public delegate void ContactRejected(Contact contact);
        public event ContactAccepted ContactAcceptEvent;
        public event ContactRejected ContactRejectEvent;
        public static Brush contactBrush;
        ColorAnimation animation;
        private Contact alertingContact;

        public UcContactAlert()
        {
            InitializeComponent();
            contactBrush = FindResource("contactBrush") as SolidColorBrush;
        }

        /// <summary>
        /// Alerts a contact in the sidebar
        /// </summary>
        /// <param name="contact"></param>
        public void AlertContact(Contact contact)
        {
            try
            {
                // Animate the brush 
                animation = new ColorAnimation();
                animation.AutoReverse = true;
                animation.To = (Color)ColorConverter.ConvertFromString("#FFDA1F");
                animation.Duration = new Duration(TimeSpan.FromMilliseconds(500));
                animation.RepeatBehavior = RepeatBehavior.Forever;
                contactBrush.BeginAnimation(SolidColorBrush.ColorProperty, animation);
                alertingContact = contact;
                lblConsult.Visibility = contact.IsConsultContact() ? Visibility.Visible : Visibility.Hidden;
                this.tbSkillset.Text = contact.Intrinsics["Skillset"];
            }
            catch (Exception ex)
            {
                log.Error("An exception occured alerting contact " + contact.ID, ex);
            }
        }


        /// <summary>
        /// Event Handler for the contact accept button
        /// </summary>
        /// <param name="sender"></param>
        /// <param name="e"></param>
        private void BtnAcceptClick(object sender, RoutedEventArgs e)
        {
            /// Call ws method
            contactBrush.BeginAnimation(SolidColorBrush.ColorProperty, null);

            // Accepting email subtype
            if (alertingContact.Intrinsics["CONTACTSUBTYPE"].ToUpper().Equals("EMAIL"))
            {
                ContactAcceptEvent(alertingContact);
                return;
            }
            if (ContactAcceptEvent != null)
            {
                ContactAcceptEvent(alertingContact);

                //Wsock
                AgentJoinRoomRequest request = new AgentJoinRoomRequest();
                request.apiVersion = AgentJoinRoomRequest.JSapiVersion.one_point_zero;
                if (alertingContact.IsConsultContact())
                {
                    request.body.guid = alertingContact.ParentID;
                    request.body.joinType = AgentJoinRoomRequest.JSjoinType.consult;
                }
                else
                {
                    request.body.guid = alertingContact.ID;
                    request.body.joinType = AgentJoinRoomRequest.JSjoinType.active;
                }
                               
                Wsock.Send(request);

            }
        }

        /// <summary>
        /// Event Handler for the contact reject button
        /// </summary>
        /// <param name="sender"></param>
        /// <param name="e"></param>
        private void BtnRejectClick(object sender, RoutedEventArgs e)
        {
            contactBrush.BeginAnimation(SolidColorBrush.ColorProperty, null);
            if (ContactRejectEvent != null)
            {
                ContactRejectEvent(alertingContact);
            }
        }
        
        /// <summary>
        /// Attempts to drop the given contact from this contact alert control.
        /// If the given contact is not the same as the currently alerting contact, no action is performed.
        /// Otherwise, stops the animation, clears the current contact and hides this control.
        /// </summary>
        /// <param name="contactID"></param>
        public void DropContact(long contactID)
        {
            if (alertingContact != null && contactID == alertingContact.ID)
            {
                contactBrush.BeginAnimation(SolidColorBrush.ColorProperty, null);
                alertingContact = null;
                this.Visibility = Visibility.Hidden;
            }
        }
    }
}
