﻿//////////////////////////////////////////////////////////////////////////////
// AgentJoinRoomNotification.cs
//
// Copyright 2015 Avaya Inc. All Rights Reserved.
//
// Usage of this source is bound to the terms described in
// Licenses/License.txt
//
// Avaya - Confidential & Proprietary. Use pursuant to your signed agreement or
// Avaya Policy
//////////////////////////////////////////////////////////////////////////////
using Newtonsoft.Json;
using Newtonsoft.Json.Linq;
using Newtonsoft.Json.Serialization;
using System;
using System.Collections.Generic;

namespace EncoderDecoder.Notifications
{
    public class AgentJoinRoomNotification : IBaseNotification
    {
        private static readonly log4net.ILog log = log4net.LogManager.GetLogger(System.Reflection.MethodBase.GetCurrentMethod().DeclaringType);

        public AgentJoinRoomNotification()
        {
            this.body = new Body();
        }

        public string apiVersion { get; set; }
        public string type { get; set; }
        public Body body { get; set; }

        public class Body
        {
            public Body()
            {
                this.conversationTranscript = new ConversationTranscript();
                this.conversationTranscript.messages = new List<TranscriptEvent>();
                this.currentParticipantList = new CurrentParticipantList();
                this.currentParticipantList.participants = new List<Participant>();
                this.timerList = new TimerList();
                this.customFields = new List<CustomField>();
                this.resources = new Resources();
                this.frequentlyUsedPhraseList = new List<FrequentlyUsedPhraseList>();
                this.pagePushURLList = new List<PagePushURLList>();
            }

            public string method { get; set; }
            public long guid { get; set; }
            public string deviceType { get; set; }
            public long previousTranscripts { get; set; }
            public ConversationTranscript conversationTranscript { get; set; }
            public CurrentParticipantList currentParticipantList { get; set; }
            public TimerList timerList { get; set; }
            public List<CustomField> customFields { get; set; }
            public Resources resources { get; set; }
            public List<FrequentlyUsedPhraseList> frequentlyUsedPhraseList { get; set; }
            public List<PagePushURLList> pagePushURLList { get; set; }
        }

        public abstract class TranscriptEvent
        {
            public TranscriptEvent()
            {
                sender = new Sender();
            }
            public Sender sender { get; set; }
            public string displayName { get; set; }
            public long timestamp { get; set; }
            
        }

        public class TranscriptMessage : TranscriptEvent
        {

            public string message { get; set; }
            public string messageType { get; set; }
        }

        public class TranscriptPagePush : TranscriptEvent
        {

            public string url { get; set; }
            public string destination { get; set; }
        }

        public class TranscriptEventConverter : JsonConverter
        {
            public override bool CanConvert(Type objectType)
            {
                return typeof(TranscriptEvent).IsAssignableFrom(objectType);
            }

            public override object ReadJson(JsonReader reader,
               Type objectType, object existingValue, JsonSerializer serializer)
            {
                JObject item = JObject.Load(reader);
                switch (item["type"].Value<string>()) {
                    case "pagePush":
                        return item.ToObject<TranscriptPagePush>();
                    case "message":
                        return item.ToObject<TranscriptMessage>();
                    default:
                        log.Warn("Ignoring unrecognized transcript event type: " + item["type"]);
                        // use null to indicate unrecognized type
                        return null;
                }
            }

            public virtual bool CanWrite
            {
                get { return false; }
            }

            public override void WriteJson(JsonWriter writer, object value, JsonSerializer serializer)
            {
                throw new NotImplementedException();
            }

        }

        public class Sender
        {
            public string type { get; set; }
            public string agentId { get; set; }
        }

        public class ConversationTranscript
        {
            // Ignore values that could not be deserialized to allow other event types 
            // to be added in the future without breaking backwards compatibility
            [JsonProperty(NullValueHandling = NullValueHandling.Ignore)]
            public List<TranscriptEvent> messages { get; set; }
        }

        public class CustomField
        {
            public string title { get; set; }
            public string value { get; set; }
        }

        public class Participant
        {
            public string agentId { get; set; }
            public string userType { get; set; }
        }

        public class CurrentParticipantList
        {
            public List<Participant> participants { get; set; }
        }

        public class TimerList
        {
            public int keepAliveTime_inSeconds { get; set; }
            public int desirableResponseToCust_inSeconds { get; set; }
            public int desirableResponseToAgent_inSeconds { get; set; }
            public int consultRequestTimeout_inSeconds { get; set; }
            public bool forceIdleCustomerCheck { get; set; }
            public int forceIdleCustomerCheckTimeout_inSeconds { get; set; }
            public bool saveTimeStampOnMessages { get; set; }
            public bool saveChatHistory { get; set; }
            public int concurrentChatsLimitPerCustomer { get; set; }
            public int requestedCallBacksLimitPerCustomer { get; set; }
            public bool emailChatLogToCustomer { get; set; }
        }

        public class Resources
        {
            public string agentLabel { get; set; }
            public string customerLabel { get; set; }
        }

        public class Phrases
        {
            public string name { get; set; }
            public string phraseText { get; set; }
        }

        public class FrequentlyUsedPhraseList
        {
            public string skillset { get; set; }
            public List<Phrases> phrases { get; set; }
        }

        public class Url
        {
            public string url { get; set; }
            public string description { get; set; }
        }

        public class PagePushURLList
        {
            public string skillset { get; set; }
            public List<Url> urls { get; set; }
        }

        /// <summary>
        /// Decodes this notifications and returns object representation.
        /// </summary>
        /// <param name="json"></param>
        /// <returns></returns>
        public IBaseNotification Decode(string json)
        {
            return JsonConvert.DeserializeObject<AgentJoinRoomNotification>(json, new TranscriptEventConverter());
        }
    }
}
