﻿//////////////////////////////////////////////////////////////////////////////
// CCTSupervisor.cs
//
// Copyright 2015 Avaya Inc. All Rights Reserved.
//
// Usage of this source is bound to the terms described in
// Licenses/License.txt
//
// Avaya - Confidential & Proprietary. Use pursuant to your signed agreement or
// Avaya Policy
//////////////////////////////////////////////////////////////////////////////

using Nortel.CCT;
using System;
using System.Collections.Generic;

namespace CustomDesktop
{
    class CCTSupervisor
    {
        // Create a logger for use in this class
        private static readonly log4net.ILog log = log4net.LogManager.GetLogger(System.Reflection.MethodBase.GetCurrentMethod().DeclaringType);

        private Toolkit cctToolkit;
        private CCTEntities cctEntities;

        private string agentId;
        private string agentName;
        private AgentReadyState readyState;
        private UserType userType;
        private bool loggedIn;

        private static Dictionary<string, List<ChatNew>> existingChats = new Dictionary<string, List<ChatNew>>();
        private static bool completedProcessingInitialData;

        private UserPropertyEventHandler SupervisedUserPropertyEventHandler;
        private TermConnStateEventHandler SupervisedTermConnStateEventHandler;
        private TermConnPropertyEventHandler SupervisedTermConnPropertyEventHandler;
        private TerminalPropertyEventHandler SupervisedTerminalPropertyEventHandler;

        public CCTSupervisor(Toolkit toolkit)
        {
            cctToolkit = toolkit;
            cctEntities = new CCTEntities();
            SupervisedUserPropertyEventHandler = cctToolkit.CreateEventHandler(new UserPropertyEventHandler(SupervisedUserPropertyEvent));
            SupervisedTermConnStateEventHandler = cctToolkit.CreateEventHandler(new TermConnStateEventHandler(SupervisedTermConnStateEvent));
            SupervisedTermConnPropertyEventHandler = cctToolkit.CreateEventHandler(new TermConnPropertyEventHandler(SupervisedTermConnPropertyEvent));
            SupervisedTerminalPropertyEventHandler = cctToolkit.CreateEventHandler(new TerminalPropertyEventHandler(SupervisedTerminalPropertyEvent));
        }

        public void SubscribeUserForEvents(IUser user)
        {
            log.Info(String.Format("Adding user {0} to base event", user.UserID));
            user.UserPropertyChanged += SupervisedUserPropertyEventHandler;

            IAgent agent = (IAgent)user;
            loggedIn = agent.IsLoggedIn;
            readyState = agent.ReadyState;
            agentId = user.UserID;
            agentName = user.FullName;
            userType = user.UserType;

            if (loggedIn)
            {
                SubscribeAgentForEvents(user);
            }

            SupervisedAgentNew newAgent = new SupervisedAgentNew();
            newAgent.agentId = user.UserID;
            newAgent.agentFirstName = user.FirstName;
            newAgent.agentLastName = user.LastName;
            newAgent.agentFullName = user.FullName;
            newAgent.loggedIn = agent.IsLoggedIn;
            newAgent.readyState = agent.ReadyState;
            newAgent.userType = user.UserType;
            cctEntities.AddAgent(newAgent);

            cctEntities.UpdateSidebar();
        }

        public void SubscribeAgentForEvents(IUser user)
        {
            IAgent agent = (IAgent)user;
            if (loggedIn)
            {
                log.Info(String.Format("Subscribing user {0} to events", user.UserID));
                agent.Terminals[0].TermConnStateChanged += SupervisedTermConnStateEventHandler;
                agent.Terminals[0].TermConnPropertyChanged += SupervisedTermConnPropertyEventHandler;
                agent.Terminals[0].PropertyChanged += SupervisedTerminalPropertyEventHandler;
            }
        }

        public void UnsubscribeAgentForEvents(IUser user)
        {
            IAgent agent = (IAgent)user;
            loggedIn = false;

            log.Info(String.Format("Unsubscribing user {0} to events", user.UserID));
            agent.Terminals[0].TermConnStateChanged -= SupervisedTermConnStateEventHandler;
            agent.Terminals[0].TermConnPropertyChanged -= SupervisedTermConnPropertyEventHandler;
            agent.Terminals[0].PropertyChanged -= SupervisedTerminalPropertyEventHandler;
        }

        public void SupervisedTermConnStateEvent(TermConnStateEventArgs e)
        {
            ChatNew chatData = new ChatNew();
            chatData.guid = Convert.ToInt64(e.Contact.ID);
            chatData.status = e.NewState;
            chatData.chatContact = e.Contact;
            chatData.chatTerminal = e.TerminalConnection.Terminal;
            chatData.chatConnection = e.Connection;

            if (e.NewState == TerminalConnectionState.Dropped)
            {
                cctEntities.RemoveChatData(agentId, chatData);
            }
            else
            {
                cctEntities.UpdateChatData(agentId, chatData);
            }

            cctEntities.UpdateSidebar();
        }

        public void SupervisedTermConnPropertyEvent(TermConnPropertyEventArgs e)
        {
            log.Debug("In SupervisedTermConnPropertyEvent");
        }

        public void SupervisedTerminalPropertyEvent(TerminalPropertyEventArgs e)
        {
            log.Debug("In SupervisedTerminalPropertyEvent");
        }

        public void SupervisedUserPropertyEvent(UserPropertyEventArgs e)
        {
            IAgent testAgent = (IAgent)e.Sender;

            if (e.ChangedProperty is LoginStatusAgentProperty)
            {
                if (loggedIn != testAgent.IsLoggedIn)
                {
                    //Update logged in status
                    UpdateEventsBasedOnLoginLogout(e.Sender);
                }
            }

            if (e.ChangedProperty is ReadyStatusAgentProperty)
            {
                //Update ready status
                UpdateBaseAgentStatus(e.Sender);
            }
        }

        private void UpdateEventsBasedOnLoginLogout(IUser user)
        {
            IAgent agent = (IAgent)user;

            //Changes to update sidebar with new login status
            UpdateBaseAgentStatus(user);

            //Subscribe or unsubscribe from events
            loggedIn = agent.IsLoggedIn;
            if (loggedIn)
            {
                cctEntities.UpdateAgentLoggedInStatus(agentId, true);
                SubscribeAgentForEvents(user);
            }
            else
            {
                cctEntities.UpdateAgentLoggedInStatus(agentId, false);
                UnsubscribeAgentForEvents(user);
            }
        }

        private void UpdateBaseAgentStatus(IUser user)
        {
            IAgent agent = (IAgent)user;
            cctEntities.UpdateAgentReadyStatus(agentId, agent.ReadyState);
        }

        public void GetInitialAgentData(ISession session)
        {
            try
            {
                log.Info("Getting initial CCT data when supervisor logs in");
                completedProcessingInitialData = false;

                ISupervisor[] supervisors = session.Supervisors;
                foreach (ISupervisor supervisor in supervisors)
                {
                    if (supervisor.UserID.Equals(Globals.agentID.ToString()))
                    {
                        log.Info("Found myself as supervisor");

                        //Get my agents
                        IAgent[] superAgents = supervisor.Agents;
                        log.Info(String.Format("Amount of Agents - {0}", superAgents.Length));

                        foreach (IAgent superAgent in superAgents)
                        {
                            //Get his/her terminals
                            IAgentTerminal[] agentsTerms = superAgent.Terminals;

                            if (agentsTerms.Length < 1)
                            {
                                continue;
                            }

                            foreach (IAgentTerminal agentsTerm in agentsTerms)
                            {
                                ITerminalConnection[] termConns = agentsTerm.TerminalConnections;

                                if (termConns.Length > 0)
                                {
                                    foreach (ITerminalConnection termConn in termConns)
                                    {
                                        ChatNew newChat = new ChatNew();
                                        newChat.status = termConn.CurrentState;
                                        newChat.guid = Convert.ToInt64(termConn.ContactID);
                                        newChat.chatConnection = termConn.Connection;
                                        newChat.chatContact = termConn.Contact;
                                        newChat.chatTerminal = termConn.Terminal;
                                        newChat.status = termConn.CurrentState;

                                        log.Info(String.Format("Found an ongoing chat on contact {0} so adding to to be populated later", termConn.ContactID));
                                        AddToExistingChatMap(superAgent.UserID, newChat);
                                    }
                                }
                            }
                        }

                        log.Info("Completed getting initial CCT data when supervisor logs in");
                        completedProcessingInitialData = true;
                        log.Info("UPDATE-SIDEBAR from GetInitialAgentData");
                        cctEntities.UpdateSidebar();
                    }
                    else
                    {
                        log.Error("Other supervisor");
                    }
                }
            }
            catch (Exception ex)
            {
                log.Info("Exception in get agent data - " + ex);
            }
        }

        private void AddToExistingChatMap(string key, ChatNew value)
        {
            log.Info("Trying to AddToExistingChatMap - Key-" + key);
            if (existingChats.ContainsKey(key))
            {
                log.Info("existingChats.ContainsKey true");
                List<ChatNew> list = existingChats[key];
                if (!list.Contains(value))
                {
                    log.Info("!list.Contains(value)");
                    list.Add(value);
                }
                log.Info("Removing and adding list");
                existingChats.Remove(key);
                existingChats.Add(key, list);
            }
            else
            {
                log.Info("existingChats.ContainsKey false");
                List<ChatNew> list = new List<ChatNew>();
                list.Add(value);
                existingChats.Add(key, list);
            }
        }

        public void AddExistingChats(string agentId)
        {
            while (!completedProcessingInitialData)
            {
                log.Info("Not finished getting initial data");
                System.Threading.Thread.Sleep(100);
            }
            try
            {
                if (existingChats.ContainsKey(agentId))
                {
                    log.Info(String.Format("Populating initial chats for agent {0}", agentId));
                    List<ChatNew> agentsExistingChats = existingChats[agentId];
                        foreach (ChatNew chat in agentsExistingChats)
                        {
                            cctEntities.UpdateChatData(agentId, chat);
                        }
                    cctEntities.UpdateSidebar();
                }
            }
            catch (Exception ex)
            {
                log.Error("Unable to add an existing chat. Prob due to the agent not let being stored in the map due to timings", ex);
            }
        }
    }
}