﻿//////////////////////////////////////////////////////////////////////////////
// JDecoder.cs
//
// Copyright 2015 Avaya Inc. All Rights Reserved.
//
// Usage of this source is bound to the terms described in
// Licenses/License.txt
//
// Avaya - Confidential & Proprietary. Use pursuant to your signed agreement or
// Avaya Policy
//////////////////////////////////////////////////////////////////////////////

using EncoderDecoder.Notifications;
using log4net;
using Newtonsoft.Json.Linq;
using System;
using System.Globalization;

namespace EncoderDecoder
{
    /// <summary>
    /// Used to deserialize notifications
    /// </summary>
    public class JDecoder
    {
        private static CultureInfo region = CultureInfo.CurrentCulture;

        private static readonly ILog Log = LogManager.GetLogger(System.Reflection.MethodBase.GetCurrentMethod().DeclaringType);

        /// <summary>
        /// Decodes incoming notifications and builds an object
        /// </summary>
        /// <param name="json"></param>
        /// <returns>IBaseNotification</returns>
        public IBaseNotification Decode(String json)
        {
            
            try
            {
                dynamic parsedJson = JValue.Parse(json);
                String notificationType = parsedJson.type;

                if (notificationType.ToUpper().Equals("NOTIFICATION"))
                {
                    return ProcessNotification(json, parsedJson);
                }
                else if (notificationType.ToUpper().Equals("ERROR"))
                {
                    return ProcessError(json);
                }
                else if (notificationType.ToUpper().Equals("ACKNOWLEDGEMENT"))
                {
                    return ProcessAcknowledgement(json);
                }
                
            }
            catch (Newtonsoft.Json.JsonReaderException)
            {
                Log.Error("Could not parse json");
                return null;
            }

            Log.Error("Websocket message is not of type Notification/Error/Acknowledgemnet - " + json);
            return null;
        }

        private IBaseNotification ProcessNotification(string json, dynamic parsedJson)
        {
            String notificationMethod = parsedJson.body.method;
            if (!string.IsNullOrEmpty(notificationMethod))
            {
                if (!notificationMethod.Equals("agentData"))
                {
                    Log.Info("Found type to decode - " + notificationMethod);
                }

                IBaseNotification notification = FindType(notificationMethod);
                return notification.Decode(json);
            }
            else
            {
                Log.Error("No method is json");
                return null;
            }
        }

        private IBaseNotification ProcessError(string json)
        {
            Log.Error("Error msg received");
            ErrorNotification error = new ErrorNotification();
            return error.Decode(json);
        }

        private IBaseNotification ProcessAcknowledgement(string json)
        {
            Log.Info("Ack msg received");
            AcknowledgementNotification ack = new AcknowledgementNotification();
            return ack.Decode(json);
        }


        /// <summary>
        /// Finds the correct subtype of IBaseNotification
        /// </summary>
        /// <param name="methodType"></param>
        /// <returns>Correct IBaseNotification object</returns>
        private IBaseNotification FindType(String methodType)
        {
            String notificationType = methodType.ToUpper();
            Notification n = new Notification();

            return n.GetNotificationType(notificationType);
        }

        /// <summary>
        /// Converts from unixtime(01-01-1970) to a DateTime object string representation
        /// </summary>
        /// <param name="unixTime"></param>
        /// <returns>DateTime</returns>
        public static String ConvertFromUnixTime(long unixTime)
        {

            DateTime dtDateTime = new DateTime(1970, 1, 1, 0, 0, 0, 0, DateTimeKind.Utc);
            dtDateTime = dtDateTime.AddMilliseconds(unixTime).ToLocalTime();

            return dtDateTime.ToLocalTime().ToString("HH:mm:ss", region);
        }

        /// <summary>
        /// Converts from unixtime(01-01-1970) to a DateTime object string representation with the date
        /// </summary>
        /// <param name="unixTime"></param>
        /// <returns>DateTime</returns>
        public static String ConvertFromUnixTimeWithDate(long unixTime)
        {
            DateTime dtDateTime = new DateTime(1970, 1, 1, 0, 0, 0, 0, DateTimeKind.Utc);
            dtDateTime = dtDateTime.AddMilliseconds(unixTime).ToLocalTime();

            return dtDateTime.ToLocalTime().ToString("dd-MMM-yyyy HH:mm:ss", region);

        }
    }
}
