﻿using System;
using System.Collections;
using System.Collections.Generic;
using System.ComponentModel;
using System.Data;
using System.Diagnostics;
using System.Drawing;
using System.IO;
using System.Linq;
using System.Net;
using System.Net.Security;
using System.Reflection;
using System.Security.Cryptography.X509Certificates;
using System.Text;
using System.Xml;
using System.Xml.Serialization;
using System.Threading.Tasks;
using System.Windows.Forms;

namespace ResourceMonitor
{
    public partial class MainForm : Form
    {
        public static MainForm Instance = null;

        private String[] fixedUrlList = null;
        private Dictionary<String, DateTime> modificationDates = new Dictionary<String, DateTime>();
        private Dictionary<String, String> etagList = new Dictionary<String, String>();
        public static Dictionary<String, String[]> enumerationValues = new Dictionary<String, String[]>();
        public static Dictionary<String, String> enumerationProperties = new Dictionary<String, String>();

        private bool copyResponseToFile = false;

        public MainForm()
        {
            InitializeComponent();
            Instance = this;

            // Just in case...
            Control.CheckForIllegalCrossThreadCalls = false;

            // Trigger events to set default states
            radioButton1_CheckedChanged(null, null);
            chkIfModified_CheckedChanged(null, null);
            tbPassword_TextChanged(null, null);

            // Set up our list of supported object URLs
            fixedUrlList = new String[]
            {
                "alarms",
                "alarms/config",
                "alarms/link",
                "alarms/service",
                "alarms/trunk",
                "alarms/qos",
                "alarms/tls",
                "alarms/srtp",
                "directory",
                "directory/entries",
                "directory/sources",
                "expansions",
                "extensions",
                "extensions/analog",
                "extensions/digital",
                "extensions/h323",
                "extensions/sip",
                "extensions/admm",
                "extensions/voip",
                "extensions/virtual",
                "extensions/unregistered",
                "extensions/simultaneous",
                "licenses",
                "locations",
                "slots",
                "switch",
                "system",
                "system/resources",
                "system/holdmusic",
                "time",
                "trunks",
                "trunks/analog",
                "trunks/analog/utilization",
                "trunks/digital",
                "trunks/digital/utilization",
                "trunks/ipoffice",
                "trunks/ipoffice/utilization",
                "trunks/sip",
                "trunks/sip/utilization",
                "trunks/voip",
                "trunks/voip/utilization",
                "voicemail",
                "voicemail/mailboxes"
            };
            cbURL.Items.AddRange(fixedUrlList);
            cbURL.SelectedIndex = 0;

            // Set our icon from bitmap resource
            Bitmap bm = Properties.Resources.icon;
            this.Icon = Icon.FromHandle(bm.GetHicon());

            // See if we can load enumeration names from co-located schema doc
            if (File.Exists("SSIReference.xsd"))
            {
                FileStream fs = File.OpenRead("SSIReference.xsd");
                StreamReader sr = new StreamReader(fs);
                try
                {
                    String line = sr.ReadLine();
                    String typeName = String.Empty;
                    while (!sr.EndOfStream)
                    {
                        if (line.Contains("<xs:element type=") && line.Contains("Enumeration"))
                        {
                            String enumName = line.Substring(line.IndexOf('"'));
                            enumName = enumName.Substring(1, enumName.LastIndexOf('"'));
                            String[] parts = enumName.Split(' ');
                            parts[0] = parts[0].Trim('"');
                            parts[1] = parts[1].Substring(6).Trim('"');

                            // Construct full field name from type and element names
                            String fieldName = typeName + "::" + parts[1];
                            if (!enumerationProperties.ContainsKey(fieldName))
                            {
                                enumerationProperties.Add(fieldName, parts[0]);
                            }
                        }
                        else if (line.Contains("<xs:complexType"))
                        {
                            // Store current object name
                            typeName = line.Substring(line.IndexOf('"') + 1);
                            typeName = typeName.Substring(0, typeName.LastIndexOf('"'));
                        }
                        else if (line.Contains("<xs:simpleType") && line.Contains("Enumeration"))
                        {
                            String enumName = line.Substring(line.IndexOf('"') + 1);
                            enumName = enumName.Substring(0, enumName.LastIndexOf('"'));
                            line = sr.ReadLine();
                            if (line.Contains("<xs:annotation>"))
                            {
                                line = sr.ReadLine();
                                line = sr.ReadLine();
                                List<String> values = new List<String>();
                                while (!line.Contains("</xs:documentation>"))
                                {
                                    String value = line.TrimStart();
                                    value = value.Substring(0, value.IndexOf('='));
                                    values.Add(value);
                                    line = sr.ReadLine();
                                }
                                enumerationValues.Add(enumName, values.ToArray());
                            }
                        }
                        line = sr.ReadLine();
                    }
                }
                catch (Exception)
                {
                }
                finally
                {
                    sr.Close();
                    fs.Close();
                }
            }
        }

        public String Authorization
        {
            get
            {
                return (tbUsername.Text.Trim() + ":" + tbPassword.Text.Trim());
            }
        }

        public String SwitchAddress
        {
            get
            {
                return tbAddress.Text;
            }
        }

        public String SwitchPort
        {
            get
            {
                return nudPort.Value.ToString();
            }
        }

        private void btnLoad_Click(object sender, EventArgs e)
        {
            bool selectedNodeOnly = false;
            String cid = "sw";
            if (radioButton1.Checked)
            {
                if (treeView1.Nodes.Count > 0 && treeView1.SelectedNode != null)
                {
                    TreeNode tn = treeView1.SelectedNode;
                    if (tn.Text.Equals("Switch (sw)") || tn.Text.Contains("sw/"))
                    {
                        // Valid component id
                        SSIObject obj = tn.Tag as SSIObject;
                        if (obj != null)
                        {
                            selectedNodeOnly = true;
                            cid = obj.CID;
                        }
                        else
                        {
                            return;
                        }
                    }
                }

                if (!selectedNodeOnly && cid == "sw")
                {
                    if (tbComponentId.Text != String.Empty)
                    {
                        cid = tbComponentId.Text;
                    }
                }
            }
            else
            {
                cid = cbURL.Text;
            }

            SSIObject data = RetrieveData(cid, chkIfModified.Checked, chkGetDeltas.Checked, (int)nudDepth.Value);

            if (data is SSIObject)
            {
                SSIObject obj = data as SSIObject;
                TreeNode node = null;

                // Attempt to find matching node current in tree
                if (selectedNodeOnly)
                {
                    node = treeView1.SelectedNode;
                }
                else if (treeView1.Nodes.Count > 0 && treeView1.Nodes[0] != null)
                {
                    if (!String.IsNullOrEmpty(obj.CID))
                    {
                        node = FindNodeByComponentId(treeView1.Nodes[0], obj.CID);
                    }
                }

                if (node != null)
                {
                    if (chkGetDeltas.Checked)
                    {
                        UnsetModifiedNodes(node);
                        bool expanded = node.IsExpanded;
                        if (!expanded)
                        {
                            node.Expand();
                        }
                        if (!MergeData(node, obj))
                        {
                            if (!expanded)
                            {
                                node.Collapse();
                            }
                        }
                    }
                    else
                    {
                        node.Tag = obj;
                        node.Nodes.Clear();
                        GenerateNodesForObject(obj, node);
                        node.Expand();
                    }
                    return;
                }

                // Default is to clear current tree and load whatever we've received
                treeView1.Nodes.Clear();

                bool hasData = true;
                if (radioButton2.Checked)
                {
                    // Check if we have received empty list
                    PropertyInfo[] objInfo = obj.GetType().GetProperties();
                    if (objInfo.Length == 2)
                    {
                        foreach (PropertyInfo pi in objInfo)
                        {
                            if (pi.PropertyType.IsArray)
                            {
                                Object[] ar = (Object[])pi.GetValue(obj);
                                if (ar == null)
                                {
                                    hasData = false;
                                    lbStatus.Text = "No data received for request " + cid;
                                    break;
                                }
                            }
                        }
                    }
                }

                if (hasData)
                {
                    String text = String.IsNullOrEmpty(obj.CID) ? cid : obj.CID;
                    TreeNode tn = treeView1.Nodes.Add(obj.GetType().ToString() + " (" + text + ")");
                    tn.Tag = obj;
                    GenerateNodesForObject(obj, tn);
                    tn.Expand();
                }
            }
        }

        public SSIObject RetrieveData(String cid, bool ifModified, bool getDeltas, int depth)
        {
            ServicePointManager.ServerCertificateValidationCallback += new System.Net.Security.RemoteCertificateValidationCallback(ValidateServerCertificate);

            String sdepth = depth == nudDepth.Maximum ? "all" : depth.ToString();
            Uri uri = new Uri("https://" + tbAddress.Text + ":" + nudPort.Value + "/ws/ssi/" + cid + (depth > 0 ? "?depth=" + sdepth + (getDeltas ? "&deltas=true" : "") : "" + (getDeltas ? "?deltas=true" : "")));
            HttpWebRequest ipoRequest = CreateWebRequest(uri, "GET", Authorization);
            if (ifModified)
            {
                DateTime lastMod;
                if (modificationDates.TryGetValue(cid, out lastMod))
                {
                    ipoRequest.IfModifiedSince = lastMod;
                }
                String etag = "";
                if (etagList.TryGetValue(cid, out etag))
                {
                    ipoRequest.Headers.Add("If-None-Match", etag);
                }
            }

            try
            {
                lbStatus.Text = "";

                HttpWebResponse ipoResponse = ipoRequest.GetResponse() as HttpWebResponse;
                if (ipoRequest.HaveResponse && ipoResponse != null)
                {
                    if (!modificationDates.ContainsKey(cid))
                    {
                        modificationDates.Add(cid, ipoResponse.LastModified);
                    }
                    else
                    {
                        modificationDates[cid] = ipoResponse.LastModified;
                    }
                    String etag = ipoResponse.GetResponseHeader("ETag");
                    if (!String.IsNullOrEmpty(etag))
                    {
                        if (!etagList.ContainsKey(cid))
                        {
                            etagList.Add(cid, etag);
                        }
                        else
                        {
                            etagList[cid] = etag;
                        }
                    }
                    if (ipoResponse.StatusCode == HttpStatusCode.NotModified)
                    {
                        return null;
                    }

                    using (MemoryStream stream = new MemoryStream())
                    {
                        using (Stream input = ipoResponse.GetResponseStream())
                        {
                            byte[] buffer = new byte[8192];
                            int bytesRead;
                            while ((bytesRead = input.Read(buffer, 0, buffer.Length)) > 0)
                            {
                                stream.Write(buffer, 0, bytesRead);
                            }
                        }
                        lbStatus.Text = "Received " + stream.Length + " bytes of data";
                        stream.Position = 0;

                        if (copyResponseToFile)
                        {
                            FileStream fs = File.Create("ssi.xml");
                            stream.CopyTo(fs);
                            fs.Close();
                            stream.Position = 0;
                        }

                        // Load the text in the xml tab
                        StreamReader reader = new StreamReader(stream, Encoding.UTF8);
                        richTextBox1.Text = reader.ReadToEnd();
                        stream.Position = 0;

                        using (XmlTextReader xmlReader = new XmlTextReader(stream))
                        {
                            XmlRootAttribute xRoot = new XmlRootAttribute();
                            xRoot.ElementName = "response";
                            xRoot.IsNullable = true;
                            XmlSerializer serializer = new XmlSerializer(typeof(Response), xRoot);
                            while (xmlReader.Read())
                            {
                                switch (xmlReader.NodeType)
                                {
                                    case XmlNodeType.Element:
                                        String name = xmlReader.Name.ToString();
                                        if (name == "response")
                                        {
                                            Response r = (Response)serializer.Deserialize(xmlReader);
                                            if (r != null && r.data != null)
                                            {
                                                if (r.data.Item is Error)
                                                {
                                                    Error err = (Error)r.data.Item;
                                                    lbStatus.Text = "Received data response code " + err.error_code + " " + err.error_desc;
                                                    return null;
                                                }
                                                else if (r.data.Item is SSIObject)
                                                {
                                                    return (r.data.Item as SSIObject);
                                                }
                                                else
                                                {
                                                    lbStatus.Text = "No data received for request " + cid;
                                                    return null;
                                                }
                                            }
                                        }
                                        break;
                                }
                            }
                            xmlReader.Close();
                        }
                    }
                    ipoResponse.Close();
                }
            }
            catch (Exception ex)
            {
                if (ex is WebException)
                {
                    WebException wex = ex as WebException;
                    if (wex.Response is HttpWebResponse)
                    {
                        HttpWebResponse ipoResponse = wex.Response as HttpWebResponse;
                        if (ipoResponse.StatusCode == HttpStatusCode.NotModified)
                        {
                            if (!modificationDates.ContainsKey(cid))
                            {
                                modificationDates.Add(cid, ipoResponse.LastModified);
                            }
                            else
                            {
                                modificationDates[cid] = ipoResponse.LastModified;
                            }
                            String etag = ipoResponse.GetResponseHeader("ETag");
                            if (!String.IsNullOrEmpty(etag))
                            {
                                if (!etagList.ContainsKey(cid))
                                {
                                    etagList.Add(cid, etag);
                                }
                                else
                                {
                                    etagList[cid] = etag;
                                }
                            }
                            lbStatus.Text = "Received response " + (int)ipoResponse.StatusCode + " " + ipoResponse.StatusDescription;
                        }
                        else
                        {
                            MessageBox.Show(Application.OpenForms[0], "Error response: " + (int)ipoResponse.StatusCode + " " + ipoResponse.StatusDescription, "Error", MessageBoxButtons.OK, MessageBoxIcon.Error);
                        }

                    }
                    else
                    {
                        if (wex.Response == null)
                        {
                            MessageBox.Show(Application.OpenForms[0], "No response from server at address " + tbAddress.Text, "Error", MessageBoxButtons.OK, MessageBoxIcon.Error);
                        }
                    }
                }
                else
                {
                    MessageBox.Show(Application.OpenForms[0], ex.Message, "Error", MessageBoxButtons.OK, MessageBoxIcon.Error);
                }
                System.Diagnostics.Trace.TraceError(ex.Message);
            }
            ServicePointManager.ServerCertificateValidationCallback -= new System.Net.Security.RemoteCertificateValidationCallback(ValidateServerCertificate);

            return null;
        }

        public bool ValidateServerCertificate(
                          object sender,
                          X509Certificate certificate,
                          X509Chain chain,
                          SslPolicyErrors sslPolicyErrors)
        {
            return true;
        }

        public static HttpWebRequest CreateWebRequest(Uri address, String method, String authorization)
        {
            HttpWebRequest request = WebRequest.Create(address) as HttpWebRequest;
            request.Method = method;
            request.Credentials = CredentialCache.DefaultCredentials;
            WebRequest.DefaultWebProxy = null;
            request.Proxy = WebRequest.DefaultWebProxy;
            WebHeaderCollection header = new WebHeaderCollection();
            header.Add("Authorization", "Basic " + Convert.ToBase64String(new ASCIIEncoding().GetBytes(authorization)));
            request.Headers = header;
            request.UserAgent = "ServiceMonitorTester";
            return request;
        }

        private SSIObject FindChild(SSIObject obj, String cid)
        {
            if (obj.CID == cid)
            {
                return obj;
            }
            PropertyInfo[] objInfo = obj.GetType().GetProperties();
            foreach (PropertyInfo pi in objInfo)
            {
                if (pi.PropertyType.IsArray)
                {
                    Object[] ar = (Object[])pi.GetValue(obj);
                    if (ar != null)
                    {
                        foreach (Object item in ar)
                        {
                            PropertyInfo[] itemInfo = item.GetType().GetProperties();
                            foreach (PropertyInfo p in itemInfo)
                            {
                                if (p.Name == "CID")
                                {
                                    String id = p.GetValue(item).ToString();
                                    if (id == cid)
                                    {
                                        return (SSIObject)item;
                                    }
                                    else if (cid.StartsWith(id))
                                    {
                                        return FindChild((SSIObject)item, cid);
                                    }
                                }
                            }
                        }
                    }
                }
                else if (pi.PropertyType.IsClass)
                {
                    Object item = (Object)pi.GetValue(obj);
                    if (item != null)
                    {
                        PropertyInfo[] itemInfo = item.GetType().GetProperties();
                        foreach (PropertyInfo p in itemInfo)
                        {
                            if (p.Name == "CID")
                            {
                                String id = p.GetValue(item).ToString();
                                if (id == cid)
                                {
                                    return (SSIObject)item;
                                }
                                else if (cid.StartsWith(id))
                                {
                                    return FindChild((SSIObject)item, cid);
                                }
                            }
                        }
                    }
                }
            }
            return null;
        }

        private TreeNode FindByType(TreeNode node, String type)
        {
            if (node.Tag is SSIObject)
            {
                SSIObject obj = node.Tag as SSIObject;
                if (obj.GetType().Name == type)
                {
                    return node;
                }

                foreach (TreeNode tn in node.Nodes)
                {
                    if (tn.Tag is SSIObject)
                    {
                        SSIObject nodeObj = tn.Tag as SSIObject;
                        if (nodeObj.GetType().Name == type)
                        {
                            return tn;
                        }
                        else
                        {
                            TreeNode childNode = FindByType(tn, type);
                            if (childNode != null)
                            {
                                return childNode;
                            }
                        }
                    }
                }
            }
            return null;
        }

        private void GenerateNodesForObject(Object obj, TreeNode tn)
        {
            DateTime lastMod = DateTime.Now;
            String etag = "";
            SSIObject ssiObj = obj as SSIObject;
            if (ssiObj != null && !String.IsNullOrEmpty(ssiObj.CID))
            {
                if (modificationDates.ContainsKey(ssiObj.CID))
                {
                    modificationDates.TryGetValue(ssiObj.CID, out lastMod);
                }
                if (etagList.ContainsKey(ssiObj.CID))
                {
                    etagList.TryGetValue(ssiObj.CID, out etag);
                }
            }

            PropertyInfo[] objInfo = obj.GetType().GetProperties();
            foreach (PropertyInfo pi in objInfo)
            {
                if (pi.PropertyType.IsArray)
                {
                    Object[] ar = (Object[])pi.GetValue(obj);
                    if (ar != null)
                    {
                        foreach (Object item in ar)
                        {
                            PropertyInfo[] itemInfo = item.GetType().GetProperties();
                            foreach (PropertyInfo p in itemInfo)
                            {
                                if (p.Name == "CID")
                                {
                                    String text = p.GetValue(item).ToString();
                                    if (!modificationDates.ContainsKey(text))
                                    {
                                        modificationDates.Add(text, lastMod);
                                    }
                                    if (!etagList.ContainsKey(text))
                                    {
                                        etagList.Add(text, etag);
                                    }
                                    if (!pi.Name.StartsWith("Field"))
                                    {
                                        text = pi.Name + " (" + text + ")";
                                    }
                                    TreeNode node = new TreeNode(text);
                                    node.Tag = item;

                                    GenerateNodesForObject(item, node);
                                    tn.Nodes.Add(node);
                                    break;
                                }
                            }
                        }
                    }
                }
                else if (pi.PropertyType.IsClass)
                {
                    Object item = (Object)pi.GetValue(obj);
                    if (item != null)
                    {
                        bool isObjectNode = false;
                        PropertyInfo[] itemInfo = item.GetType().GetProperties();
                        foreach (PropertyInfo p in itemInfo)
                        {
                            if (p.Name == "CID" || item is SSIGenericObject)
                            {
                                isObjectNode = true;
                                if (p.Name == "CID")
                                {
                                    String cid = p.GetValue(item).ToString();
                                    if (!modificationDates.ContainsKey(cid))
                                    {
                                        modificationDates.Add(cid, lastMod);
                                    }
                                    if (!etagList.ContainsKey(cid))
                                    {
                                        etagList.Add(cid, etag); 
                                    }
                                }

                                String text = p.GetValue(item).ToString() + ": " + item.GetType().Name;
                                if (!pi.Name.StartsWith("Field"))
                                {
                                    text = pi.Name + (p.Name == "CID" ? " (" + p.GetValue(item).ToString() + ")" : "");
                                }
                                if (obj is SSIGenericObject)
                                {
                                    text = item.GetType() + (p.Name == "CID" ? " (" + p.GetValue(item).ToString() + ")" : "");
                                }
                                TreeNode node = new TreeNode(text);
                                node.Tag = item;
                                node.Name = p.GetValue(item).ToString();

                                GenerateNodesForObject(item, node);
                                tn.Nodes.Add(node);
                                break;
                            }
                        }

                        if (!isObjectNode && pi.Name != "CID")
                        {
                            TreeNode node = new TreeNode(pi.Name + ": " + Convert.ToString(item));
                            tn.Nodes.Add(node);
                        }
                    }
                }
                else
                {
                    if (pi.PropertyType.Name == "Int32")
                    {
                        // Potential enumeration value we can translate
                        try
                        {
                            String fieldName = obj.GetType().Name + "::" + pi.Name;
                            if (enumerationProperties.ContainsKey(fieldName))
                            {
                                String enumType = enumerationProperties[fieldName];
                                if (enumerationValues.ContainsKey(enumType))
                                {
                                    Int32 value = Convert.ToInt32(pi.GetValue(obj));
                                    if (value > 0)
                                    {
                                        value--;
                                        String[] values = enumerationValues[enumType];
                                        if (values != null && values.Length > value)
                                        {
                                            tn.Nodes.Add(new TreeNode(pi.Name + ": " + values[value]));
                                            continue;
                                        }
                                    }
                                    else
                                    {
                                        // Invalid value
                                        continue;
                                    }
                                }
                            }
                        }
                        catch (Exception)
                        {
                        }
                    }
                    else if (pi.PropertyType.Name == "Int64")
                    {
                        // Exclude uninitialised numeric fields
                        if ((Int64)pi.GetValue(obj) == -1)
                        {
                            continue;
                        }
                    }
                    else if (pi.PropertyType.Name == "Boolean")
                    {
                        // Exclude uninitialised boolean fields
                        if (!(Boolean)pi.GetValue(obj))
                        {
                            continue;
                        }
                    }

                    tn.Nodes.Add(new TreeNode(pi.Name + ": " + Convert.ToString(pi.GetValue(obj))));
                }
            }
        }

        private TreeNode FindNodeByComponentId(TreeNode node, String componentId)
        {
            if (node.Tag is SSIObject)
            {
                String cid = ((SSIObject)node.Tag).CID;
                if (cid != null && cid.Equals(componentId))
                {
                    return node;
                }
            }

            foreach (TreeNode tn in node.Nodes)
            {
                TreeNode foundNode = FindNodeByComponentId(tn, componentId);
                if (foundNode != null)
                {
                    return foundNode;
                }
            }
            return null;
        }

        private TreeNode FindNodeByProperty(TreeNode node, String property)
        {
            foreach (TreeNode tn in node.Nodes)
            {
                if (tn.Text.StartsWith(property + ":"))
                {
                     return tn;
                }
            }
            return null;
        }

        private void UnsetModifiedNodes(TreeNode node)
        {
            foreach (TreeNode tn in node.Nodes)
            {
                tn.ForeColor = Color.Black;
                UnsetModifiedNodes(tn);
            }
        }

        private bool MergeData(TreeNode node, SSIObject obj)
        {
            bool result = false;
            SSIObject currentObj = node.Tag as SSIObject;
            if (currentObj != null)
            {
                if (currentObj.GetType().Equals(currentObj.GetType()))
                {
                    // Copy any altered data field values and flag as changed
                    PropertyInfo[] objInfo = obj.GetType().GetProperties();
                    foreach (PropertyInfo pi in objInfo)
                    {
                        PropertyInfo currentProperty = currentObj.GetType().GetProperty(pi.Name);
                        if (pi.PropertyType.IsArray)
                        {
                            Object[] ar = (Object[])pi.GetValue(obj);
                            if (ar != null)
                            {
                                List<Object> additions = new List<Object>();
                                foreach (Object item in ar)
                                {
                                    SSIObject ssiObj = item as SSIObject;
                                    if (ssiObj != null)
                                    {
                                        TreeNode child = FindNodeByComponentId(node, ssiObj.CID);
                                        if (child != null)
                                        {
                                            bool expanded = child.IsExpanded;
                                            if (!expanded)
                                            {
                                                child.Expand();
                                            }
                                            if (MergeData(child, ssiObj))
                                            {
                                                result = true;
                                            }
                                            else
                                            {
                                                if (!expanded)
                                                {
                                                    child.Collapse();
                                                }
                                            }
                                        }
                                        else
                                        {
                                            PropertyInfo[] itemInfo = item.GetType().GetProperties();
                                            foreach (PropertyInfo p in itemInfo)
                                            {
                                                if (p.Name == "CID")
                                                {
                                                    // Append item to additions list and generate new nodes
                                                    additions.Add(item);

                                                    String text = p.GetValue(item).ToString();
                                                    if (!pi.Name.StartsWith("Field"))
                                                    {
                                                        text = pi.Name + " (" + p.GetValue(item).ToString() + ")";
                                                    }
                                                    TreeNode tn = new TreeNode(text);
                                                    tn.Tag = item;
                                                    tn.ForeColor = Color.Red;

                                                    GenerateNodesForObject(item, tn);
                                                    node.Nodes.Add(tn);
                                                    result = true;
                                                }
                                            }
                                        }
                                    }
                                }

                                if (additions.Count > 0)
                                {
                                    // Update the original array field with additions
                                    var newArray = Array.CreateInstance(pi.PropertyType.GetElementType(), 0);
                                    int i = 0;
                                    Object[] currentArray = (Object[])currentProperty.GetValue(currentObj);
                                    if (currentArray != null)
                                    {
                                        newArray = Array.CreateInstance(pi.PropertyType.GetElementType(), currentArray.Length + additions.Count);
                                        for (; i < currentArray.Length; i++)
                                        {
                                            newArray.SetValue(currentArray[i], i);
                                        }
                                    }
                                    else
                                    {
                                        newArray = Array.CreateInstance(pi.PropertyType.GetElementType(), additions.Count);
                                    }
                                    for (int j = 0; j < additions.Count; i++, j++)
                                    {
                                        newArray.SetValue(additions[j], i);
                                    }

                                    if (currentProperty != null && currentProperty.CanWrite)
                                    {
                                        currentProperty.SetValue(currentObj, newArray);
                                    }
                                }
                            }
                        }
                        else if (pi.PropertyType.IsClass)
                        {
                            Object item = (Object)pi.GetValue(obj);
                            if (item is SSIObject)
                            {
                                TreeNode child = FindNodeByComponentId(node, ((SSIObject)item).CID);
                                if (child != null)
                                {
                                    bool expanded = child.IsExpanded;
                                    if (!expanded)
                                    {
                                        child.Expand();
                                    }
                                    if (MergeData(child, item as SSIObject))
                                    {
                                        result = true;
                                    }
                                    else
                                    {
                                        if (!expanded)
                                        {
                                            child.Collapse();
                                        }
                                    }
                                }
                                else
                                {
                                    // Set the original object's property and generate new node(s)
                                    if (currentProperty != null && currentProperty.CanWrite)
                                    {
                                        currentProperty.SetValue(currentObj, item);
                                    }
                                    String text = item.GetType() + " (" + ((SSIObject)item).CID + ")";
                                    TreeNode tn = new TreeNode(text);
                                    tn.Tag = item;
                                    tn.Name = ((SSIObject)item).CID;
                                    tn.ForeColor = Color.Red;

                                    GenerateNodesForObject(item, tn);
                                    node.Nodes.Add(tn);
                                    result = true;
                                }
                            }
                            else if (pi.Name != "CID")
                            {
                                String value = pi.Name + ": " + Convert.ToString(item);
                                TreeNode field = FindNodeByProperty(node, pi.Name);
                                if (field != null && !field.Text.Equals(value))
                                {
                                    // Set both the original object's property and new node text
                                    if (currentProperty != null && currentProperty.CanWrite)
                                    {
                                        currentProperty.SetValue(currentObj, item);
                                    }
                                    field.Text = value;
                                    field.ForeColor = Color.Red;
                                    result = true;
                                }
                            }
                        }
                        else
                        {
                            String newText = "";
                            if (pi.PropertyType.Name == "Boolean")
                            {
                                Boolean bvalue = (Boolean)pi.GetValue(obj);
                                newText = pi.Name + ": " + Convert.ToString(true);
                                TreeNode field = FindNodeByProperty(node, pi.Name);
                                if (bvalue && field == null)
                                {
                                    // Not present so set the original object's property and add new node
                                    if (currentProperty != null && currentProperty.CanWrite)
                                    {
                                        currentProperty.SetValue(currentObj, bvalue);
                                    }
                                    field = new TreeNode(newText);
                                    field.Text = newText;
                                    field.ForeColor = Color.Red;
                                    node.Nodes.Add(field);
                                    result = true;
                                }
                                else if (!bvalue && field != null)
                                {
                                    // Present so set the original object's property and remove node
                                    if (currentProperty != null && currentProperty.CanWrite)
                                    {
                                        currentProperty.SetValue(currentObj, bvalue);
                                    }
                                    node.Nodes.Remove(field);
                                    result = true;
                                }
                                continue;
                            }
                            else if (pi.PropertyType.Name == "Int64")
                            {
                                Int64 ivalue = (Int64)pi.GetValue(obj);
                                newText = pi.Name + ": " + Convert.ToString(ivalue);
                                TreeNode field = FindNodeByProperty(node, pi.Name);
                                if (ivalue >= 0 && field == null)
                                {
                                    // Not present so set the original object's property and add new node
                                    if (currentProperty != null && currentProperty.CanWrite)
                                    {
                                        currentProperty.SetValue(currentObj, ivalue);
                                    }
                                    field = new TreeNode(newText);
                                    field.Text = newText;
                                    field.ForeColor = Color.Red;
                                    node.Nodes.Add(field);
                                    result = true;
                                    continue;
                                }
                                else if (ivalue == -1 && field != null)
                                {
                                    // Present so set the original object's property and remove node
                                    if (currentProperty != null && currentProperty.CanWrite)
                                    {
                                        currentProperty.SetValue(currentObj, ivalue);
                                    }
                                    node.Nodes.Remove(field);
                                    result = true;
                                    continue;
                                }
                            }
                            else if (pi.PropertyType.Name == "Int32")
                            {
                                try
                                {
                                    String fieldName = obj.GetType().Name + "::" + pi.Name;
                                    if (enumerationProperties.ContainsKey(fieldName))
                                    {
                                        String enumType = enumerationProperties[fieldName];
                                        if (enumerationValues.ContainsKey(enumType))
                                        {
                                            Int32 value = Convert.ToInt32(pi.GetValue(obj));
                                            if (value > 0)
                                            {
                                                value--;
                                                String[] values = enumerationValues[enumType];
                                                if (values != null && values.Length > value)
                                                {
                                                    newText = pi.Name + ": " + values[value];
                                                }
                                            }
                                            else
                                            {
                                                continue;
                                            }
                                        }
                                    }
                                    else
                                    {
                                        newText = pi.Name + ": " + Convert.ToString(pi.GetValue(obj));
                                    }
                                }
                                catch (Exception)
                                {
                                }
                            }
                            else
                            {
                                newText = pi.Name + ": " + Convert.ToString(pi.GetValue(obj));
                            }
 
                            if (newText != "")
                            {
                                TreeNode field = FindNodeByProperty(node, pi.Name);
                                if (field != null && !field.Text.Equals(newText))
                                {
                                    // Set the original object's property and new value text
                                    if (currentProperty != null && currentProperty.CanWrite)
                                    {
                                        currentProperty.SetValue(currentObj, pi.GetValue(obj));
                                    }
                                    field.Text = newText;
                                    field.ForeColor = Color.Red;
                                    result = true;
                                }
                            }
                        }
                    }
                }
            }
            return result;
        }

        private void tbCustomURL_TextChanged(object sender, EventArgs e)
        {
            if (treeView1.SelectedNode != null)
            {
                treeView1.SelectedNode = null;
            }
        }

        private void tbPassword_TextChanged(object sender, EventArgs e)
        {
            btnLoad.Enabled = (tbPassword.Text != String.Empty);
        }

        private void treeView1_AfterSelect(object sender, TreeViewEventArgs e)
        {
            if (treeView1.SelectedNode != null)
            {
                if (treeView1.SelectedNode.Tag is SSIObject)
                {
                    tbComponentId.TextChanged -= tbCustomURL_TextChanged;
                    if (((SSIObject)treeView1.SelectedNode.Tag).CID != null)
                    {
                        tbComponentId.Text = ((SSIObject)treeView1.SelectedNode.Tag).CID;
                    }
                    else
                    {
                        int index = treeView1.SelectedNode.Text.IndexOf('(');
                        if (index >= 0)
                        {
                            String url = treeView1.SelectedNode.Text.Substring(index + 1);
                            url = url.Trim(')');
                            tbComponentId.Text = url;
                        }
                    }
                    tbComponentId.TextChanged += tbCustomURL_TextChanged;
                }
            }
        }

        private void treeView1_NodeMouseDoubleClick(object sender, TreeNodeMouseClickEventArgs e)
        {
            // For references, attempt to locate the referenced object node
            if (e.Node != null)
            {
                String txt = e.Node.Text;
                String[] strings = txt.Split(' ');
                if (strings.Length == 2 && strings[1].StartsWith("sw/"))
                {
                    TreeNode node = FindNodeByComponentId(treeView1.Nodes[0], strings[1]);
                    if (node != null)
                    {
                        node.EnsureVisible();
                        node.ExpandAll();
                        treeView1.SelectedNode = node;
                    }
                }
            }
        }

        private void btnClear_Click(object sender, EventArgs e)
        {
            tbComponentId.Text = "";
            lbStatus.Text = "";
            treeView1.Nodes.Clear();
            richTextBox1.Clear();
            modificationDates.Clear();
            etagList.Clear();
        }

        private void radioButton1_CheckedChanged(object sender, EventArgs e)
        {
            tableLayoutPanel2.SuspendLayout();
            lbComponentId.Visible = radioButton1.Checked;
            tbComponentId.Visible = radioButton1.Checked;
            lbURL.Visible = !radioButton1.Checked;
            cbURL.Visible = !radioButton1.Checked;
            tableLayoutPanel2.Controls.Remove(btnLoad);
            tableLayoutPanel2.Controls.Remove(btnClear);
            if (radioButton1.Checked)
            {
                tableLayoutPanel2.Controls.Add(btnLoad, 2, 3);
                tableLayoutPanel2.Controls.Add(btnClear, 3, 3);
            }
            else
            {
                tableLayoutPanel2.Controls.Add(btnLoad, 2, 4);
                tableLayoutPanel2.Controls.Add(btnClear, 3, 4);
            }
            tableLayoutPanel2.ResumeLayout();
        }

        private void chkIfModified_CheckedChanged(object sender, EventArgs e)
        {
            chkGetDeltas.Enabled = chkIfModified.Checked;
            if (!chkIfModified.Checked)
            {
                chkGetDeltas.Checked = false;
            }
        }

        private TrunkGroup currentTrunkGroup = null;

        private void treeView1_MouseUp(object sender, System.Windows.Forms.MouseEventArgs e)
        {
            if (e.Button == MouseButtons.Right)
            {
                Point p = new Point(e.X, e.Y);
                TreeNode node = treeView1.GetNodeAt(p);
                if (node != null)
                {
                    if (node.Tag is TrunkGroup)
                    {
                        currentTrunkGroup = node.Tag as TrunkGroup;
                        if (currentTrunkGroup.TrunkNumber > 0) // Exclude analog trunks
                        {
                            treeView1.ContextMenuStrip = contextMenuStrip1;
                            contextMenuStrip1.Show(treeView1, e.Location);
                            treeView1.ContextMenuStrip = null;
                        }
                        else
                        {
                            currentTrunkGroup = null;
                        }
                    }
                }
            }
        }

        private void trunkStatusToolStripMenuItem_Click(object sender, EventArgs e)
        {
            if (currentTrunkGroup != null)
            {
                using (TrunkStatus dlg = new TrunkStatus(currentTrunkGroup))
                {
                    dlg.ShowDialog(this);
                }
                currentTrunkGroup = null;
            }
        }

        public String ConvertToEnumerationValue(String fieldName, int value)
        {
            String result = value.ToString();
            if (enumerationProperties.ContainsKey(fieldName))
            {
                String enumType = enumerationProperties[fieldName];
                if (enumerationValues.ContainsKey(enumType))
                {
                    if (value > 0)
                    {
                        // We do not specify the 0 'invalid' value in the dictionary
                        value = value - 1;
                        String[] values = enumerationValues[enumType];
                        if (values != null && values.Length > value)
                        {
                            result = values[value];
                        }
                    }
                }
            }
            return result;
        }

        public long SwitchTimeToClockTime(String timeString)
        {
            try
            {
                String[] ar = timeString.Split(' ');
                if (ar.Length == 4)
                {
                    long lt = Convert.ToInt64(ar[3]);
                    return lt;
                }
            }
            catch
            {
            }
            return 0;
        }

        public DateTime SwitchTimeToDateTime(String timeString)
        {
            DateTime dt = new DateTime();
            try
            {
                String[] ar = timeString.Split(' ');
                if (ar.Length >= 2)
                {
                    dt = DateTime.ParseExact(ar[0] + " " + ar[1], "MM/dd/yyyy HH:mm:ss", System.Globalization.CultureInfo.InvariantCulture);
                }
            }
            catch
            {
            }
            return dt;
        }
    }
}
