﻿using System;
using System.Collections.Generic;
using System.Linq;
using System.Text;
using System.Threading.Tasks;
using System.ServiceModel.Web;
using System.ServiceModel;
using System.IO;
using System.Collections.ObjectModel;
using System.Threading;
using System.Xml;
using System.Net.Sockets;
using System.Net.Security;
using System.Net;
using System.Security.Authentication;
using OneXAdapterSample;
using System.Diagnostics;

namespace OneXOpenAPISample
{
    /// <summary>
    /// Interface for CSTA Services.
    /// </summary>
    [ServiceContract]
    public interface ICSTAService
    {
        [OperationContract]
        [WebInvoke
        (Method = "POST",
        RequestFormat = WebMessageFormat.Xml,
        UriTemplate = "/GetEvents/"
        )]
        void GetCallEvents(Stream request);
        void GetCallEvents(string request);
        bool Start(string serverUrl, string callbackUrl);
        bool Stop();
        ObservableCollection<object> Events { get; set; }
        string CallbackUrl { get; }
        bool IsStarted { get; }
    }

    public class WebSocketService : ICSTAService
    {
        Deserilizer deSel;
        string SessionId;
        bool IsUserLoggedIn;
        AutoResetEvent eventArrived = new AutoResetEvent(false);
        Thread eventQueueThread;
        Queue<string> eventQueue = new Queue<string>();
        private WebSocket webSocket;
        private string _callbackUrl = string.Empty;
        bool isConnected = false;
        bool LogoutOnHeartbeatFailure = false;

        public WebSocketService(string sessionId, bool isUserLoggedIn)
        {
            deSel = new Deserilizer();
            SessionId = sessionId;
            IsUserLoggedIn = isUserLoggedIn;
            _events = new ObservableCollection<object>();
        }

        private void ProcessEvents(object text)
        {
            bool forwardEvent = false;
            string eventstoLog = string.Empty;
            string eventName = string.Empty;

            try
            {
                XmlDocument doc = new XmlDocument();
                doc.LoadXml((string)text);

                if (doc.ChildNodes[1] != null && doc.ChildNodes[1].Attributes != null && doc.ChildNodes[1].Attributes.Count > 0)
                {
                    foreach (XmlAttribute attribute in doc.ChildNodes[1].Attributes)
                    {
                        if (attribute.Name == "clientSessionId")
                        {
                            string clientSessionId = attribute.Value;
                            if (!string.IsNullOrEmpty(clientSessionId) && SessionId == clientSessionId)
                            {
                                forwardEvent = true;
                                break;
                            }
                        }
                    }
                }

                if (forwardEvent)
                {
                    foreach (XmlNode node in doc.ChildNodes[1].ChildNodes)
                    {
                        int indexOfColon = 0;
                        if (node.Name.Contains(':'))
                        {
                            indexOfColon = node.Name.IndexOf(':');
                        }
                        eventName = (node.Name.Substring((indexOfColon + 1), node.Name.Length - (indexOfColon + 1)));

                        eventstoLog += eventName + ",";

                        Object eventObject = deSel.Deserlize(eventName, node.OuterXml);

                        if (eventObject != null)
                        {
                            _events.Add(eventObject);
                        }
                    }

                    Logger.LogInfo("ProcessEvents", "Events received : " + eventstoLog);
                }
                else
                {
                    Logger.LogWarning("ProcessEvents", "Events received for incorrect client session Id (Correct): " + SessionId);
                    try
                    {
                        if (doc.ChildNodes[1] != null && !string.IsNullOrEmpty(doc.ChildNodes[1].InnerText))
                        {
                            if (doc.ChildNodes[1].InnerText.Contains("INVALID_CLIENT_SESSION_ID"))
                            {
                                string invalid_session_id = string.Empty;
                                foreach (XmlNode node in doc.ChildNodes[1].ChildNodes)
                                {
                                    //Find INVALID_CLIENT_SESSION_ID.
                                    foreach (XmlNode nodeError in node.ChildNodes)
                                    {
                                        if (nodeError.Name == "errorValue")
                                        {
                                            invalid_session_id = nodeError.InnerText;
                                        }
                                    }
                                    Logger.LogWarning("ProcessEvents", "Events received for incorrect client session Id (Incorrect) : " + invalid_session_id);

                                    //Generate error event for INVALID_CLIENT_SESSION_ID.
                                    int indexOfColon = 0;
                                    if (node.Name.Contains(':'))
                                    {
                                        indexOfColon = node.Name.IndexOf(':');
                                    }
                                    eventName = (node.Name.Substring((indexOfColon + 1), node.Name.Length - (indexOfColon + 1)));

                                    eventstoLog += eventName + ",";

                                    Object eventObject = deSel.Deserlize(eventName, node.OuterXml);

                                    if (eventObject != null)
                                    {
                                        _events.Add(eventObject);
                                    }
                                }
                            }
                        }
                    }
                    catch { }
                }
            }
            catch (Exception ex)
            {
                Logger.LogWarning("ProcessEvents", "Exception occured while processing event : " + eventName + " EXCEPTION : " + ex.ToString());
            }
        }

        public void GetCallEvents(Stream request)
        {
            try
            {
                StreamReader reader = new StreamReader((Stream)request, Encoding.UTF8, false, 368640);
                string text = reader.ReadToEnd();
                reader.Close();
                Logger.LogInfo("GetCallEvents", "Events Batch received : " + text.Trim());

                eventQueue.Enqueue(text);
                Logger.LogInfo("GetCallEvents", "Event batch queued in event queue");
                eventArrived.Set();
            }
            catch (Exception ex)
            {
                Logger.LogError("GetCallEvents", "Exception occured in GetCallEvents", "EXCEPTION MESSAGE : " + ex.Message, "STACKTRACE : " + ex.StackTrace);
            }
        }

        public void GetCallEvents(string request)
        {
            try
            {
                Logger.LogInfo("GetCallEvents", "Events Batch received : " + request.Trim());

                eventQueue.Enqueue(request);
                Logger.LogInfo("GetCallEvents", "Event batch queued in event queue");
                eventArrived.Set();
            }
            catch (Exception ex)
            {
                Logger.LogError("GetCallEvents", "Exception occured in GetCallEvents", "EXCEPTION MESSAGE : " + ex.Message, "STACKTRACE : " + ex.StackTrace);
            }
        }

        public bool Start(string serverUrl, string callbackUrl)
        {
            try
            {
                _callbackUrl = callbackUrl;

                webSocket = new WebSocket(new Uri(serverUrl));

                webSocket.SetHeaders(new Dictionary<string, string>() { { "Sec-WebSocket-Key: ", SessionId } });

                try
                {
                    isConnected = webSocket.Connect();
                }
                catch (AuthenticationException ex)
                {
                    throw new WebSocketServiceException("SSL authentication denied", ex, WebSocketServiceExceptionType.SSLAuthenticationFailure); ; //Forward exception to above layer.
                }
                catch (Exception ex)
                {
                    throw; //Forward exception to above layer.
                }

                if (isConnected)
                {
                    StartProcessingEventQueue();

                    //Send heartbeat(to keep the websocket stream alive) to server every 10 secs.
                    Timer webSocketHeartbeatTimer = null;

                    webSocketHeartbeatTimer = new Timer(new TimerCallback(delegate(object o)
                    {
                        if (isConnected && IsUserLoggedIn)
                        {
                            try
                            {
                                if (webSocket != null)
                                {
                                    string hbeat = "heartbeat";
                                    webSocket.Send(hbeat);
                                }
                            }
                            catch (Exception ex)
                            {
                                Logger.LogError("Start", "Exception occured while sending data to server on websocket", "EXCEPTION:" + ex.ToString());
                            }
                        }

                        if (!IsUserLoggedIn)
                        {
                            try
                            {
                                webSocketHeartbeatTimer.Change(Timeout.Infinite, Timeout.Infinite); //Stop the timer
                            }
                            catch (Exception exc)
                            {
                                Logger.LogError("Start", "Exception occured while closing the webSocketHeartbeatTimer", "EXCEPTION:" + exc.ToString());
                            }
                        }
                    }));
                    webSocketHeartbeatTimer.Change(0, 10000);

                    new Thread(new ThreadStart(delegate()
                    {
                        while (isConnected)
                        {
                            try
                            {
                                string s = webSocket.Receive();
                                if (!string.IsNullOrEmpty(s))
                                {
                                    s = s.Trim('\0'); //removing "\0" null char from string
                                    if (s != string.Empty)
                                        GetCallEvents(s);
                                }
                                else
                                {
                                    Logger.LogInfo("Start", "Empty/Null data received from server");
                                }
                            }
                            catch (Exception ex)
                            {
                                isConnected = false;//IMP
                                if (ex is WebSocketServiceException)
                                {
                                    bool reConnected = false;

                                    //Try only until 1)successful reconnection AND 2) adapter is still loggedin..(since adapter logs out when OpenAPI heartbeat fails..)
                                    while (!reConnected && IsUserLoggedIn)
                                    {
                                        //Wait for 10 secs and then try again...
                                        Thread.Sleep(10000);

                                        reConnected = Reconnect(ex);
                                        isConnected = reConnected; //IMP
                                    }
                                }
                                else
                                {
                                    Logger.LogError("Start", "Exception occured while receive data on websocket", "EXCEPTION:" + ex.ToString());
                                }
                            }
                        }
                    })).Start();
                }
            }
            catch (Exception ex)
            {
                if (ex is UriFormatException)
                {
                    throw new WebSocketServiceException("Incorrect service URL received from server", ex, WebSocketServiceExceptionType.IncorrectURL);
                }

                Logger.LogError("Start", "Exception occured while starting websocket service", "EXCEPTION:" + ex.ToString());
                throw ex;
            }

            return true;
        }

        private bool Reconnect(Exception ex)
        {
            bool isReconnected = false;
            if (((WebSocketServiceException)ex).Type == WebSocketServiceExceptionType.EndOfStreamException ||
                ((WebSocketServiceException)ex).Type == WebSocketServiceExceptionType.NetworkFailure ||
                ((WebSocketServiceException)ex).Type == WebSocketServiceExceptionType.ErrorReadingDataFromStream ||
                ((WebSocketServiceException)ex).Type == WebSocketServiceExceptionType.ErrorWritingDataToStream)
            {
                Logger.LogError("Start", "WebSocketServiceException occured while receive data on websocket", "EXCEPTION:" + ex.ToString() + "ExceptionType:" + ((WebSocketServiceException)ex).Type.ToString());

                Logger.LogInfo("Start", "Websocket reconnecting to server...");

                try
                {
                    //Reconnect to server.
                    isReconnected = webSocket.Connect();
                }
                catch (Exception e)
                {
                    Logger.LogError("Start", "Exception occured while trying to reconnect websocket", "EXCEPTION:" + e.ToString());
                }


                if (!isReconnected)
                    Logger.LogInfo("Start", "Reconnect to server failed.");
                else
                    Logger.LogInfo("Start", "Websocket reconnected to server.");
            }

            if (LogoutOnHeartbeatFailure)
            {
                if (((WebSocketServiceException)ex).Type == WebSocketServiceExceptionType.SpecificIOException10054)
                {
                    Logger.LogError("Start", "IOException occured while receive data on websocket", "EXCEPTION:" + ex.ToString());
                    Logger.LogInfo("Start", "Websocket reconnecting to server...");

                    try
                    {
                        //Reconnect to server.
                        webSocket.Connect();
                    }
                    catch (Exception exception)
                    {
                        Logger.LogError("Start", "Exception occured while trying to reconnect for SpecificIOException10054", "EXCEPTION:" + exception.ToString());
                    }

                    if (!isConnected)
                        Logger.LogInfo("Start", "Reconnect to server failed.");
                    else
                        Logger.LogInfo("Start", "Websocket reconnected to server.");
                }
            }

            return isReconnected;
        }

        public bool Stop()
        {
            try
            {
                isConnected = false; //V.Imp
                IsUserLoggedIn = false;
                StopProcessingEventQueue();
                webSocket.Close();
            }
            catch (Exception ex)
            {
                Logger.LogError("Stop", "Failed to stop websocket service", "EXCEPTION:" + ex.ToString());
                return false;
            }
            return true;
        }

        public ObservableCollection<object> Events
        {
            get { return _events; }
            set { _events = value; }
        }

        public string CallbackUrl
        {
            get { return _callbackUrl; }
        }

        public bool IsStarted
        {
            get { return isConnected; }
        }

        private void StartProcessingEventQueue()
        {
            eventQueueThread = new Thread(new ThreadStart(ProcessEventQueueThread));
            eventQueueThread.Start();
        }

        private void StopProcessingEventQueue()
        {
            eventArrived.Set();
            try
            {
                eventQueueThread.Abort();
            }
            catch (Exception ex)
            {
                Console.WriteLine(ex.ToString());
            }

            if (webSocket != null)
                webSocket.Close();
        }

        private void ProcessEventQueueThread()
        {
            try
            {
                while (IsUserLoggedIn)
                {
                    eventArrived.WaitOne();

                    Logger.LogInfo("ProcessEventQueueThread", "Beginning processing event queue...");
                    while (eventQueue.Count > 0)
                    {
                        lock (((System.Collections.ICollection)eventQueue).SyncRoot)
                        {
                            Thread.Sleep(100);
                            ProcessEvents(eventQueue.Dequeue());
                        }
                    }
                    Logger.LogInfo("ProcessEventQueueThread", "Event queue processing complete");

                    Thread.Sleep(0);
                }
            }
            catch (Exception ex)
            {
                Logger.LogError("ProcessEventQueueThread", "Exception occured in ProcessEventQueueThread", "EXCEPTION MESSAGE : " + ex.Message, "STACKTRACE : " + ex.StackTrace);
            }
        }

        private ObservableCollection<object> _events;
    }

    public class WebSocket
    {
        private Uri mUrl;
        private TcpClient mClient;
        private NetworkStream mStream;
        private bool handshakeComplete;
        private Dictionary<string, string> mHeaders;
        private SslStream fSslStream;
        private bool isSecuredmode;
        private byte[] NonFragmentedData;

        public WebSocket(Uri url)
        {
            mUrl = url;

            string protocol = mUrl.Scheme;
            if (!protocol.Equals("ws") && !protocol.Equals("wss"))
                throw new WebSocketServiceException("Unsupported protocol: " + protocol, WebSocketServiceExceptionType.UnsupportedProtocol);
        }

        public void SetHeaders(Dictionary<string, string> headers)
        {
            mHeaders = headers;
        }

        public bool Connect()
        {
            string request = string.Empty;

            string host = mUrl.DnsSafeHost;
            string path = mUrl.PathAndQuery;
            string origin = "http://" + host;

            mClient = CreateSocket(mUrl);


            if (mUrl.Scheme == "wss")
            {
                isSecuredmode = true;
                ServicePointManager.ServerCertificateValidationCallback = new System.Net.Security.RemoteCertificateValidationCallback(CertificateValidationCallback);
                System.Net.ServicePointManager.Expect100Continue = false;

                fSslStream = new SslStream(mClient.GetStream(), false, new RemoteCertificateValidationCallback(CertificateValidationCallback), null);
                fSslStream.AuthenticateAsClient(host, null, SslProtocols.Tls12, true);
            }
            else
            {
                isSecuredmode = false;
                mStream = mClient.GetStream();
            }
            //IPOFFICE-40992 - Coverity Fix Defect: 534951 
            if (mClient != null && mClient.Client != null && mClient.Client.RemoteEndPoint != null && mClient.Client.RemoteEndPoint is IPEndPoint)
            {
                int port = ((IPEndPoint)mClient.Client.RemoteEndPoint).Port;
                if (port != 80)
                    host = host + ":" + port;
            }

            StringBuilder extraHeaders = new StringBuilder();
            if (mHeaders != null)
            {
                foreach (KeyValuePair<string, string> header in mHeaders)
                {
                    extraHeaders.Append(header.Key + ": " + header.Value + "\r\n");
                    break;
                }
            }

            request = "GET " + path + " HTTP/1.1\r\n" +
                             "Sec-WebSocket-Version: 13\r\n" +
                             "Upgrade: WebSocket\r\n" +
                             "Connection: Upgrade\r\n" +
                             "Host: " + host + "\r\n" +
                             "Origin: " + origin + "\r\n" +
                             extraHeaders.ToString() + "\r\n";
            byte[] sendBuffer = Encoding.UTF8.GetBytes(request);

            if (mUrl.Scheme != "wss")
            {
                mStream.Write(sendBuffer, 0, sendBuffer.Length);

                StreamReader reader = new StreamReader(mStream);
                {
                    string header = reader.ReadLine();

                    if (!header.StartsWith("HTTP/1.1 101"))
                        throw new WebSocketServiceException("Invalid handshake response. Header 1 :" + header, WebSocketServiceExceptionType.InvalidHandshakeResponse);
                }
            }
            else
            {
                fSslStream.Write(sendBuffer, 0, sendBuffer.Length);

                StreamReader reader = new StreamReader(fSslStream);
                {
                    string header = reader.ReadLine();
                    if (!header.StartsWith("HTTP/1.1 101"))
                        throw new WebSocketServiceException("Invalid handshake response. Header 1 :" + header, WebSocketServiceExceptionType.InvalidHandshakeResponse);
                }
            }

            handshakeComplete = true;
            return handshakeComplete;
        }

        private bool CertificateValidationCallback(object sender, System.Security.Cryptography.X509Certificates.X509Certificate certification, System.Security.Cryptography.X509Certificates.X509Chain chain, SslPolicyErrors sslPolicyErrors)
        {
            bool retVal = false;
            bool sslCertError = false;
            try
            {
                if (((sslPolicyErrors & SslPolicyErrors.None) == SslPolicyErrors.None) ||
                            ((sslPolicyErrors & SslPolicyErrors.RemoteCertificateChainErrors) == SslPolicyErrors.RemoteCertificateChainErrors))
                {
                    sslCertError = false;
                }
                DateTime dtEnd = DateTime.Parse(certification.GetExpirationDateString());
                DateTime dtStart = DateTime.Parse(certification.GetEffectiveDateString());

                if (dtStart > DateTime.Now || dtEnd < DateTime.Now)
                {
                    retVal = false;
                    return retVal;
                }

                if (sslCertError == false)
                {
                    retVal = true;
                }
            }
            catch (Exception)
            {
                retVal = false;
            }

            return retVal;
        }

        public void Send(string str)
        {
            try
            {
                if (!handshakeComplete)
                    throw new WebSocketServiceException("Cannot send data without Handshake", WebSocketServiceExceptionType.HandshakeNotComplete);

                byte[] sendBuffer = Encoding.UTF8.GetBytes(str);

                byte[] frame = GetWebSocketDataFrameToSend(sendBuffer);

                if (isSecuredmode)
                {
                    fSslStream.Write(frame, 0, frame.Length);
                    fSslStream.Flush();
                }
                else
                {
                    mStream.Write(frame, 0, frame.Length);
                    mStream.Flush();
                }
            }
            catch (Exception ex)
            {
                throw new WebSocketServiceException("Error while writing data to be sent over the web socket", ex, WebSocketServiceExceptionType.ErrorWritingDataToStream);
            }
        }

        private byte[] GetWebSocketDataFrameToSend(byte[] sendBuffer)
        {
            ///* PayloadLen */

            var dataLen = Convert.ToInt64(sendBuffer.Length);
            var payloadLen = dataLen < 126
                           ? (byte)dataLen
                           : dataLen < 0x010000
                             ? (byte)126
                             : (byte)127;


            /* ExtPayloadLen */

            byte[] ExtPayloadLen = payloadLen < 126
                          ? new byte[] { }
                          : payloadLen == 126
                            ? ((ushort)dataLen).ToByteArrayInternally(ByteOrder.BIG)
                            : ((ulong)dataLen).ToByteArrayInternally(ByteOrder.BIG);


            using (var buffer = new MemoryStream())
            {
                int header = (int)0x1;
                header = (header << 1) + (int)0x0;
                header = (header << 1) + (int)0x0;
                header = (header << 1) + (int)0x0;
                header = (header << 4) + (int)0x1;
                header = (header << 1) + (int)0x1;
                header = (header << 7) + (int)payloadLen;
                buffer.Write(((ushort)header).ToByteArrayInternally(ByteOrder.BIG), 0, 2);

                if (payloadLen > 125)
                    buffer.Write(ExtPayloadLen, 0, ExtPayloadLen.Length);

                byte[] MaskingKey = createMaskingKey();
                buffer.Write(MaskingKey, 0, MaskingKey.Length);


                for (long i = 0; i < sendBuffer.LongLength; i++)
                    sendBuffer[i] = (byte)(sendBuffer[i] ^ MaskingKey[i % 4]);

                if (payloadLen > 0)
                {
                    if (payloadLen < 127)
                        buffer.Write(sendBuffer, 0, sendBuffer.Length);
                    else
                        buffer.WriteBytes(sendBuffer);
                }
                buffer.Close();
                return buffer.ToArray();
            }
        }

        private static byte[] createMaskingKey()
        {
            var key = new byte[4];
            var rand = new Random();
            rand.NextBytes(key);

            return key;
        }

        public string Receive()
        {
            try
            {
                if (!handshakeComplete)
                    throw new WebSocketServiceException("Cannot receive data without Handshake", WebSocketServiceExceptionType.HandshakeNotComplete);

                if (isSecuredmode)
                {
                    var header = fSslStream.ReadBytes(2);
                    if (header.Length != 2)
                        throw new WebSocketServiceException(
                          "The header part of a frame cannot be read from the data source.", WebSocketServiceExceptionType.ErrorReadingDataFromStream);
                    return ConvertByteArrayToUTF8String(this.GetDataFromWebSocketFrame(header, fSslStream));
                }
                else
                {
                    var header = mStream.ReadBytes(2);
                    if (header.Length != 2)
                        throw new WebSocketServiceException(
                          "The header part of a frame cannot be read from the data source.", WebSocketServiceExceptionType.ErrorReadingDataFromStream);
                    return ConvertByteArrayToUTF8String(this.GetDataFromWebSocketFrame(header, mStream));
                }
            }
            catch (Exception ex)
            {
                throw new WebSocketServiceException(
                      ex.StackTrace, WebSocketServiceExceptionType.ErrorReadingDataFromStream);
            }
        }

        private byte[] GetDataFromWebSocketFrame(byte[] header, Stream stream)
        {
            var opcode = (Opcode)(header[0] & 0x0f);

            var finalFrame = (header[0] & 0x80) == 0x80 ? Fin.FINAL : Fin.MORE;

            var payloadLen = (byte)(header[1] & 0x7f);

            var extLen = payloadLen < 126
               ? 0
               : payloadLen == 126
                 ? 2
                 : 8;

            var extPayloadLen = extLen > 0
                              ? stream.ReadBytes(extLen)
                              : new byte[] { };

            if (extLen > 0 && extPayloadLen.Length != extLen)
                throw new WebSocketServiceException(
                  "The 'Extended Payload Length' of a frame cannot be read from the data source.", WebSocketServiceExceptionType.UnsupportedScheme);

            ulong dataLen = payloadLen < 126
                 ? payloadLen
                 : payloadLen == 126
                   ? extPayloadLen.ToUInt16(ByteOrder.BIG)
                   : extPayloadLen.ToUInt64(ByteOrder.BIG);

            byte[] data = null;
            if (dataLen > 0)
            {
                // Check if allowable payload data length.
                if (payloadLen > 126 && dataLen > long.MaxValue)
                    throw new WebSocketServiceException(
                      "The 'Payload Data' length is greater than the allowable length.", WebSocketServiceExceptionType.UnsupportedScheme);

                data = payloadLen > 126
                     ? stream.ReadBytes((long)dataLen, 1024)
                     : stream.ReadBytes((int)dataLen);

                if (data.LongLength != (long)dataLen)
                    throw new WebSocketServiceException(
                      "The 'Payload Data' of a frame cannot be read from the data source.", WebSocketServiceExceptionType.UnsupportedScheme);
            }
            else
            {
                data = new byte[] { };
            }

            if (opcode == Opcode.TEXT || opcode == Opcode.CONT)
            {
                if (NonFragmentedData == null)
                    NonFragmentedData = new byte[] { };

                byte[] dataReceivedTillNow = new byte[NonFragmentedData.Length + data.Length];
                System.Buffer.BlockCopy(NonFragmentedData, 0, dataReceivedTillNow, 0, NonFragmentedData.Length);
                System.Buffer.BlockCopy(data, 0, dataReceivedTillNow, NonFragmentedData.Length, data.Length);

                NonFragmentedData = dataReceivedTillNow;
            }

            if (finalFrame == Fin.FINAL && (opcode == Opcode.TEXT || opcode == Opcode.CONT))
            {
                byte[] dataReceived = new byte[NonFragmentedData.Length];
                NonFragmentedData.CopyTo(dataReceived, 0);
                NonFragmentedData = null;

                return dataReceived;
            }
            else
                return null;
        }

        private string ConvertByteArrayToString(byte[] input)
        {
            System.Text.ASCIIEncoding decoder = new System.Text.ASCIIEncoding();
            return decoder.GetString(input);
        }

        private string ConvertByteArrayToUTF8String(byte[] input)
        {
            if (input != null)
            {
                System.Text.UTF8Encoding decoder = new System.Text.UTF8Encoding();
                return decoder.GetString(input);
            }
            else
                return string.Empty;
        }

        public void Close()
        {
            if (isSecuredmode)
            {
                if (fSslStream != null)
                {
                    fSslStream.Dispose();
                    fSslStream.Close();
                    fSslStream = null;
                }
            }
            else
            {
                if (mStream != null)
                {
                    mStream.Dispose();
                    mStream.Close();
                    mStream = null;
                }
            }

            if (mClient != null)
            {
                mClient.Close();
                mClient = null;
            }
        }

        private static TcpClient CreateSocket(Uri url)
        {
            string scheme = url.Scheme;
            string host = url.DnsSafeHost;

            int port = url.Port;
            if (port <= 0)
            {
                if (scheme.Equals("wss"))
                    port = 443;
                else if (scheme.Equals("ws"))
                    port = 80;
                else
                    throw new WebSocketServiceException("Unsupported scheme: " + scheme, WebSocketServiceExceptionType.UnsupportedScheme);
            }

            try
            {
                return new TcpClient(host, port);
            }
            catch (Exception ex)
            {
                throw new WebSocketServiceException("Exception while creating socket to connect to host : " + host + " and port : " + port, ex, WebSocketServiceExceptionType.UnableToConnect);
            }
        }
    }

    /// <summary>
    /// Provides a set of static methods for the websocket-sharp.
    /// </summary>
    public static class Ext
    {
        private static byte[] readBytes(
          this Stream stream, byte[] buffer, int offset, int length)
        {
            var len = stream.Read(buffer, offset, length);
            if (len < 1)
                return buffer.SubArray(0, offset);

            var tmp = 0;
            while (len < length)
            {
                tmp = stream.Read(buffer, offset + len, length - len);
                if (tmp < 1)
                    break;

                len += tmp;
            }

            return len < length
                   ? buffer.SubArray(0, offset + len)
                   : buffer;
        }

        private static bool readBytes(
          this Stream stream, byte[] buffer, int offset, int length, Stream dest)
        {
            var bytes = stream.readBytes(buffer, offset, length);
            var len = bytes.Length;
            dest.Write(bytes, 0, len);

            return len == offset + length;
        }

        internal static byte[] ReadBytes(this Stream stream, int length)
        {
            return stream.readBytes(new byte[length], 0, length);
        }

        internal static byte[] ReadBytes(
          this Stream stream, long length, int bufferLength)
        {
            using (var result = new MemoryStream())
            {
                var count = length / bufferLength;
                var rem = (int)(length % bufferLength);

                var buffer = new byte[bufferLength];
                var end = false;
                for (long i = 0; i < count; i++)
                {
                    if (!stream.readBytes(buffer, 0, bufferLength, result))
                    {
                        end = true;
                        break;
                    }
                }

                if (!end && rem > 0)
                    stream.readBytes(new byte[rem], 0, rem, result);

                result.Close();
                return result.ToArray();
            }
        }

        internal static byte[] ToByteArrayInternally(
          this ushort value, ByteOrder order)
        {
            var buffer = BitConverter.GetBytes(value);
            return order.IsHostOrder()
                   ? buffer
                   : buffer.Reverse().ToArray();
        }

        internal static byte[] ToByteArrayInternally(
          this ulong value, ByteOrder order)
        {
            var buffer = BitConverter.GetBytes(value);
            return order.IsHostOrder()
                   ? buffer
                   : buffer.Reverse().ToArray();
        }

        internal static ushort ToUInt16(this byte[] src, ByteOrder srcOrder)
        {
            return BitConverter.ToUInt16(src.ToHostOrder(srcOrder), 0);
        }

        internal static ulong ToUInt64(this byte[] src, ByteOrder srcOrder)
        {
            return BitConverter.ToUInt64(src.ToHostOrder(srcOrder), 0);
        }

        internal static void WriteBytes(this Stream stream, byte[] value)
        {
            using (var src = new MemoryStream(value))
            {
                src.CopyTo(stream);
            }
        }

        /// <summary>
        /// Determines whether the specified <see cref="ByteOrder"/> is host (this
        /// computer architecture) byte order.
        /// <returns>
        /// <c>true</c> if <paramref name="order"/> is host byte order; otherwise,
        /// <c>false</c>.
        /// </returns>
        /// <param name="order">
        /// A <see cref="ByteOrder"/> to test.
        /// </param>
        public static bool IsHostOrder(this ByteOrder order)
        {
            // true : !(true ^ true)  or !(false ^ false)
            // false: !(true ^ false) or !(false ^ true)
            return !(BitConverter.IsLittleEndian ^ (order == ByteOrder.LITTLE));
        }


        /// <summary>
        /// Retrieves a sub-array from the specified <paramref name="array"/>.
        /// A sub-array starts at the specified element position.
        /// </summary>
        /// <returns>
        /// An array of T that receives a sub-array, or an empty array of T if any
        /// problems with the parameters.
        /// </returns>
        /// <param name="array">
        /// An array of T that contains the data to retrieve a sub-array.
        /// </param>
        /// <param name="startIndex">
        /// An <see cref="int"/> that contains the zero-based starting position of
        /// a sub-array in <paramref name="array"/>.
        /// </param>
        /// <param name="length">
        /// An <see cref="int"/> that contains the number of elements to retrieve
        /// a sub-array.
        /// </param>
        /// <typeparam name="T">
        /// The type of elements in the <paramref name="array"/>.
        /// </typeparam>
        public static T[] SubArray<T>(this T[] array, int startIndex, int length)
        {
            if (array == null || array.Length == 0)
                return new T[0];

            if (startIndex < 0 || length <= 0)
                return new T[0];

            if (startIndex + length > array.Length)
                return new T[0];

            if (startIndex == 0 && array.Length == length)
                return array;

            T[] subArray = new T[length];
            Array.Copy(array, startIndex, subArray, 0, length);

            return subArray;
        }

        /// <summary>
        /// Converts the order of the specified array of <see cref="byte"/> to the
        /// host byte order.
        /// </summary>
        /// <returns>
        /// An array of <see cref="byte"/> converted from <paramref name="src"/>.
        /// </returns>
        /// <param name="src">
        /// An array of <see cref="byte"/> to convert.
        /// </param>
        /// <param name="srcOrder">
        /// One of <see cref="ByteOrder"/> values that indicate the byte order of
        /// <paramref name="src"/>.
        /// </param>
        /// <exception cref="ArgumentNullException">
        /// <paramref name="src"/> is <see langword="null"/>.
        /// </exception>
        public static byte[] ToHostOrder(this byte[] src, ByteOrder srcOrder)
        {
            if (src == null)
                throw new ArgumentNullException("src");

            return src.Length <= 1 || srcOrder.IsHostOrder()
                   ? src
                   : src.Reverse().ToArray();
        }
    }

    public class WebSocketServiceException : System.Exception //todo:ws - Add SerializationInfo/StreamingContext  in constructor
    {
        // The default constructor needs to be defined
        // explicitly now since it would be gone otherwise.

        public WebSocketServiceException()
        {
        }

        public WebSocketServiceException(string message, Exception innerException, WebSocketServiceExceptionType type)
            : base(message, innerException)
        {
            _type = type;
        }

        public WebSocketServiceException(string message, WebSocketServiceExceptionType type)
            : base(message)
        {
            _type = type;
        }

        private WebSocketServiceExceptionType _type = WebSocketServiceExceptionType.Unknown;
        public WebSocketServiceExceptionType Type
        {
            get { return _type; }
        }
    }

    public enum WebSocketServiceExceptionType
    {
        Unknown,
        InvalidHandshakeResponse,
        NetworkFailure,
        ServerNotReachable,
        UnsupportedProtocol,
        UnsupportedScheme,
        SSLNotSupported,
        HandshakeNotComplete,
        ErrorWritingDataToStream,
        ErrorReadingDataFromStream,
        IncorrectURL,
        UnableToConnect,
        EndOfStreamException,
        SpecificIOException10054,
        SSLAuthenticationFailure,
    }

    /// <summary>
    /// Contains the values that indicate whether the byte order is a Little-endian or Big-endian.
    /// </summary>
    public enum ByteOrder : byte
    {
        /// <summary>
        /// Indicates a Little-endian.
        /// </summary>
        LITTLE,
        /// <summary>
        /// Indicates a Big-endian.
        /// </summary>
        BIG
    }

    public enum Opcode : byte
    {
        /// <summary>
        /// Equivalent to numeric value 0. Indicates a continuation frame.
        /// </summary>
        CONT = 0x0,
        /// <summary>
        /// Equivalent to numeric value 1. Indicates a text frame.
        /// </summary>
        TEXT = 0x1,
        /// <summary>
        /// Equivalent to numeric value 2. Indicates a binary frame.
        /// </summary>
        BINARY = 0x2,
        /// <summary>
        /// Equivalent to numeric value 8. Indicates a connection close frame.
        /// </summary>
        CLOSE = 0x8,
        /// <summary>
        /// Equivalent to numeric value 9. Indicates a ping frame.
        /// </summary>
        PING = 0x9,
        /// <summary>
        /// Equivalent to numeric value 10. Indicates a pong frame.
        /// </summary>
        PONG = 0xa
    }

    public enum Fin : byte
    {
        MORE = 0x0,
        FINAL = 0x1
    }

    public static class Logger
    {
        internal static void LogInfo(string p1, string p2)
        {
            Debug.Print(p1 + Environment.NewLine + p2);
        }

        internal static void LogWarning(string p1, string p2)
        {
            Debug.Print(p1 + Environment.NewLine + p2);
        }

        internal static void LogError(string p1, string p2, string p3, string p4)
        {
            Debug.Print(p1 + Environment.NewLine + p2 + Environment.NewLine + p3 + Environment.NewLine + p4);
        }

        internal static void LogError(string p1, string p2, string p3)
        {
            Debug.Print(p1 + Environment.NewLine + p2 + Environment.NewLine + p3);
        }
    }
}
