﻿using OneXOpenAPISample;
using System;
using System.Collections.Generic;
using System.IO;
using System.Linq;
using System.Net;
using System.Net.Security;
using System.Text;
using System.Threading.Tasks;
using System.Xml;

namespace OneXAdapterSample
{
    public interface iAdapter
    {
        bool Login();
        void Logout();
        List<Contact> GetDirectory();
        List<CallLog> GetCallLogs();
        void MakeCall(string number);
    }

    public class SampleAdapter : iAdapter
    {
        private ServerSettings serverSettings;
        private LoginResponseDetails responseDetails;
        private WebSocketService service;
        private List<AVMsnapshotData> voiceMessages = new List<AVMsnapshotData>(); 

        public SampleAdapter(ServerSettings Server_Settings)
        {
            serverSettings = Server_Settings;
        }

        public bool Login()
        {
            userLoginRequest obj = new userLoginRequest();
            LoginSubscriptionRequest req = new LoginSubscriptionRequest();
            req.eventTypes = new EventType[] { EventType.CallControlFeaturesEvents, EventType.CallLogEvents, EventType.PhysicalDeviceFeaturesEvents, EventType.DeviceMaintenanceEvents, EventType.VoicemailServicesEvents, EventType.CapabilityExchangeServicesCallbackEvents, EventType.SystemServicesCallbackEvents, EventType.MonitoringServicesCallbackEvents, EventType.IOServicesFeaturesCallbackEvents, EventType.LogicalDeviceFeaturesEvents, EventType.SnapshotServicesCallbackEvents, EventType.CallAssociatedFeaturesEvents, EventType.LogoutEvents, EventType.PresenceEvents, EventType.DirectoryEvents, EventType.RecordingServicesEvents, EventType.ImEvents, EventType.PhotoChangeEvents, EventType.IPOConfigChangeEvents };

            req.httpMethod = HttpMethod.WEBSOCKET;
            req.httpMethodSpecified = true;

            obj.loginSubscriptionRequest = req;
            obj.applicationName = "AvayaIPOfficePlugin";
            obj.userName = this.serverSettings.UserId;
            obj.userPassword = this.serverSettings.Password;
            obj.applicationVersion = this.serverSettings.AppVersion;

            Serializer serializer = new Serializer();

            HttpStatusCode requestStatus = HttpStatusCode.Continue;
            userLoginResponseLoginFailureCode loginFailureCode = userLoginResponseLoginFailureCode.UNKNOWN;
            String postData = serializer.Serialize(obj);
            string response = SendRESTRequest("/inyama/service/session", postData, HTTPMethod.POST, "", out requestStatus);
            responseDetails.serverResponse = response;

            if (requestStatus == HttpStatusCode.OK || requestStatus == HttpStatusCode.Accepted)
            {
                XmlDocument doc = new XmlDocument();
                doc.LoadXml(response);
                Deserilizer deserializer = new Deserilizer();
                Object responseObj = deserializer.Deserlize("userLoginResponse", doc.InnerXml);

                if (responseObj != null)
                {
                    userLoginResponse loginResponse = (userLoginResponse)responseObj;
                    if (loginResponse != null)
                    {
                        if (!string.IsNullOrEmpty(loginResponse.clientSessionID))
                        {
                            responseDetails.sessionId = loginResponse.clientSessionID;
                            responseDetails.serverVersion = loginResponse.applicationVersion;
                            responseDetails.loginFailureCode = loginResponse.loginFailureCode;
                            responseDetails.webSocketWssUrl = loginResponse.webSocketWssUrl;
                            responseDetails.webSocketWsUrl = loginResponse.webSocketWsUrl;
                            responseDetails.switchDevice = loginResponse.userSwitchDevice;
                            responseDetails.userExtension = loginResponse.userExtension;
                            responseDetails.isLoggedIn = true;

                            OpenWebSocket();
                        }
                    }
                }
            }
            else
            {
                if (requestStatus == HttpStatusCode.Unauthorized || requestStatus == HttpStatusCode.InternalServerError || requestStatus == HttpStatusCode.ServiceUnavailable)
                {
                    Deserilizer deserializer = new Deserilizer();
                    Error error = (Error)deserializer.Deserlize("Error", response);
                    if (error != null)
                    {
                        loginFailureCode = GetFailureCode(error.errorCode);
                        if (error.errorCode == OpenApiErrorCode.UNSUPPORTED_APPLICATION_VERSION)
                            responseDetails.serverVersion = error.errorValue;
                    }
                    else
                    {
                        switch (requestStatus)
                        {
                            case HttpStatusCode.Unauthorized:
                                loginFailureCode = userLoginResponseLoginFailureCode.UNAUTHORIZED;
                                break;
                            case HttpStatusCode.ServiceUnavailable:
                                loginFailureCode = userLoginResponseLoginFailureCode.SERVICE_UNAVAILABLE;
                                break;
                            case HttpStatusCode.InternalServerError:
                                loginFailureCode = userLoginResponseLoginFailureCode.SERVER_PROBLEM;
                                break;
                        }
                    }
                    responseDetails.loginFailureCode = loginFailureCode;
                }
                else
                {
                    loginFailureCode = userLoginResponseLoginFailureCode.UNKNOWN;
                    responseDetails.loginFailureCode = loginFailureCode;
                }
            }

            return responseDetails.isLoggedIn;
        }

        private void OpenWebSocket()
        {
            service = new WebSocketService(responseDetails.sessionId, true);
            service.Events.CollectionChanged -= new System.Collections.Specialized.NotifyCollectionChangedEventHandler(Events_CollectionChanged);
            service.Events.CollectionChanged += new System.Collections.Specialized.NotifyCollectionChangedEventHandler(Events_CollectionChanged);
            bool IsEventDeliveryServiceStarted = service.Start(responseDetails.webSocketWssUrl, string.Empty);
        }

        public void Logout()
        {
            if (responseDetails != null && responseDetails.isLoggedIn == true)
            {
                userLoginRequest obj = new userLoginRequest();
                obj.applicationName = "AvayaIPOfficePlugin";
                obj.userName = this.serverSettings.UserId;
                obj.userPassword = this.serverSettings.Password;
                Serializer serializer = new Serializer();

                HttpStatusCode requestStatus = HttpStatusCode.Continue;
                String postData = serializer.Serialize(obj);
                string response = SendRESTRequest("/inyama/service/session", postData, HTTPMethod.DELETE, "", out requestStatus);
                responseDetails.isLoggedIn = false;
            }
        }

        public List<Contact> GetDirectory()
        {
            List<Contact> contacts = new List<Contact>();
            HttpStatusCode requestStatus = HttpStatusCode.Continue;
            string response = SendRESTRequest("/inyama/service/directory/system", "", HTTPMethod.GET, this.responseDetails.sessionId, out requestStatus);
            responseDetails.serverResponse = response;
            
            if (requestStatus == HttpStatusCode.OK || requestStatus == HttpStatusCode.Accepted)
            {
                XmlDocument doc = new XmlDocument();
                doc.LoadXml(response);
                Deserilizer deserializer = new Deserilizer();
                Object responseObj = deserializer.Deserlize("directorySearchResponse", doc.InnerXml);
                if (responseObj != null)
                {
                    foreach (directoryContact entry in ((directorySearchResponse)responseObj).list)
                    {
                        contacts.Add(new Contact() { Name = entry.firstName, Number = entry.workPhone });
                    }
                }
            }
            return contacts;
        }

        public List<CallLog> GetCallLogs()
        {
            List<CallLog> callLogs = new List<CallLog>();
            SendStoredCallLog obj = new SendStoredCallLog();
            byte[] byteArray = Encoding.UTF8.GetBytes(this.responseDetails.switchDevice);
            DeviceID callingDev = new DeviceID() { Value = this.responseDetails.userExtension };
            callingDev.typeOfNumber = typeOfNumberEnum.deviceNumber;
            callingDev.switchingSubDomainInformationElements = byteArray;
            MediaClassComponents[] mediaClassArray = new MediaClassComponents[] { MediaClassComponents.voice };
            callingDev.mediaClass = mediaClassArray;
            obj.device = callingDev;
            Serializer serializer = new Serializer();
            String postData = serializer.Serialize(obj);

            HttpStatusCode requestStatus = HttpStatusCode.Continue;
            string response = SendRESTRequest("/inyama/service/calllog", postData, HTTPMethod.GET, this.responseDetails.sessionId, out requestStatus);
            responseDetails.serverResponse = response;

            if (requestStatus == HttpStatusCode.OK || requestStatus == HttpStatusCode.Accepted)
            {
                XmlDocument doc = new XmlDocument();
                doc.LoadXml(response);
                Deserilizer deserializer = new Deserilizer();
                Object responseObj = deserializer.Deserlize("SendStoredCallLogResponse", doc.InnerXml);
                if (responseObj != null)
                {
                    foreach (callLogEntry entry in ((SendStoredCallLogResponse)responseObj).@return)
                    {
                        CallLog log = new CallLog();
                        log.LogType = entry.call_type;
                        log.Name = string.IsNullOrEmpty(entry.calling_party_name) ? entry.dialback_number : entry.calling_party_name;
                        log.Number = entry.dialback_number;
                        log.DateTime = entry.call_timestamp;
                        log.Duration = entry.duration;
                        callLogs.Add(log);
                    }
                }
            }
            return callLogs;
        }
        
        public void GetVoiceMessages()
        {
            AVMsnapshotVoiceMailbox obj = new AVMsnapshotVoiceMailbox();
            byte[] byteArray = Encoding.UTF8.GetBytes(this.responseDetails.switchDevice);
            DeviceID callingDev = new DeviceID() { Value = this.responseDetails.userExtension };
            callingDev.typeOfNumber = typeOfNumberEnum.deviceNumber;
            callingDev.switchingSubDomainInformationElements = byteArray;
            MediaClassComponents[] mediaClassArray = new MediaClassComponents[] { MediaClassComponents.voice };
            callingDev.mediaClass = mediaClassArray;
            obj.device = callingDev;
            Serializer serializer = new Serializer();
            String postData = serializer.Serialize(obj);

            HttpStatusCode requestStatus = HttpStatusCode.Continue;
            string response = SendRESTRequest("/inyama/service/voicemail/getMessages", postData, HTTPMethod.POST, this.responseDetails.sessionId, out requestStatus);
            if (requestStatus == HttpStatusCode.OK || requestStatus == HttpStatusCode.Accepted)
            {

            }
        }

        public void MakeCall(string number)
        {
            try
            {
                if (!string.IsNullOrEmpty(number))
                {
                    MakeCall obj = new MakeCall();
                    DeviceID calledDev = new DeviceID() { Value = number };
                    MediaClassComponents[] mediaClassArray = new MediaClassComponents[] { MediaClassComponents.audio };
                    calledDev.mediaClass = mediaClassArray;
                    byte[] byteArray = Encoding.UTF8.GetBytes(this.responseDetails.switchDevice);
                    calledDev.typeOfNumber = typeOfNumberEnum.dialingNumber;
                    obj.calledDirectoryNumber = calledDev;
                    DeviceID callingDev = new DeviceID() { Value = this.responseDetails.userExtension };
                    callingDev.mediaClass = mediaClassArray;
                    callingDev.typeOfNumber = typeOfNumberEnum.dialingNumber;
                    callingDev.switchingSubDomainInformationElements = byteArray;
                    obj.callingDevice = callingDev;
                    MediaCallCharacteristics mediaCallChar = new MediaCallCharacteristics();
                    mediaCallChar.mediaClass = new MediaClass() { voice = true, voiceSpecified = true };
                    obj.mediaCallCharacteristics = mediaCallChar;
                    obj.autoOriginate = AutoOriginate.doNotPrompt;
                    string callIdentifier = Guid.NewGuid().ToString();
                    CorrelatorData corrData = new CorrelatorData();
                    byte[] byteArrayString = Encoding.UTF8.GetBytes(callIdentifier);
                    corrData.Item = byteArrayString;
                    obj.correlatorData = corrData;
                    obj.subjectOfCall = "Testing";
                    HttpStatusCode requestStatus = HttpStatusCode.Continue;
                    Serializer serializer = new Serializer();
                    String postData = serializer.Serialize(obj);
                    string response = SendRESTRequest("/inyama/service/callcontrol/makeCall", postData, HTTPMethod.POST, responseDetails.sessionId, out requestStatus);
                }
            }
            catch { }
        }

        private string SendRESTRequest(string uri, string postData, HTTPMethod method, string sessionId, out HttpStatusCode HTTPStatusCode)
        {
            ServicePointManager.ServerCertificateValidationCallback = new System.Net.Security.RemoteCertificateValidationCallback(CertificateValidationCallback);
            HTTPStatusCode = HttpStatusCode.Continue;
            try
            {
                HttpWebResponse response;
                System.Net.ServicePointManager.Expect100Continue = false;
                String requestUri = "https://" + serverSettings.Server + ":" + serverSettings.Port + uri;
                HttpWebRequest request = (HttpWebRequest)HttpWebRequest.Create(requestUri);
                WebHeaderCollection header = new WebHeaderCollection();
                header.Add("ClientSessionId", sessionId);
                request.Headers = header;
                request.ProtocolVersion = HttpVersion.Version10;
                request.Method = method.ToString();
                Stream dataStream;

                if (method == HTTPMethod.POST || method == HTTPMethod.PUT || method == HTTPMethod.DELETE)
                {
                    String queryData = postData;
                    byte[] byteArray = Encoding.UTF8.GetBytes(queryData);
                    request.ContentType = "text/xml";
                    request.UserAgent = "AvayaIPOfficePlugin";
                    request.ContentLength = byteArray.Length;
                    dataStream = request.GetRequestStream();
                    dataStream.Write(byteArray, 0, byteArray.Length);
                    dataStream.Close();
                }
                response = (HttpWebResponse)request.GetResponse();
                HTTPStatusCode = ((HttpWebResponse)response).StatusCode;
                dataStream = response.GetResponseStream();

                StreamReader reader = null;
                string responseFromServer = string.Empty;
                if (response.ContentLength <= 0)
                {
                    reader = new StreamReader(dataStream, Encoding.UTF8, false, 368640);
                    responseFromServer = reader.ReadToEnd();
                    reader.Close();
                }
 
                dataStream.Close();
                response.Close();
                return responseFromServer;
            }
            catch (WebException ex)
            {
                string responseFromServer = string.Empty;
                if (ex.Response != null)
                {
                    HTTPStatusCode = ((HttpWebResponse)(ex.Response)).StatusCode;
                    Stream dataStream = ex.Response.GetResponseStream();
                    StreamReader reader = new StreamReader(dataStream, Encoding.UTF8, false, 368640);
                    responseFromServer = reader.ReadToEnd();
                }
                else
                {
                    responseFromServer = ex.Status.ToString();
                }
                return responseFromServer;
            }
            catch (Exception)
            {
                return "";
            }
        }

        private void Events_CollectionChanged(object sender, System.Collections.Specialized.NotifyCollectionChangedEventArgs e)
        {
            foreach (var _event in e.NewItems)
            {
                switch (_event.ToString())
                {
                    case "AVMSnapshotEvent":
                        {
                            voiceMessages.Clear();
                            AVMSnapshotEvent evtData = (AVMSnapshotEvent)_event;
                            if (evtData.snapshotDataList != null)
                            {
                                foreach (AVMsnapshotData vmData in evtData.snapshotDataList)
                                {
                                    voiceMessages.Add(vmData);
                                    //To Do: Get Data from Here to client Application
                                }
                            }
                        }
                        break;
                    case "DeliveredEvent":
                        OnCallDeliveredEvent(_event);
                        break;
                }
            }
        }

        private void OnCallDeliveredEvent(object _event)
        {
            try
            {
                DeliveredEvent evtData = (DeliveredEvent)_event;
                DeviceID callingDevice = null;
                DeviceID calledDevice = null;
                DeviceID alertingDevice = null;

                if (evtData.calledDevice != null && evtData.callingDevice != null && evtData.alertingDevice != null && evtData.connection != null)
                {
                    callingDevice = (DeviceID)evtData.callingDevice.Item;
                    calledDevice = (DeviceID)evtData.calledDevice.Item;
                    alertingDevice = (DeviceID)evtData.calledDevice.Item;
                    try
                    {
                        if (evtData.associatedCalledDevice != null || evtData.associatedCallingDevice != null)
                        {
                            return;
                        }
                    }
                    catch { }
                    string callerPartyName = "";
                    string calledPartyName = "";
                    try
                    {
                        if (evtData.extensions.privateData != null && evtData.extensions.privateData.@private != null)
                        {
                            Deserilizer deserializer = new Deserilizer();
                            object additionalCallInfo = deserializer.Deserlize(typeof(additionalCallInfo).Name, ((CSTAPrivateDataPrivate)evtData.extensions.privateData.@private).Any[0].InnerXml);
                            if (additionalCallInfo is additionalCallInfo)
                            {
                                callerPartyName = ((additionalCallInfo)additionalCallInfo).callingPartyInfo.callerName;
                                calledPartyName = ((additionalCallInfo)additionalCallInfo).calledPartyInfo.callerName;
                            }
                        }
                    }
                    catch { }

                    if (evtData.localConnectionInfo == LocalConnectionState.alerting)
                    {
                        if (!string.IsNullOrEmpty(callingDevice.Value) && callingDevice.Value == responseDetails.userExtension)
                        {
                            if (evtData.servicesPermitted != null && evtData.servicesPermitted.callControlServices != null && evtData.servicesPermitted.callControlServices.answerCall)
                            {
                                //Incoming Call// To Do: Get Data from Here to client Application
                            }
                            else
                            {
                                //Outgoing Call// To Do: Get Data from Here to client Application
                            }
                        }
                        else
                        {
                            //Incoming Call// To Do: Get Data from Here to client Application
                        }
                    }
                    else if (evtData.localConnectionInfo == LocalConnectionState.connected)
                    {
                        //Outgoing Call// To Do: Get Data from Here to client Application
                    }
                }
            }
            catch { }
        }

        private userLoginResponseLoginFailureCode GetFailureCode(OpenApiErrorCode errorValue)
        {
            switch (errorValue)
            {
                case OpenApiErrorCode.NONE:
                    return userLoginResponseLoginFailureCode.NONE;

                case OpenApiErrorCode.INVALID_CREDENTIALS:
                    return userLoginResponseLoginFailureCode.INVALID_CREDENTIALS;

                case OpenApiErrorCode.ALREADY_LOGED_IN:
                    return userLoginResponseLoginFailureCode.ALREADY_LOGED_IN;

                case OpenApiErrorCode.LICENSE_NOT_AVAILABLE:
                    return userLoginResponseLoginFailureCode.LICENSE_NOT_AVAILABLE;

                case OpenApiErrorCode.SERVER_PROBLEM:
                    return userLoginResponseLoginFailureCode.SERVER_PROBLEM;

                case OpenApiErrorCode.UNKNOWN:
                    return userLoginResponseLoginFailureCode.UNKNOWN;

                case OpenApiErrorCode.SYSTEM_STARTUP_IN_PROGRESS:
                    return userLoginResponseLoginFailureCode.SYSTEM_NOT_UP;

                case OpenApiErrorCode.SYSTEM_DOWN:
                    return userLoginResponseLoginFailureCode.SYSTEM_DOWN;

                case OpenApiErrorCode.USER_CONFIGURATION_PROBLEM:
                    return userLoginResponseLoginFailureCode.USER_CONFIGURATION_PROBLEM;

                case OpenApiErrorCode.CSTA_PROVIDER_NOT_AVAILABLE:
                    return userLoginResponseLoginFailureCode.CSTA_PROVIDER_NOT_AVAILABLE;

                case OpenApiErrorCode.CSTA_RESOURCE_NOT_AVAILABLE:
                    return userLoginResponseLoginFailureCode.CSTA_RESOURCE_NOT_AVAILABLE;

                case OpenApiErrorCode.SESSION_ERROR:
                    return userLoginResponseLoginFailureCode.SESSION_ERROR;

                case OpenApiErrorCode.UNAUTHORIZED:
                    return userLoginResponseLoginFailureCode.UNAUTHORIZED;

                case OpenApiErrorCode.UNSUPPORTED_APPLICATION_VERSION:
                    return userLoginResponseLoginFailureCode.UNAUTHORIZED;

                case OpenApiErrorCode.PREFFERED_EDITION_LICENSE_NOT_AVAILABLE:
                    return userLoginResponseLoginFailureCode.PREFERRED_EDITION_LICENSE_NOT_AVAILABLE;

                case OpenApiErrorCode.FORCE_NEW_PASSWORD:
                    return userLoginResponseLoginFailureCode.FORCE_NEW_PASSWORD;

                default:
                    return userLoginResponseLoginFailureCode.UNKNOWN;
            }
        }

        private bool CertificateValidationCallback(object sender, System.Security.Cryptography.X509Certificates.X509Certificate certification, System.Security.Cryptography.X509Certificates.X509Chain chain, SslPolicyErrors sslPolicyErrors)
        {
            bool retVal = false;
            bool sslCertError = false;
            try
            {
                if (((sslPolicyErrors & SslPolicyErrors.None) == SslPolicyErrors.None) ||
                            ((sslPolicyErrors & SslPolicyErrors.RemoteCertificateChainErrors) == SslPolicyErrors.RemoteCertificateChainErrors))
                {
                    sslCertError = false;
                }
                DateTime dtEnd = DateTime.Parse(certification.GetExpirationDateString());
                DateTime dtStart = DateTime.Parse(certification.GetEffectiveDateString());

                if (dtStart > DateTime.Now || dtEnd < DateTime.Now)
                {
                    retVal = false;
                    return retVal;
                }

                if (sslCertError == false)
                {
                    retVal = true;
                }
            }
            catch (Exception)
            {
                retVal = false;
            }

            return retVal;
        }

        private class LoginResponseDetails
        {
            public string sessionId { get; set; }
            public string serverVersion { get; set; }
            public userLoginResponseLoginFailureCode loginFailureCode { get; set; }
            public string webSocketWssUrl { get; set; }
            public string webSocketWsUrl { get; set; }
            public string serverResponse { get; set; }
            public bool isLoggedIn { get; set; }
            public string userExtension { get; set; }
            public string switchDevice { get; set; }
        }
    }

    public class ServerSettings
    {
        public string Server { get; set; }
        public string Port { get; set; }
        public bool Secure { get; set; }
        public string UserId { get; set; }
        public string Password { get; set; }
        public string AppVersion { get; set; }
    }

    public class Contact
    {
        public string Name { get; set; }
        public string Number { get; set; }
    }

    public class CallLog
    {
        public callType LogType { get; set; }
        public string Name { get; set; }
        public string Number { get; set; }
        public DateTime DateTime { get; set; }
        public int Duration { get; set; }
    }

    public enum HTTPMethod
    {
        GET,
        POST,
        DELETE,
        PUT
    }

    public class Deserilizer
    {
        public Object Deserlize(String eventType, String xmlData)
        {
            try
            {
                TextReader reader = new StringReader(xmlData);
                Type typeOfEvent = Type.GetType(eventType);

                System.Xml.Serialization.XmlSerializer eventSerializer = new System.Xml.Serialization.XmlSerializer(typeOfEvent);
                Object evt = (Object)eventSerializer.Deserialize(reader);
                return evt;
            }
            catch (Exception ex)
            {
                System.Diagnostics.Debug.Print(ex.ToString());
            }
            return null;
        }
    }

    public class Serializer
    {
        public String Serialize(Object obj)
        {
            System.Xml.Serialization.XmlSerializer serializer = new System.Xml.Serialization.XmlSerializer(obj.GetType());
            StringBuilder sb = new StringBuilder();
            TextWriter writer = new StringWriter(sb);
            serializer.Serialize(writer, obj);
            return (sb.ToString());
        }
    }
}
