/*****************************************************************************
 * © 2017 Avaya Inc. All rights reserved.
 ****************************************************************************/
package com.avaya.services.CallDeflection;

import org.apache.commons.lang3.StringUtils;

import com.avaya.collaboration.businessdata.api.NoAttributeFoundException;
import com.avaya.collaboration.businessdata.api.NoServiceProfileFoundException;
import com.avaya.collaboration.businessdata.api.NoUserFoundException;
import com.avaya.collaboration.businessdata.api.ServiceNotFoundException;
import com.avaya.collaboration.call.Call;
import com.avaya.collaboration.call.Participant;
import com.avaya.collaboration.call.ParticipantFactory;
import com.avaya.collaboration.util.logger.Logger;
import com.avaya.services.CallDeflection.email.CallDeflectionEmailSender;
import com.avaya.services.CallDeflection.sms.CallDeflectionSmsSender;
import com.avaya.services.CallDeflection.util.ServiceAttributeReader;
import com.avaya.services.CallDeflection.util.ServiceAttributeReaderImpl;

public final class CallDeflectionOperationsImpl implements CallDeflectionOperations
{
    private final ServiceAttributeReader serviceAttributeReader;
    private final CallDeflectionEmailSender callDeflectionEmailSender;
    private final CallDeflectionSmsSender callDeflectionSmsSender;
    private final Logger logger;

    public CallDeflectionOperationsImpl()
    {
        this(new ServiceAttributeReaderImpl(), new CallDeflectionEmailSender(), new CallDeflectionSmsSender(),
                Logger.getLogger(CallDeflectionOperationsImpl.class));
    }

    CallDeflectionOperationsImpl(final ServiceAttributeReader serviceAttributeReader,
            final CallDeflectionEmailSender callDeflectionEmailSender, final CallDeflectionSmsSender callDeflectionSmsSender,
            final Logger logger)
    {
        this.serviceAttributeReader = serviceAttributeReader;
        this.callDeflectionEmailSender = callDeflectionEmailSender;
        this.callDeflectionSmsSender = callDeflectionSmsSender;
        this.logger = logger;
    }

    /**
     * This method will add new party to a call.
     */
    @Override
    public void addPartyToCall(final Call call)
    {
        logger.finer("addPartyToCall ENTER");
        try
        {
            call.addParticipant(getPartyToAdd());
        }
        catch (final Exception e)
        {
            logger.error("addPartyToCall: failed operation exception=", e);
            call.drop();
        }
        logger.finer("addPartyToCall EXIT");
    }

    @Override
    public void dropCall(final Call call)
    {
        logger.finer("dropCall ENTER");
        try
        {
            call.drop();
        }
        catch (final Exception e)
        {
            logger.error("dropCall: failed operation exception=", e);
        }
        logger.finer("dropCall EXIT");
    }

    @Override
    public void sendEmail()
    {
        logger.finer("sendEmail ENTER");
        try
        {
            callDeflectionEmailSender.sendEmail();
        }
        catch (final NoAttributeFoundException | ServiceNotFoundException e)
        {
            logger.error("sendEmail : failed to send email=\n", e);
        }
        logger.finer("sendEmail EXIT");
    }

    @Override
    public void sendSms(final Call call)
    {
        logger.finer("sendSms ENTER");
        try
        {
            callDeflectionSmsSender.sendSms(call);
        }
        catch (final NoAttributeFoundException | ServiceNotFoundException | NoUserFoundException | NoServiceProfileFoundException e)
        {
            logger.error("sendSms : failed to send SMS=\n", e);
        }
        logger.finer("sendSms EXIT");
    }

    private Participant getPartyToAdd() throws NoUserFoundException,
            NoAttributeFoundException, ServiceNotFoundException, NoServiceProfileFoundException
    {
        logger.finer("getPartyToAdd ENTER");
        final Participant partyToAdd;
        final String agentNumber = serviceAttributeReader.getAgentNumber();
        if (StringUtils.isBlank(agentNumber))
        {
            throw new IllegalArgumentException("Agent Number cannot be blank");
        }
        else
        {
            final String domain = serviceAttributeReader.getAgentDomain();
            if (StringUtils.isBlank(domain))
            {
                partyToAdd = ParticipantFactory.create(agentNumber);
            }
            else
            {
                partyToAdd = ParticipantFactory.create(agentNumber, domain);
            }
            if (logger.isFinerEnabled())
            {
                logger.finer("getPartyToAdd " + partyToAdd);
            }
            return partyToAdd;
        }
    }
}
