/*****************************************************************************
 * © 2017 Avaya Inc. All rights reserved.
 ****************************************************************************/
package com.avaya.services.CallDeflection;

import static org.mockito.Mockito.doThrow;
import static org.mockito.Mockito.verify;
import static org.mockito.Mockito.when;
import static org.powermock.api.mockito.PowerMockito.mockStatic;

import org.junit.Before;
import org.junit.Test;
import org.junit.runner.RunWith;
import org.mockito.Mock;
import org.powermock.core.classloader.annotations.PrepareForTest;
import org.powermock.modules.junit4.PowerMockRunner;

import com.avaya.collaboration.businessdata.api.NoAttributeFoundException;
import com.avaya.collaboration.businessdata.api.NoServiceProfileFoundException;
import com.avaya.collaboration.businessdata.api.NoUserFoundException;
import com.avaya.collaboration.businessdata.api.ServiceNotFoundException;
import com.avaya.collaboration.call.Call;
import com.avaya.collaboration.call.Participant;
import com.avaya.collaboration.call.ParticipantFactory;
import com.avaya.collaboration.util.logger.Logger;
import com.avaya.services.CallDeflection.email.CallDeflectionEmailSender;
import com.avaya.services.CallDeflection.sms.CallDeflectionSmsSender;
import com.avaya.services.CallDeflection.util.ServiceAttributeReader;

@RunWith(PowerMockRunner.class)
@PrepareForTest(
{ ServiceAttributeReader.class, ParticipantFactory.class, Logger.class, CallDeflectionSmsSender.class, CallDeflectionEmailSender.class })
public final class CallDeflectionOperationsImplTest
{
    @Mock
    private ServiceAttributeReader serviceAttributeReader;
    @Mock
    private CallDeflectionEmailSender callDeflectionEmailSender;
    @Mock
    private CallDeflectionSmsSender callDeflectionSmsSender;
    @Mock
    private Logger logger;
    @Mock
    private Call call;

    @Mock
    private Participant participant;

    @Mock
    private NoAttributeFoundException exception;
    @Mock
    private IllegalStateException illegalStateException;

    CallDeflectionOperationsImpl callDeflectionOperationsImpl;

    @Before
    public void setupn()
    {
        callDeflectionOperationsImpl =
                new CallDeflectionOperationsImpl(serviceAttributeReader, callDeflectionEmailSender, callDeflectionSmsSender, logger);
    }

    @Test
    public void addPartyToCall()
            throws NoAttributeFoundException, ServiceNotFoundException, NoUserFoundException, NoServiceProfileFoundException
    {
        when(logger.isFinestEnabled()).thenReturn(true);
        when(logger.isFinerEnabled()).thenReturn(true);
        when(serviceAttributeReader.getAgentNumber()).thenReturn("111");
        mockStatic(ParticipantFactory.class);
        when(ParticipantFactory.create("111")).thenReturn(participant);
        callDeflectionOperationsImpl.addPartyToCall(call);
        verify(call).addParticipant(participant);
    }

    @Test
    public void addPartyToCallWithDomain()
            throws NoAttributeFoundException, ServiceNotFoundException, NoUserFoundException, NoServiceProfileFoundException
    {
        when(logger.isFinestEnabled()).thenReturn(true);
        when(logger.isFinerEnabled()).thenReturn(true);
        when(serviceAttributeReader.getAgentNumber()).thenReturn("111");
        when(serviceAttributeReader.getAgentDomain()).thenReturn("domain");
        mockStatic(ParticipantFactory.class);
        when(ParticipantFactory.create("111", "domain")).thenReturn(participant);
        callDeflectionOperationsImpl.addPartyToCall(call);
        verify(call).addParticipant(participant);
    }

    @Test
    public void addPartyToCallException()
            throws NoAttributeFoundException, ServiceNotFoundException, NoUserFoundException, NoServiceProfileFoundException
    {
        when(logger.isFinestEnabled()).thenReturn(true);
        when(logger.isFinerEnabled()).thenReturn(true);
        callDeflectionOperationsImpl.addPartyToCall(call);
        verify(call).drop();
    }

    @Test
    public void sendEmail() throws NoAttributeFoundException, ServiceNotFoundException
    {
        callDeflectionOperationsImpl.sendEmail();
        verify(callDeflectionEmailSender).sendEmail();
    }

    @Test
    public void sendEmailException() throws NoAttributeFoundException, ServiceNotFoundException
    {
        doThrow(exception).when(callDeflectionEmailSender).sendEmail();
        callDeflectionOperationsImpl.sendEmail();
        verify(logger).error("sendEmail : failed to send email=\n", exception);
    }

    @Test
    public void sendSms() throws NoAttributeFoundException, ServiceNotFoundException, NoUserFoundException, NoServiceProfileFoundException
    {
        callDeflectionOperationsImpl.sendSms(call);
        verify(callDeflectionSmsSender).sendSms(call);
    }

    @Test
    public void sendSmsException()
            throws NoAttributeFoundException, ServiceNotFoundException, NoUserFoundException, NoServiceProfileFoundException
    {
        doThrow(exception).when(callDeflectionSmsSender).sendSms(call);
        callDeflectionOperationsImpl.sendSms(call);
        verify(logger).error("sendSms : failed to send SMS=\n", exception);
    }

    @Test
    public void dropCall()
    {
        callDeflectionOperationsImpl.dropCall(call);
        verify(call).drop();
    }

    @Test
    public void dropCallException()
    {
        doThrow(illegalStateException).when(call).drop();
        callDeflectionOperationsImpl.dropCall(call);
        verify(logger).error("dropCall: failed operation exception=", illegalStateException);
    }

}