/****************************************************************************
 * Copyright Avaya Inc., All Rights Reserved.
 * THIS IS UNPUBLISHED PROPRIETARY SOURCE CODE OF Avaya Inc.
 * The copyright notice above does not evidence any actual or intended publication of such source code.
 * Some third-party source code components may have been modified from their original versions by Avaya Inc.
 * The modifications are Copyright Avaya Inc., All Rights Reserved.
 * Avaya Confidential & Restricted. May not be distributed further without written permission of
 * the Avaya owner.
 ****************************************************************************/
package com.avaya.zephyr.services.sample_services.Authorization.listener;

import com.avaya.collaboration.businessdata.api.NoAttributeFoundException;
import com.avaya.collaboration.businessdata.api.ServiceData;
import com.avaya.collaboration.businessdata.api.ServiceNotFoundException;
import com.avaya.collaboration.dal.factory.CollaborationDataFactory;
import com.avaya.collaboration.util.logger.Logger;
import com.avaya.zephyr.platform.dal.api.ServiceUtil;
import com.avaya.zephyr.platform.dal.api.objectapi.ClusterDefaultAttribute;
import com.avaya.zephyr.platform.dal.api.objectapi.DefaultAttribute;
import com.avaya.zephyr.platform.dm.AbstractDMListener;
import com.avaya.zephyr.services.sample_services.Authorization.util.ServiceAttributeDefaults;
import com.avaya.zephyr.services.sample_services.Authorization.util.ServiceAttributeStore;

public final class AttributeChangeListener extends AbstractDMListener
{
    private final ServiceData serviceData;
    private final ServiceAttributeStore serviceAttributeStore;
    private final Logger logger;

    public AttributeChangeListener()
    {
        this(CollaborationDataFactory.getServiceData(ServiceUtil.getServiceDescriptor().getName(),
                ServiceUtil.getServiceDescriptor().getVersion()),
                ServiceAttributeStore.getInstance(),
                Logger.getLogger(AttributeChangeListener.class));
    }

    AttributeChangeListener(final ServiceData serviceData, final ServiceAttributeStore serviceAttributeStore,
            final Logger logger)
    {
        this.serviceData = serviceData;
        this.serviceAttributeStore = serviceAttributeStore;
        this.logger = logger;
    }

    @Override
    public void objectChanged(final Object oldObject, final Object newObject)
    {
        if (oldObject instanceof DefaultAttribute || newObject instanceof DefaultAttribute)
        {
            final DefaultAttribute defaultAttribute;
            if (oldObject != null)
            {
                defaultAttribute = (DefaultAttribute) oldObject;
            }
            else
            {
                defaultAttribute = (DefaultAttribute) newObject;
            }
            updateAttributeValue(defaultAttribute.getAttributeName());
        }
        else if (oldObject instanceof ClusterDefaultAttribute || newObject instanceof ClusterDefaultAttribute)
        {
            final ClusterDefaultAttribute clusterAttribute;
            if (oldObject != null)
            {
                clusterAttribute = (ClusterDefaultAttribute) oldObject;
            }
            else
            {
                clusterAttribute = (ClusterDefaultAttribute) newObject;
            }
            updateAttributeValue(clusterAttribute.getAttributeName());
        }
    }

    private void updateAttributeValue(final String attributeName)
    {
        if (logger.isFinestEnabled())
        {
            logger.finest("AttributeChangeListener updateAttributeValue : " +
                    "updating attribute value for : " + attributeName);
        }
        try
        {
            if (attributeName.equals(ServiceAttributeDefaults.ENABLE_TOKENLESS_ACCESS.getAttributeName()))
            {
                updateEnableTokenlessAccess(attributeName);
            }
            else if (attributeName.equals(ServiceAttributeDefaults.CLIENT_REDIRECTION_PATH.getAttributeName()))
            {
                updateClientRedirectionPath(attributeName);
            }
            else if (attributeName.equals(ServiceAttributeDefaults.SESSION_COOKIE_NAME.getAttributeName()))
            {
                updateSessionCookieName(attributeName);
            }
        }
        catch (final NoAttributeFoundException e)
        {
            logger.error("AttributeChangeListener updateAttributeValue : " +
                    "Error while retrieving value of attribute " + attributeName, e);
        }
        catch (final ServiceNotFoundException e)
        {
            logger.error("AttributeChangeListener updateAttributeValue : Service not found ", e);
        }
    }

    private void updateSessionCookieName(final String attributeName) throws NoAttributeFoundException, ServiceNotFoundException
    {
        serviceAttributeStore.setSessionCookieName(serviceData.getServiceAttribute
                (ServiceAttributeDefaults.SESSION_COOKIE_NAME.getAttributeName()));
    }

    private void updateClientRedirectionPath(final String attributeName) throws NoAttributeFoundException, ServiceNotFoundException
    {
        serviceAttributeStore.setSessionCookieName(serviceData.getServiceAttribute
                (ServiceAttributeDefaults.CLIENT_REDIRECTION_PATH.getAttributeName()));
    }

    private void updateEnableTokenlessAccess(final String attributeName) throws NoAttributeFoundException, ServiceNotFoundException
    {
        serviceAttributeStore.setEnableTokenlessAccess(Boolean.valueOf(serviceData.getServiceAttribute
                (ServiceAttributeDefaults.ENABLE_TOKENLESS_ACCESS.getAttributeName())));
    }
}
