///////////////////////////////////////////////////////////////////////////////
//// DynamicTeamFormationServlet.java
////
////Description: DynamicTeamFormationServlet initiates video conference followed by SMS and email with conference details.
////
//// Copyright 2013 Avaya Inc. All rights reserved.
//// Usage of this source is bound to the terms described
//// in http://www.avaya.com/devconnect
//// Avaya - Confidential & Proprietary. Use pursuant to your signed agreement or Avaya Policy.
/////////////////////////////////////////////////////////////////////////////////

package com.avaya.zephyr.services.dynamicteamformation;

import java.io.IOException;
import java.io.PrintWriter;

import javax.servlet.ServletException;
import javax.servlet.annotation.WebServlet;
import javax.servlet.http.HttpServlet;
import javax.servlet.http.HttpServletRequest;
import javax.servlet.http.HttpServletResponse;

import com.avaya.collaboration.businessdata.api.ServiceData;
import com.avaya.collaboration.conference.scheduled.SchedConf;
import com.avaya.collaboration.dal.factory.CollaborationDataFactory;
import com.avaya.collaboration.util.logger.Logger;
import com.avaya.zephyr.platform.dal.api.ServiceDescriptor;
import com.avaya.zephyr.platform.dal.api.ServiceUtil;
import com.avaya.zephyr.services.dynamicteamformation.conference.DynamicTeamFormationConferenceScheduler;
import com.avaya.zephyr.services.dynamicteamformation.email.DynamicTeamFormationEmailSender;
import com.avaya.zephyr.services.dynamicteamformation.sms.DynamicTeamFormationSmsSender;

/**
 * Servlet implementation class DynamicTeamFormationServlet
 */
@WebServlet("/DynamicTeamFormationServlet")
public class DynamicTeamFormationServlet extends HttpServlet
{
    private static final long serialVersionUID = 781607438398942140L;
    private final transient Logger logger;

    /**
     * @see HttpServlet#HttpServlet()
     */
    public DynamicTeamFormationServlet()
    {
        super();
        this.logger = Logger.getLogger(DynamicTeamFormationServlet.class);
    }

    DynamicTeamFormationServlet(final Logger logger)
    {
        super();
        this.logger = logger;
    }

    /**
     * @see HttpServlet#doPost(HttpServletRequest request, HttpServletResponse
     *      response)
     */
    @Override
    protected void doPost(final HttpServletRequest httpServletRequest,
            final HttpServletResponse httpServletResponse) throws ServletException, IOException
    {
        logger.info("doPost");

        final PrintWriter printWriter = httpServletResponse.getWriter();
        httpServletResponse.setContentType("text/plain");

        // Get service descriptor to allow getting of service attributes from
        // SMGR GUI
        final ServiceDescriptor serviceDescriptor = ServiceUtil.getServiceDescriptor();
        ServiceData serviceData = null;
        if (serviceDescriptor != null)
        {
            serviceData = CollaborationDataFactory.getServiceData(serviceDescriptor.getName(),
                    serviceDescriptor.getVersion());
        }
        else
        {
            logger.error("Cant get service descriptor");
            printWriter.println("Error occured getting the service descriptor");
            httpServletResponse.setStatus(HttpServletResponse.SC_INTERNAL_SERVER_ERROR);
            return;
        }

        final SchedConf conference;

        // CONFERENCE
        try
        {
            final DynamicTeamFormationConferenceScheduler conferenceScheduler =
                    new DynamicTeamFormationConferenceScheduler(serviceData, httpServletRequest,
                            printWriter);
            conference = conferenceScheduler.scheduleConference();
        }
        catch (final Exception e)
        {
            logger.error("Error: Error scheduling the conference.", e);
            printWriter.println("Error: Error scheduling the conference.");
            httpServletResponse.setStatus(HttpServletResponse.SC_INTERNAL_SERVER_ERROR);
            return;
        }

        // NOTIFICATION EMAIL
        final DynamicTeamFormationEmailSender emailSender =
                new DynamicTeamFormationEmailSender(serviceData, httpServletRequest, printWriter);
        final boolean sentEmail = emailSender.sendEmail(conference);

        // NOTIFICATION SMS
        final DynamicTeamFormationSmsSender smsSender =
                new DynamicTeamFormationSmsSender(serviceData, httpServletRequest, printWriter);
        final boolean sentSms = smsSender.sendSms(conference);

        // If no email or sms can be sent, notify it was a bad request.
        if (!sentEmail && !sentSms)
        {
            httpServletResponse.setStatus(HttpServletResponse.SC_INTERNAL_SERVER_ERROR);
        }
        else
        {
            httpServletResponse.setStatus(HttpServletResponse.SC_OK);
        }
    }

    /**
     * @see HttpServlet#doGet(HttpServletRequest request, HttpServletResponse
     *      response)
     */
    @Override
    protected void doGet(final HttpServletRequest request, final HttpServletResponse response)
            throws ServletException, IOException
    {
        logger.info("doGet");
        doPost(request, response);
    }

}
