///////////////////////////////////////////////////////////////////////////////
//// DynamicTeamFormationConferenceScheduler.java
////
//// Description: DynamicTeamFormationConferenceScheduler creates a video conference.
//// Copyright 2013 Avaya Inc. All rights reserved.
//// Usage of this source is bound to the terms described
//// in http://www.avaya.com/devconnect
//// Avaya - Confidential & Proprietary. Use pursuant to your signed agreement or Avaya Policy.
/////////////////////////////////////////////////////////////////////////////////

package com.avaya.zephyr.services.dynamicteamformation.conference;

import java.io.PrintWriter;
import java.util.Calendar;

import javax.servlet.http.HttpServletRequest;

import com.avaya.collaboration.businessdata.api.ServiceData;
import com.avaya.collaboration.conference.scheduled.SchedConf;
import com.avaya.collaboration.conference.scheduled.SchedConfFactory;
import com.avaya.collaboration.util.logger.Logger;

public class DynamicTeamFormationConferenceScheduler
{
    private final Logger logger;
    private ServiceData serviceData = null;
    private HttpServletRequest httpServletRequest = null;
    private PrintWriter printWriter = null;

    public DynamicTeamFormationConferenceScheduler(final ServiceData serviceData,
            final HttpServletRequest httpServletRequest,
            final PrintWriter printWriter)
    {
        this.serviceData = serviceData;
        this.httpServletRequest = httpServletRequest;
        this.printWriter = printWriter;
        this.logger = Logger.getLogger(DynamicTeamFormationConferenceScheduler.class);
    }

    // Junit Constructor
    public DynamicTeamFormationConferenceScheduler(final ServiceData serviceData,
            final HttpServletRequest httpServletRequest,
            final PrintWriter printWriter, final Logger logger)
    {
        this.serviceData = serviceData;
        this.httpServletRequest = httpServletRequest;
        this.printWriter = printWriter;
        this.logger = logger;
    }

    private static final String CONF_PARAM_SMGR_DURATION = "conferenceMaxDurationHours";
    private static final String CONF_ERROR_DURATION_LOG =
                "Error: Error occured acquiring 'Conference Maximum Duration' from SMGR. Defaulting to 2 hours.";
    private static final String CONF_INFO_NO_CONF_DURATION_LOG =
                "Info: No 'Conference Maximum Duration' specified in the DynamicTeamFormationService attribute page on SMGR. Defaulting to 2 hours.";
    private static final String CONF_INFO_TOO_LARGE_CONF_DURATION_LOG =
                "Info: 'Conference Maximum Duration' specified in the DynamicTeamFormationService attribute page on SMGR is too large. Setting to maximum 720 hours.";
    private static final int DEFAULT_DURATION_IN_HOURS = 2;
    private static final int MAX_DURATION_IN_HOURS = 720;

    private Integer getMaximumDurationOfConferenceInHours()
    {
        // Required - From SMGR GUI
        Integer conferenceDurationInHours = 0;

        try
        {
            final String conferenceDurationInHoursString =
                    serviceData.getServiceAttribute(CONF_PARAM_SMGR_DURATION);

            if (conferenceDurationInHoursString.isEmpty())
            {
                logHttpResponseInfo(CONF_INFO_NO_CONF_DURATION_LOG);
                conferenceDurationInHours = DEFAULT_DURATION_IN_HOURS;
            }
            else
            {
                conferenceDurationInHours = Integer.parseInt(conferenceDurationInHoursString);

                if (conferenceDurationInHours > MAX_DURATION_IN_HOURS)
                {
                    logHttpResponseInfo(CONF_INFO_TOO_LARGE_CONF_DURATION_LOG);
                    conferenceDurationInHours = MAX_DURATION_IN_HOURS;
                }
            }
        }
        catch (final Exception e)
        {
            logHttpResponseError(CONF_ERROR_DURATION_LOG, e);
        }

        return conferenceDurationInHours;
    }

    private static final String CONF_PARAM_HTTP_SUBJECT = "conferenceSubject";
    private static final String CONF_INFO_NO_SUBJECT_LOG = "Info: No conference subject specified in HTTP request.";

    private String getConferenceSubject()
    {
        // Required - From HTTP request - Only 1st with data used
        String conferenceSubject = "";

        final String[] conferenceSubjectFromHttpRequest =
                httpServletRequest.getParameterValues(CONF_PARAM_HTTP_SUBJECT);

        if (conferenceSubjectFromHttpRequest != null)
        {
            for (final String conferenceSubjectEntry : conferenceSubjectFromHttpRequest)
            {
                if (!conferenceSubjectEntry.isEmpty())
                {
                    conferenceSubject = conferenceSubjectEntry;
                    break;
                }
            }
        }

        if (conferenceSubject.isEmpty())
        {
            logHttpResponseInfo(CONF_INFO_NO_SUBJECT_LOG);
            conferenceSubject = "No conference subject";
        }

        return conferenceSubject;
    }

    private static final String CONF_PARAM_HTTP_DESCRIPTION = "conferenceDescription";
    private static final String CONF_INFO_NO_DESCRIPTION_LOG =
                "Info: No conference description specified in HTTP request.";

    private String getConferenceDescription()
    {
        // Required - From HTTP request - Only 1st with data used
        String conferenceDescription = "";

        final String[] conferenceDescriptionFromHttpRequest =
                httpServletRequest.getParameterValues(CONF_PARAM_HTTP_DESCRIPTION);

        if (conferenceDescriptionFromHttpRequest != null)
        {
            for (final String conferenceDescriptiontEntry : conferenceDescriptionFromHttpRequest)
            {
                if (!conferenceDescriptiontEntry.isEmpty())
                {
                    conferenceDescription = conferenceDescriptiontEntry;
                    break;
                }
            }
        }

        if (conferenceDescription.isEmpty())
        {
            logHttpResponseInfo(CONF_INFO_NO_DESCRIPTION_LOG);
            conferenceDescription = "No conference subject";
        }

        return conferenceDescription;
    }

    public SchedConf scheduleConference() throws Exception
    {
        // Conference Duration (Required) - From SMGR GUI
        final Integer conferenceDuration = getMaximumDurationOfConferenceInHours();
        // Conference Subject (Not required) - From HTTP request.
        final String conferenceSubject = getConferenceSubject();
        // Conference Description (Not required) - From HTTP request.
        final String conferenceDescription = getConferenceDescription();

        final SchedConf conference = SchedConfFactory.createConf();
        conference.setSubject(conferenceSubject);
        conference.setDescription(conferenceDescription);
        conference.setStartTime(Calendar.getInstance());
        conference.setDuration(0, conferenceDuration, 0);
        conference.schedule();

        return conference;
    }

    private void logHttpResponseInfo(final String log)
    {
        printWriter.println(log);
        logger.info(log);
    }

    private void logHttpResponseError(final String log, final Exception e)
    {
        printWriter.println(log);
        logger.error(log, e);
    }
}
