package com.avaya.zephyr.services.multichanbroadcast.announcementcall;

import static org.easymock.EasyMock.anyObject;
import static org.easymock.EasyMock.expect;
import static org.junit.Assert.assertFalse;
import static org.junit.Assert.assertTrue;
import static org.powermock.api.easymock.PowerMock.createMock;
import static org.powermock.api.easymock.PowerMock.expectLastCall;
import static org.powermock.api.easymock.PowerMock.mockStatic;
import static org.powermock.api.easymock.PowerMock.replayAll;
import static org.powermock.api.easymock.PowerMock.verifyAll;

import java.io.PrintWriter;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.List;

import javax.servlet.http.HttpServletRequest;

import org.junit.Before;
import org.junit.Test;
import org.junit.runner.RunWith;
import org.powermock.core.classloader.annotations.PrepareForTest;
import org.powermock.modules.junit4.PowerMockRunner;

import com.avaya.collaboration.businessdata.api.NoAttributeFoundException;
import com.avaya.collaboration.businessdata.api.ServiceData;
import com.avaya.collaboration.businessdata.api.ServiceNotFoundException;
import com.avaya.collaboration.call.Call;
import com.avaya.collaboration.call.CallFactory;
import com.avaya.collaboration.call.Participant;
import com.avaya.collaboration.call.ParticipantFactory;
import com.avaya.collaboration.util.logger.Logger;

@RunWith(PowerMockRunner.class)
@PrepareForTest(
{ ParticipantFactory.class, CallFactory.class, HttpServletRequest.class,
        MultiChanBroadcastAnnouncementSender.class })
public class MultiChanBroadcastAnnouncementSenderTest
{

    private static final String ANNOUNCEMENT_PARAM_HTTP_FILE_URI = "announcementFileUri";
    private static final String ANNOUNCEMENT_INFO_NO_FILE_URI_LOG =
            "Info: No announcement specified in the HTTP request.";
    private static final String ANNOUNCEMENT_INFO_NOT_SENT_LOG = "Info: Announcements will not be sent.";
    private static final String ANNOUNCEMENT_PARAM_HTTP_TO = "announcementTo";
    private static final String ANNOUNCEMENT_INFO_NO_TO_LOG =
            "Info: No announcement recipients specified in HTTP request.";
    private static final String ANNOUNCEMENT_PARAM_SMGR_CALLING_NUMBER = "announcementCallingNumber";
    private static final String ANNOUNCEMENT_ERROR_CALLING_NUMBER_LOG =
            "Error: Error occured acquiring 'announcementCallingNumber'.";
    private static final String ANNOUNCEMENT_INFO_NO_CALLING_NUMBER_LOG =
            "Info: No calling handle defined in the MultiChanBroadcastService attribute page on SMGR.";
    private static final String ANNOUNCEMENT_PARAM_SMGR_CALLING_DOMAIN = "announcementCallingDomian";
    private static final String ANNOUNCEMENT_ERROR_CALLING_DOMAIN_LOG =
            "Error: Error occured acquiring 'announcementCallingDomian'.";
    private static final String ANNOUNCEMENT_INFO_NO_CALLING_DOMAIN_LOG =
            "Info: No calling domain defined in the MultiChanBroadcastService attribute page on SMGR.";
    private static final String ANNOUNCEMENT_PARAM_SMGR_CALLING_DISPLAY = "announcementCallingName";
    private static final String ANNOUNCEMENT_ERROR_CALLING_DISPLAY_LOG =
            "Error: Error occured acquiring 'announcementCallingName'.";
    private static final String ANNOUNCEMENT_PARAM_SMGR_PROMPTANDCOLLECT_TIMEOUT =
            "announcementPromptAndCollectTimeout";
    private static final String ANNOUNCEMENT_ERROR_PROMPTANDCOLLECT_TIMEOUT_LOG =
            "Error: Error occured acquiring 'announcementPromptAndCollectTimeout'. Defaulting to 90 seconds.";
    private static final String ANNOUNCEMENT_PARAM_CALL_FILE_URIS = ANNOUNCEMENT_PARAM_HTTP_FILE_URI + "s";
    private static final String ANNOUNCEMENT_PARAM_CALL_PROMPTANDCOLLECT_TIMEOUT =
            "announcementPromptAndCollectTimeout";

    private MultiChanBroadcastAnnouncementSender multiChanBroadcastAnnouncementSenderTest;
    private Logger mockLogger;
    private PrintWriter mockPrintWriter;
    private HttpServletRequest mockHttpServletRequest;
    private ServiceData mockServiceData;
    private Participant mockCeParticipant;
    private Call mockAnnouncementCall;

    @Before
    public void setUpMockCommon() throws Exception
    {
        mockLogger = createMock(Logger.class);
        mockPrintWriter = createMock(PrintWriter.class);
        mockServiceData = createMock(ServiceData.class);
        mockHttpServletRequest = createMock(HttpServletRequest.class);
        mockCeParticipant = createMock(Participant.class);
        mockAnnouncementCall = createMock(Call.class);

        multiChanBroadcastAnnouncementSenderTest =
                new MultiChanBroadcastAnnouncementSender(mockServiceData, mockHttpServletRequest,
                        mockPrintWriter, mockLogger);
    }

    private void setupMockAnnouncementFileUri(String[] announcementFileUri)
    {
        expect(
                mockHttpServletRequest
                        .getParameterValues(ANNOUNCEMENT_PARAM_HTTP_FILE_URI))
                .andReturn(announcementFileUri);

        boolean fileUriEveryEntryEmpty = true;
        if (announcementFileUri != null)
        {
            for (String announcementToEntry : announcementFileUri)
            {
                if (!announcementToEntry.isEmpty())
                {
                    fileUriEveryEntryEmpty = false;
                }
            }
        }
        if (announcementFileUri == null || fileUriEveryEntryEmpty)
        {
            setupMockLogHttpResponseInfo(ANNOUNCEMENT_INFO_NO_FILE_URI_LOG);
            setupMockLogHttpResponseInfo(ANNOUNCEMENT_INFO_NOT_SENT_LOG);
        }
    }

    private void setupMockAnnouncementTo(String[] announcementTo)
    {
        expect(mockHttpServletRequest
                .getParameterValues(ANNOUNCEMENT_PARAM_HTTP_TO))
                .andReturn(announcementTo);

        boolean announcementToEveryEntryEmpty = true;
        if (announcementTo != null)
        {
            for (String announcementToEntry : announcementTo)
            {
                if (!announcementToEntry.isEmpty())
                {
                    announcementToEveryEntryEmpty = false;
                }
            }
        }
        if (announcementTo == null || announcementToEveryEntryEmpty)
        {
            setupMockLogHttpResponseInfo(ANNOUNCEMENT_INFO_NO_TO_LOG);
            setupMockLogHttpResponseInfo(ANNOUNCEMENT_INFO_NOT_SENT_LOG);
        }
    }


    private void setupMockAnnouncementCallingHandle(String announcementCallingHandle)
            throws NoAttributeFoundException, ServiceNotFoundException
    {
        if (announcementCallingHandle == null)
        {
            expect(
                    mockServiceData
                            .getServiceAttribute(ANNOUNCEMENT_PARAM_SMGR_CALLING_NUMBER))
                    .andThrow(new NoAttributeFoundException(
                            ANNOUNCEMENT_PARAM_SMGR_CALLING_NUMBER));
            setupMockLogHttpResponseError(ANNOUNCEMENT_ERROR_CALLING_NUMBER_LOG);
            setupMockLogHttpResponseInfo(ANNOUNCEMENT_INFO_NOT_SENT_LOG);
        }
        else
        {
            expect(
                    mockServiceData
                            .getServiceAttribute(ANNOUNCEMENT_PARAM_SMGR_CALLING_NUMBER))
                    .andReturn(announcementCallingHandle);
            if (announcementCallingHandle.isEmpty())
            {
                setupMockLogHttpResponseInfo(ANNOUNCEMENT_INFO_NO_CALLING_NUMBER_LOG);
                setupMockLogHttpResponseInfo(ANNOUNCEMENT_INFO_NOT_SENT_LOG);
            }
        }
    }

    private void setupMockAnnouncementCallingDomain(String announcementCallingDomain)
            throws NoAttributeFoundException, ServiceNotFoundException
    {
        if (announcementCallingDomain == null)
        {
            expect(
                    mockServiceData
                            .getServiceAttribute(ANNOUNCEMENT_PARAM_SMGR_CALLING_DOMAIN))
                    .andThrow(new NoAttributeFoundException(
                            ANNOUNCEMENT_PARAM_SMGR_CALLING_DOMAIN));
            setupMockLogHttpResponseError(ANNOUNCEMENT_ERROR_CALLING_DOMAIN_LOG);
            setupMockLogHttpResponseInfo(ANNOUNCEMENT_INFO_NOT_SENT_LOG);
        }
        else
        {
            expect(
                    mockServiceData
                            .getServiceAttribute(ANNOUNCEMENT_PARAM_SMGR_CALLING_DOMAIN))
                    .andReturn(announcementCallingDomain);
            if (announcementCallingDomain.isEmpty())
            {
                setupMockLogHttpResponseInfo(ANNOUNCEMENT_INFO_NO_CALLING_DOMAIN_LOG);
                setupMockLogHttpResponseInfo(ANNOUNCEMENT_INFO_NOT_SENT_LOG);
            }
        }
    }

    private void setupMockAnnouncementCallingDisplay(String announcementCallingDisplay)
            throws NoAttributeFoundException, ServiceNotFoundException
    {
        if (announcementCallingDisplay == null)
        {
            expect(
                    mockServiceData
                            .getServiceAttribute(ANNOUNCEMENT_PARAM_SMGR_CALLING_DISPLAY))
                    .andThrow(new NoAttributeFoundException(
                            ANNOUNCEMENT_PARAM_SMGR_CALLING_DISPLAY));
            setupMockLogHttpResponseError(ANNOUNCEMENT_ERROR_CALLING_DISPLAY_LOG);
        }
        else
        {
            expect(
                    mockServiceData
                            .getServiceAttribute(ANNOUNCEMENT_PARAM_SMGR_CALLING_DISPLAY))
                    .andReturn(announcementCallingDisplay);
        }
    }

    private void setupMockPromptAndCollectTimeout(String promptAndCollectTimeout)
            throws NoAttributeFoundException, ServiceNotFoundException
    {
        if (promptAndCollectTimeout == null)
        {
            expect(
                    mockServiceData
                            .getServiceAttribute(ANNOUNCEMENT_PARAM_SMGR_PROMPTANDCOLLECT_TIMEOUT))
                    .andThrow(
                            new NoAttributeFoundException(
                                    ANNOUNCEMENT_PARAM_SMGR_PROMPTANDCOLLECT_TIMEOUT));
            setupMockLogHttpResponseError(ANNOUNCEMENT_ERROR_PROMPTANDCOLLECT_TIMEOUT_LOG);
        }
        else
        {
            expect(
                    mockServiceData
                            .getServiceAttribute(ANNOUNCEMENT_PARAM_SMGR_PROMPTANDCOLLECT_TIMEOUT))
                    .andReturn(promptAndCollectTimeout);
        }
    }

    private void setupMockAnnouncementCall(List<String> announcementTosSplitAndConcatDefaultDomain,
            ArrayList<String> announcementFileUris, Integer promptAndCollectTimeout)
            throws Exception
    {
        mockStatic(ParticipantFactory.class);
        expect(ParticipantFactory.create((String) anyObject(), (String) anyObject(),
                (String) anyObject())).andReturn(mockCeParticipant);
        mockStatic(CallFactory.class);
        for (final String announcementTo : announcementTosSplitAndConcatDefaultDomain)
        {
            expect(CallFactory.create(mockCeParticipant, announcementTo))
                    .andReturn(mockAnnouncementCall);
            expect(mockAnnouncementCall.setAttribute(
                    ANNOUNCEMENT_PARAM_CALL_FILE_URIS,
                    announcementFileUris)).andReturn(mockAnnouncementCall);
            expect(
                    mockAnnouncementCall
                            .setAttribute(
                                    ANNOUNCEMENT_PARAM_CALL_PROMPTANDCOLLECT_TIMEOUT,
                                    promptAndCollectTimeout)).andReturn(mockAnnouncementCall);
            mockAnnouncementCall.initiate();
            expectLastCall();
            expectLastCall();
            setupMockLogHttpResponseInfo(null);
        }
    }

    private void setupMockLogHttpResponseInfo(String logInfo)
    {
        if (logInfo == null)
        {
            mockLogger.info((String) anyObject());
            expectLastCall();
            mockPrintWriter.println((String) anyObject());
            expectLastCall();
        }
        else
        {
            mockLogger.info(logInfo);
            expectLastCall();
            mockPrintWriter.println(logInfo);
            expectLastCall();
        }
    }

    private void setupMockLogHttpResponseError(String logError)
    {
        if (logError == null)
        {
            mockLogger.error((String) anyObject());
            expectLastCall();
            mockPrintWriter.println((String) anyObject());
            expectLastCall();
        }
        else
        {
            mockLogger.error((String) anyObject(), (Exception) anyObject());
            expectLastCall();
            mockPrintWriter.println(logError);
            expectLastCall();
        }
    }

    @Test
    public void testSendAnnouncement() throws Exception
    {
        // Test Info
        // sendAnnouncement needs the HTTP request to have both the the 'To',
        // FileUri', and the SMGR attribute fields 'Calling Handle' and 'Calling
        // Display' fields to be present

        // Test Data
        final String[] announcementFileUriAsArray =
        { "file:///somefilelocation" };
        ArrayList<String> announcementFileUri = new ArrayList<String>();
        announcementFileUri.add("file:///somefilelocation");
        final String[] announcementToMultipleCommaMixed =
                { "1111111@domain.com", "2222222@domain.com,3333333@domain.com", "4444444@defaultDomain.com",
                        "5555555@defaultDomain.com,6666666@defaultDomain.com",
                        "7777777@domain.com,8888888@defaultDomain.com" };
        final List<String> announcementTosSplitAndConcatDefaultDomain =
                Arrays.asList("1111111@domain.com", "2222222@domain.com", "3333333@domain.com",
                        "4444444@defaultDomain.com",
                        "5555555@defaultDomain.com", "6666666@defaultDomain.com",
                        "7777777@domain.com", "8888888@defaultDomain.com");
        final String announcementCallingHandle = "testHandle";
        final String announcementCallingDomain = "testDomain.com";
        final String announcementCallingDisplay = "Test Calling Display";
        final String promptAndCollectTimeout = "120";
        final Integer promptAndCollectTimeoutMilli = 120000;

        // Setup mocks
        setupMockAnnouncementFileUri(announcementFileUriAsArray);
        setupMockAnnouncementTo(announcementToMultipleCommaMixed);
        setupMockAnnouncementCallingHandle(announcementCallingHandle);
        setupMockAnnouncementCallingDomain(announcementCallingDomain);
        setupMockAnnouncementCallingDisplay(announcementCallingDisplay);
        setupMockPromptAndCollectTimeout(promptAndCollectTimeout);
        setupMockAnnouncementCall(announcementTosSplitAndConcatDefaultDomain,
                announcementFileUri,
                promptAndCollectTimeoutMilli);

        // Replay, Execute, and Verify
        replayAll();
        assertTrue(multiChanBroadcastAnnouncementSenderTest.sendAnnouncement());
        verifyAll();
    }

    @Test
    public void testSendAnnouncementFileUriNull() throws Exception
    {
        // Test Info
        // sendAnnouncement needs the HTTP request to have both the the 'To' and
        // 'FileUri' fields to be present

        // Test Data
        final String[] announcementFileUriAsArray = null;
        final String[] announcementToMultipleCommaMixed =
        { "1111111@domain.com", "2222222@domain.com,3333333@domain.com", "4444444", "5555555,6666666",
                "7777777@domain.com,8888888" };

        // Setup mocks
        setupMockAnnouncementFileUri(announcementFileUriAsArray);
        setupMockAnnouncementTo(announcementToMultipleCommaMixed);

        // Replay, Execute, and Verify
        replayAll();
        assertFalse(multiChanBroadcastAnnouncementSenderTest.sendAnnouncement());
        verifyAll();
    }

    @Test
    public void testSendAnnouncementFileUriEmpty() throws Exception
    {
        // Test Info
        // sendAnnouncement needs the HTTP request to have both the the 'To' and
        // 'FileUri' fields to be present

        // Test Data
        final String[] announcementFileUriAsArray =
        { "" };
        final String[] announcementToMultipleCommaMixed =
        { "1111111@domain.com", "2222222@domain.com,3333333@domain.com", "4444444", "5555555,6666666",
                "7777777@domain.com,8888888" };

        // Setup mocks
        setupMockAnnouncementFileUri(announcementFileUriAsArray);
        setupMockAnnouncementTo(announcementToMultipleCommaMixed);

        // Replay, Execute, and Verify
        replayAll();
        assertFalse(multiChanBroadcastAnnouncementSenderTest.sendAnnouncement());
        verifyAll();
    }

    @Test
    public void testSendAnnouncementToNull() throws Exception
    {
        // Test Info
        // sendAnnouncement needs the HTTP request to have both the the 'To' and
        // 'FileUri' fields to be present

        // Test Data
        final String[] announcementFileUriAsArray =
        { "file:///somefilelocation" };
        final String[] announcementToNull = null;

        // Setup mocks
        setupMockAnnouncementFileUri(announcementFileUriAsArray);
        setupMockAnnouncementTo(announcementToNull);

        // Replay, Execute, and Verify
        replayAll();
        assertFalse(multiChanBroadcastAnnouncementSenderTest.sendAnnouncement());
        verifyAll();
    }

    @Test
    public void testSendAnnouncementToEmpty() throws Exception
    {
        // Test Info
        // sendAnnouncement needs the HTTP request to have both the the 'To' and
        // 'FileUri' fields to be present

        // Test Data
        final String[] announcementFileUriAsArray =
        { "file:///somefilelocation" };
        final String[] announcementToNull =
        { "" };

        // Setup mocks
        setupMockAnnouncementFileUri(announcementFileUriAsArray);
        setupMockAnnouncementTo(announcementToNull);

        // Replay, Execute, and Verify
        replayAll();
        assertFalse(multiChanBroadcastAnnouncementSenderTest.sendAnnouncement());
        verifyAll();
    }

    @Test
    public void testSendAnnouncementCallingHandleException() throws Exception
    {
        // Test Info
        // sendAnnouncement needs the HTTP request to have both the the 'To',
        // FileUri', and the SMGR attribute fields 'Calling Handle' and 'Calling
        // Display' fields to be present

        // Test Data
        final String[] announcementFileUriAsArray =
        { "file:///somefilelocation" };
        final String[] announcementToMultipleCommaMixed =
        { "1111111@domain.com", "2222222@domain.com,3333333@domain.com", "4444444", "5555555,6666666",
                "7777777@domain.com,8888888" };
        final String announcementCallingHandle = null;
        final String announcementCallingDomain = "testDomain.com";
        final String announcementCallingDisplay = "test display";
        final String promptAndCollectTimeout = "120";

        // Setup mocks
        setupMockAnnouncementFileUri(announcementFileUriAsArray);
        setupMockAnnouncementTo(announcementToMultipleCommaMixed);
        setupMockAnnouncementCallingHandle(announcementCallingHandle);
        setupMockAnnouncementCallingDomain(announcementCallingDomain);
        setupMockAnnouncementCallingDisplay(announcementCallingDisplay);
        setupMockPromptAndCollectTimeout(promptAndCollectTimeout);

        // Replay, Execute, and Verify
        replayAll();
        assertFalse(multiChanBroadcastAnnouncementSenderTest.sendAnnouncement());
        verifyAll();
    }

    @Test
    public void testSendAnnouncementCallingHandleEmpty() throws Exception
    {
        // Test Info
        // sendAnnouncement needs the HTTP request to have both the the 'To',
        // FileUri', and the SMGR attribute fields 'Calling Handle' and 'Calling
        // Display' fields to be present

        // Test Data
        final String[] announcementFileUriAsArray =
        { "file:///somefilelocation" };
        final String[] announcementToMultipleCommaMixed =
        { "1111111@domain.com", "2222222@domain.com,3333333@domain.com", "4444444", "5555555,6666666",
                "7777777@domain.com,8888888" };
        final String announcementCallingHandle = "";
        final String announcementCallingDomain = "testDomain.com";
        final String announcementCallingDisplay = "test display";
        final String promptAndCollectTimeout = "120";

        // Setup mocks
        setupMockAnnouncementFileUri(announcementFileUriAsArray);
        setupMockAnnouncementTo(announcementToMultipleCommaMixed);
        setupMockAnnouncementCallingHandle(announcementCallingHandle);
        setupMockAnnouncementCallingDomain(announcementCallingDomain);
        setupMockAnnouncementCallingDisplay(announcementCallingDisplay);
        setupMockPromptAndCollectTimeout(promptAndCollectTimeout);

        // Replay, Execute, and Verify
        replayAll();
        assertFalse(multiChanBroadcastAnnouncementSenderTest.sendAnnouncement());
        verifyAll();
    }

    @Test
    public void testSendAnnouncementCallingDomainException() throws Exception
    {
        // Test Info
        // sendAnnouncement needs the HTTP request to have both the the 'To',
        // FileUri', and the SMGR attribute fields 'Calling Handle' and 'Calling
        // Display' fields to be present

        // Test Data
        final String[] announcementFileUriAsArray =
        { "file:///somefilelocation" };
        final String[] announcementToMultipleCommaMixed =
        { "1111111@domain.com", "2222222@domain.com,3333333@domain.com", "4444444", "5555555,6666666",
                "7777777@domain.com,8888888" };
        final String announcementCallingHandle = "testHandle";
        final String announcementCallingDomain = null;
        final String announcementCallingDisplay = "test display";
        final String promptAndCollectTimeout = "120";

        // Setup mocks
        setupMockAnnouncementFileUri(announcementFileUriAsArray);
        setupMockAnnouncementTo(announcementToMultipleCommaMixed);
        setupMockAnnouncementCallingHandle(announcementCallingHandle);
        setupMockAnnouncementCallingDomain(announcementCallingDomain);
        setupMockAnnouncementCallingDisplay(announcementCallingDisplay);
        setupMockPromptAndCollectTimeout(promptAndCollectTimeout);

        // Replay, Execute, and Verify
        replayAll();
        assertFalse(multiChanBroadcastAnnouncementSenderTest.sendAnnouncement());
        verifyAll();
    }

    @Test
    public void testSendAnnouncementCallingDomainEmpty() throws Exception
    {
        // Test Info
        // sendAnnouncement needs the HTTP request to have both the the 'To',
        // FileUri', and the SMGR attribute fields 'Calling Handle' and 'Calling
        // Display' fields to be present

        // Test Data
        final String[] announcementFileUriAsArray =
        { "file:///somefilelocation" };
        final String[] announcementToMultipleCommaMixed =
        { "1111111@domain.com", "2222222@domain.com,3333333@domain.com", "4444444", "5555555,6666666",
                "7777777@domain.com,8888888" };
        final String announcementCallingHandle = "testHandle";
        final String announcementCallingDomain = "";
        final String announcementCallingDisplay = "test display";
        final String promptAndCollectTimeout = "120";

        // Setup mocks
        setupMockAnnouncementFileUri(announcementFileUriAsArray);
        setupMockAnnouncementTo(announcementToMultipleCommaMixed);
        setupMockAnnouncementCallingHandle(announcementCallingHandle);
        setupMockAnnouncementCallingDomain(announcementCallingDomain);
        setupMockAnnouncementCallingDisplay(announcementCallingDisplay);
        setupMockPromptAndCollectTimeout(promptAndCollectTimeout);

        // Replay, Execute, and Verify
        replayAll();
        assertFalse(multiChanBroadcastAnnouncementSenderTest.sendAnnouncement());
        verifyAll();
    }

    @Test
    public void testSendAnnouncementCallingDisplayException() throws Exception
    {
        // Test Info
        // sendAnnouncement does NOT need the Calling Display to pass

        // Test Data
        final String[] announcementFileUriAsArray =
        { "file:///somefilelocation" };
        ArrayList<String> announcementFileUri = new ArrayList<String>();
        announcementFileUri.add("file:///somefilelocation");
        final String[] announcementToMultipleCommaMixed =
        { "1111111@domain.com", "2222222@domain.com,3333333@domain.com", "4444444@defaultDomain.com", "5555555@defaultDomain.com,6666666@defaultDomain.com",
                "7777777@domain.com,8888888@defaultDomain.com" };
        final List<String> announcementTosSplitAndConcatDefaultDomain =
                Arrays.asList(
                        "1111111@domain.com", "2222222@domain.com", "3333333@domain.com",
                        "4444444@defaultDomain.com",
                        "5555555@defaultDomain.com", "6666666@defaultDomain.com",
                        "7777777@domain.com", "8888888@defaultDomain.com");
        final String announcementCallingHandle = "testHandle";
        final String announcementCallingDomain = "testDomain.com";
        final String announcementCallingDisplay = null;
        final String promptAndCollectTimeout = "120";
        final Integer promptAndCollectTimeoutMilli = 120000;

        // Setup mocks
        setupMockAnnouncementFileUri(announcementFileUriAsArray);
        setupMockAnnouncementTo(announcementToMultipleCommaMixed);
        setupMockAnnouncementCallingHandle(announcementCallingHandle);
        setupMockAnnouncementCallingDomain(announcementCallingDomain);
        setupMockAnnouncementCallingDisplay(announcementCallingDisplay);
        setupMockPromptAndCollectTimeout(promptAndCollectTimeout);
        setupMockAnnouncementCall(announcementTosSplitAndConcatDefaultDomain,
                announcementFileUri,
                promptAndCollectTimeoutMilli);

        // Replay, Execute, and Verify
        replayAll();
        assertTrue(multiChanBroadcastAnnouncementSenderTest.sendAnnouncement());
        verifyAll();
    }

    @Test
    public void testSendAnnouncementCallingDisplayEmpty() throws Exception
    {
        // Test Info
        // sendAnnouncement does NOT need the Calling Display to pass

        // Test Data
        final String[] announcementFileUriAsArray =
        { "file:///somefilelocation" };
        ArrayList<String> announcementFileUri = new ArrayList<String>();
        announcementFileUri.add("file:///somefilelocation");
        final String[] announcementToMultipleCommaMixed =
                { "1111111@domain.com", "2222222@domain.com,3333333@domain.com", "4444444@defaultDomain.com",
                        "5555555@defaultDomain.com,6666666@defaultDomain.com",
                        "7777777@domain.com,8888888@defaultDomain.com" };
        final List<String> announcementTosSplitAndConcatDefaultDomain =
                Arrays.asList(
                        "1111111@domain.com", "2222222@domain.com", "3333333@domain.com",
                        "4444444@defaultDomain.com",
                        "5555555@defaultDomain.com", "6666666@defaultDomain.com",
                        "7777777@domain.com", "8888888@defaultDomain.com");
        final String announcementCallingHandle = "testHandle";
        final String announcementCallingDomain = "testDomain.com";
        final String announcementCallingDisplay = null;
        final String promptAndCollectTimeout = "120";
        final Integer promptAndCollectTimeoutMilli = 120000;

        // Setup mocks
        setupMockAnnouncementFileUri(announcementFileUriAsArray);
        setupMockAnnouncementTo(announcementToMultipleCommaMixed);
        setupMockAnnouncementCallingHandle(announcementCallingHandle);
        setupMockAnnouncementCallingDomain(announcementCallingDomain);
        setupMockAnnouncementCallingDisplay(announcementCallingDisplay);
        setupMockPromptAndCollectTimeout(promptAndCollectTimeout);
        setupMockAnnouncementCall(announcementTosSplitAndConcatDefaultDomain,
                announcementFileUri,
                promptAndCollectTimeoutMilli);

        // Replay, Execute, and Verify
        replayAll();
        assertTrue(multiChanBroadcastAnnouncementSenderTest.sendAnnouncement());
        verifyAll();
    }
}
