/*****************************************************************************
 * Copyright Avaya Inc., All Rights Reserved.
 * THIS IS UNPUBLISHED PROPRIETARY SOURCE CODE OF Avaya Inc.
 * The copyright notice above does not evidence any actual or intended publication of such source code.
 * Some third-party source code components may have been modified from their original versions by Avaya Inc.
 * The modifications are Copyright Avaya Inc., All Rights Reserved.
 * Avaya  Confidential & Restricted. May not be distributed further without written permission of
 * the Avaya owner.
 ****************************************************************************/

package com.avaya.collaboration.authorization;

import java.io.IOException;
import java.security.GeneralSecurityException;
import java.security.KeyStore;
import java.util.List;

import javax.servlet.ServletRequest;

import org.apache.commons.lang.Validate;
import org.apache.commons.validator.routines.UrlValidator;

import com.avaya.collaboration.authorization.modules.ClientModule;
import com.avaya.collaboration.authorization.sample.SampleAuthorizationClient;
import com.google.inject.Guice;
import com.google.inject.Injector;

/**
 * Helper used by clients of the Authorization Service that authenticates the client and returns the granted token.
 *
 * Refer {@link SampleAuthorizationClient} for an example usage.
 *
 * @author Avaya
 * @since 3.2
 *
 *        TODO: Javadoc improvements desired: see maven-javadoc-plugin warnings
 */

public final class AuthorizationClientHelper // NOPMD
{

    private AccessTokenProvider tokenProvider;

    private AuthorizationClientHelper()
    {
    }

    /**
     *
     * Helps in the creation of an instance of AuthorizationClientHelper. <br>
     * Initialize the builder and call the following components to complete its construction: <br>
     * <br>
     *
     * tokenEndpoint - Authorization Service Token Endpoint address <br>
     * clientIdentifier - Application's client identifier <br>
     * keyStore - Application's keystore having its private key. The private key is used to sign a JWT (JSON Web Token) on <br>
     * behalf of the cllent which is then used to authenticate the client with Authorization Service. <br>
     * trustStore - Application's truststore. This should contain a CA certificate trusted both by the client and the node on <br>
     * which Authorization Service runs.
     *
     * @author Avaya
     *
     * @since 3.2
     *
     */
    public static final class Builder
    {
        private final AuthorizationClientHelper authorizationClientHelper;
        private final ClientParams clientParams;

        public Builder()
        {
            authorizationClientHelper = new AuthorizationClientHelper();
            clientParams = new ClientParams();
        }

        public ClientIdBuilder tokenEndpoint(final String tokenEndpoint)
        {
            clientParams.setTokenEndpoint(tokenEndpoint);
            return new ClientIdBuilder();
        }

        public final class ClientIdBuilder
        {
            private ClientIdBuilder()
            {
            }

            public KeyStoreBuilder clientIdentifier(final String clientIdentifier)
            {
                clientParams.setClientId(clientIdentifier);
                return new KeyStoreBuilder();
            }
        }

        public final class KeyStoreBuilder
        {
            private KeyStoreBuilder()
            {
            }

            public TrustStoreBuilder keyStore(final KeyStore keyStore, final String keyStorePassword, final String keyAlias)
            {
                clientParams.setKeyStore(keyStore);
                clientParams.setKeyStorePassword(keyStorePassword);
                clientParams.setKeyAlias(keyAlias);
                return new TrustStoreBuilder();
            }
        }

        public final class TrustStoreBuilder
        {
            private TrustStoreBuilder()
            {
            }

            public ClientHelperBuilder trustStore(final KeyStore trustStore)
            {
                clientParams.setTrustStore(trustStore);
                return new ClientHelperBuilder();
            }
        }

        public final class ClientHelperBuilder
        {
            private ClientHelperBuilder()
            {
            }

            public AuthorizationClientHelper build() throws AuthorizationHelperException
            {
                validateNotNull(clientParams);

                authorizationClientHelper.initializeAccessTokenProvider(clientParams);

                return authorizationClientHelper;
            }

            private void validateNotNull(final ClientParams clientParams)
            {
                Validate.notNull(clientParams.getClientId(), "Client Id cannot be null.");
                Validate.notNull(clientParams.getKeyAlias(), "Key alias cannot be null");
                Validate.notNull(clientParams.getKeyStore(), "Key store cannot be null");
                Validate.notNull(clientParams.getKeyStorePassword(), "Key store password cannot be null");
                Validate.notNull(clientParams.getTokenEndpoint(), "Token endpoint cannot be null");

                final String[] schemes =
                { "https" };
                final UrlValidator urlValidator =
                        new UrlValidator(schemes, UrlValidator.ALLOW_LOCAL_URLS);

                Validate.isTrue(urlValidator.isValid(clientParams.getTokenEndpoint()), "Token endpoint must be a valid URL");

                Validate.notNull(clientParams.getTrustStore(), "Trust store cannot be null");
            }
        }
    }

    private AccessTokenProvider getAccessTokenProvider()
    {
        return tokenProvider;
    }

    private void initializeAccessTokenProvider(final ClientParams clientParams) throws AuthorizationHelperException
    {
        if (tokenProvider == null)
        {
            Injector injector;
            try
            {
                injector = Guice.createInjector(new ClientModule(clientParams));
            }
            catch (IOException | GeneralSecurityException e)
            {
                throw new AuthorizationHelperException("Caught exception while initializing helper: ", e);
            }
            tokenProvider = injector.getInstance(AccessTokenProvider.class);
        }
    }

    /**
     * Authenticates the client with the Authorization Service, obtains an access token and returns it.
     *
     * @return {@link AccessToken}
     *
     * @throws AuthorizationHelperException
     *             Indicates error conditions in the Authorization Helper APIs.
     * @throws HttpResponseException
     *             Indicates HTTP error responses propagated by the Authorization Helper APIs.
     */
    public AccessToken getAccessToken() throws AuthorizationHelperException, HttpResponseException
    {
        return getAccessTokenProvider().getAccessToken();
    }

    /**
     * Authenticates the client with the Authorization Service and obtains a token aggregate.
     *
     * @return {@link TokenAggregate}
     *
     * @throws AuthorizationHelperException
     *             Indicates error conditions in the Authorization Helper APIs.
     * @throws HttpResponseException
     *             Indicates HTTP error responses propagated by the Authorization Helper APIs.
     *
     * @since 3.8
     */
    public TokenAggregate getTokenAggregate() throws AuthorizationHelperException, HttpResponseException
    {
        return getAccessTokenProvider().getTokenAggregate();
    }

    /**
     * Authenticates the client with the Authorization Service, obtains an access token and returns it; the client can request for specific
     * scopes during authentication by using the "scopes" parameter.
     *
     * @param scopes
     *            The list of scopes the client wants to specify while making the token request.
     *
     * @return {@link AccessToken}
     *
     * @throws AuthorizationHelperException
     *             Indicates error conditions in the Authorization Helper APIs.
     * @throws HttpResponseException
     *             Indicates HTTP error responses propagated by the Authorization Helper APIs.
     */
    public AccessToken getAccessToken(final List<String> scopes) throws AuthorizationHelperException, HttpResponseException
    {
        return getAccessTokenProvider().getAccessToken(scopes);
    }

    /**
     * Authenticates the client with the Authorization Service, obtains a token aggregate and returns it; the client can request for
     * specific scopes during authentication by using the "scopes" parameter.
     *
     * @param scopes
     *            The list of scopes the client wants to specify while making the token request.
     *
     * @return {@link TokenAggregate}
     *
     * @throws AuthorizationHelperException
     *             Indicates error conditions in the Authorization Helper APIs.
     *
     * @throws HttpResponseException
     *             Indicates HTTP error responses propagated by the Authorization Helper APIs.
     *
     * @since 3.8
     */
    public TokenAggregate getTokenAggregate(final List<String> scopes) throws AuthorizationHelperException, HttpResponseException
    {
        return getAccessTokenProvider().getTokenAggregate(scopes);
    }

    /**
     * Authenticates the client (along with the user's auth code) with the Authorization Service, obtains an access token and returns it.
     *
     * Note: This API would primarily be used by a web filter.If the packaged filter is being used then the application will not need invoke
     * these APIs as it is handled by the filer. Please refer to integration guide document for more details.
     *
     * @param servletRequest
     *            The request which the application received.
     *
     * @return {@link AccessToken}
     *
     * @throws AuthorizationHelperException
     *             Indicates error conditions in the Authorization Helper APIs.
     * @throws HttpResponseException
     *             Indicates HTTP error responses propagated by the Authorization Helper APIs.
     */
    public AccessToken getAccessTokenForUser(final ServletRequest servletRequest) throws AuthorizationHelperException,
            HttpResponseException
    {
        return getAccessTokenProvider().getAccessTokenForUser(servletRequest);
    }

    /**
     * Authenticates the client (along with the user's auth code) with the Authorization Service, obtains a token aggregate and returns it.
     *
     * Note: This API would primarily be used by a web filter.If the packaged filter is being used then the snap-in will not need invoke
     * these APIs as it is handled by the filer. Please refer to developer guide document for more details.
     *
     * @param servletRequest
     *            This used to construct the Redirection URI which will be validated by the Authorization Service to ensure that the
     *            redirection URI sent in auth code request and redirection URI in access token request are identical.
     * @return {@link TokenAggregate}
     *
     * @throws AuthorizationHelperException
     *             Indicates error conditions in the Authorization Helper APIs.
     * @throws HttpResponseException
     *             Indicates HTTP error responses propagated by the Authorization Helper APIs.
     *
     * @since 3.8
     */
    public TokenAggregate getTokenAggregateForUser(final ServletRequest servletRequest) throws AuthorizationHelperException,
            HttpResponseException
    {
        return getAccessTokenProvider().getTokenAggregateForUser(servletRequest);
    }

    /**
     * Authenticates the client and the end user with the Authorization Service, obtains an access token and returns it.
     *
     * @param userName
     *            The user name for whom the token is being requested.
     * @param userPassword
     *            The user's password.
     * @return {@link AccessToken}
     *
     * @throws AuthorizationHelperException
     *             Indicates error conditions in the Authorization Helper APIs.
     * @throws HttpResponseException
     *             Indicates HTTP error responses propagated by the Authorization Helper APIs.
     */
    public AccessToken getAccessTokenForUser(final String userName, final String userPassword)
            throws AuthorizationHelperException,
            HttpResponseException
    {
        return getAccessTokenProvider().getAccessTokenForUser(userName, userPassword);
    }

    /**
     * Authenticates the client and the end user with the Authorization Service, obtains a token aggregate and returns it.
     *
     * @param userName
     *            The user name for whom the token is being requested.
     * @param userPassword
     *            The user's password.
     * @return {@link TokenAggregate}
     *
     * @throws AuthorizationHelperException
     *             Indicates error conditions in the Authorization Helper APIs.
     * @throws HttpResponseException
     *             Indicates HTTP error responses propagated by the Authorization Helper APIs.
     *
     * @since 3.8
     */
    public TokenAggregate getTokenAggregateForUser(final String userName, final String userPassword)
            throws AuthorizationHelperException,
            HttpResponseException
    {
        return getAccessTokenProvider().getTokenAggregateForUser(userName, userPassword);
    }

    /**
     * Authenticates the client and the end user with the Authorization Service, obtains an access token and returns it; the client can
     * request for specific scopes during authentication by using the "scopes" parameter.
     *
     * @param userName
     *            The user name for whom the token is being requested.
     * @param userPassword
     *            The user's password.
     * @param scopes
     *            The list of scopes the client wants to specify while making the token request.
     * @return {@link AccessToken}
     *
     * @throws AuthorizationHelperException
     *             Indicates error conditions in the Authorization Helper APIs.
     * @throws HttpResponseException
     *             Indicates HTTP error responses propagated by the Authorization Helper APIs.
     */
    public AccessToken getAccessTokenForUser(final String userName, final String userPassword, final List<String> scopes)
            throws AuthorizationHelperException, HttpResponseException
    {
        return getAccessTokenProvider().getAccessTokenForUser(userName, userPassword, scopes);
    }

    /**
     * Authenticates the client and the end user with the Authorization Service, obtains an access token and returns it; the client can
     * request for specific scopes during authentication by using the "scopes" parameter.
     *
     * @param userName
     *            The user name for whom the token is being requested.
     * @param userPassword
     *            The user's password.
     * @param scopes
     *            The list of scopes the client wants to specify while making the token request.
     * @return {@link AccessToken}
     *
     * @throws AuthorizationHelperException
     *             Indicates error conditions in the Authorization Helper APIs.
     * @throws HttpResponseException
     *             Indicates HTTP error responses propagated by the Authorization Helper APIs.
     *
     * @since 3.8
     */
    public TokenAggregate getTokenAggregateForUser(final String userName, final String userPassword, final List<String> scopes)
            throws AuthorizationHelperException, HttpResponseException
    {
        return getAccessTokenProvider().getTokenAggregateForUser(userName, userPassword, scopes);
    }

    /**
     * Authenticates the client with the Authorization Service, obtains a token aggregate using a refresh token and returns it.
     *
     * @param refreshToken
     *            A refresh token obtained previously by the client.
     *
     * @return {@link TokenAggregate}
     *
     * @throws AuthorizationHelperException
     *             Indicates error conditions in the Authorization Helper APIs.
     * @throws HttpResponseException
     *             Indicates HTTP error responses propagated by the Authorization Helper APIs.
     *
     * @since 3.8
     */
    public TokenAggregate getTokenAggregate(final String refreshToken) throws AuthorizationHelperException, HttpResponseException
    {
        return getAccessTokenProvider().getTokenAggregate(refreshToken);
    }

    /**
     * Shuts down the helper library and releases any system resources associated with it.
     *
     * @throws AuthorizationHelperException
     *             Indicates error conditions in the Authorization Helper APIs.
     */
    public void shutdown() throws AuthorizationHelperException
    {
        getAccessTokenProvider().shutDown();
    }
}
