/****************************************************************************
 * Copyright Avaya Inc., All Rights Reserved.
 ****************************************************************************/
package com.avaya.zephyr.sdk.authorization.samples.client.interceptors;

import static org.junit.Assert.assertFalse;
import static org.junit.Assert.assertTrue;
import static org.mockito.Mockito.when;

import java.io.File;

import javax.servlet.http.Cookie;
import javax.servlet.http.HttpServletRequest;
import javax.servlet.http.HttpServletResponse;

import org.apache.commons.io.FileUtils;
import org.junit.Before;
import org.junit.Test;
import org.junit.runner.RunWith;
import org.mockito.Mock;
import org.mockito.Mockito;
import org.mockito.runners.MockitoJUnitRunner;
import org.springframework.core.env.Environment;

import com.avaya.collaboration.authorization.AccessToken;
import com.avaya.collaboration.authorization.AccessTokenProvider;
import com.avaya.zephyr.common.testutils.deencapsulation.Deencapsulation;

@RunWith(MockitoJUnitRunner.class)
public class AccessTokenInterceptorTest
{
    AccessTokenInterceptor classToTest;
    private static final String AUTH_CODE_GRANT_TYPE = "code";

    @Mock
    AccessToken mockAccessToken;

    @Mock
    HttpServletRequest mockRequest;

    @Mock
    HttpServletResponse mockResponse;

    @Mock
    AccessTokenProvider mockAccessTokenProvider;

    @Mock
    Object mockHandler;

    @Mock
    Environment mockEnv;

    String keyStorePath;
    String trustStorePath;
    Cookie[] requestCookies;

    @Before
    public void setUp() throws Exception
    {
        final File keyStoreFile = new File("src/test/resources/clientkeystore.jks");
        keyStorePath = keyStoreFile.getAbsolutePath();

        final File trustStoreFile = new File("src/test/resources/clienttruststore.jks");
        trustStorePath = trustStoreFile.getAbsolutePath();

        when(mockEnv.getProperty("clientKeystorePath")).thenReturn(keyStorePath);
        when(mockEnv.getProperty("clientTruststorePath")).thenReturn(trustStorePath);

        when(mockEnv.getProperty("clientKeystorePassword")).thenReturn("password");
        when(mockEnv.getProperty("clientTruststorePassword")).thenReturn("password");

        when(mockEnv.getProperty("tokenEndpointURL")).thenReturn("https://127.0.0.1:9443/services/AuthorizationService/token");
        when(mockEnv.getProperty("clientId")).thenReturn("client_id");
        when(mockEnv.getProperty("clientCertificateAlias")).thenReturn("clientcert");
        when(mockEnv.getProperty("authCertAliasList")).thenReturn("breeze14");

        classToTest = new AccessTokenInterceptor(mockEnv);
        Deencapsulation.setField(classToTest.getAuthorizationClientHelper(), "tokenProvider", mockAccessTokenProvider);
    }

    @Test
    public void testReturnsTrueIfRequestHasValidCookie() throws Exception
    {
        final File authDataFile = new File("src/test/resources/sessionData.txt");

        requestCookies = new Cookie[1];
        requestCookies[0] = new Cookie("UserSession", FileUtils.readFileToString(authDataFile));

        when(mockRequest.getCookies()).thenReturn(requestCookies);
        when(mockEnv.getProperty("sessionCookieName")).thenReturn("UserSession");

        assertTrue(classToTest.preHandle(mockRequest, mockResponse, mockHandler));
    }

    @Test
    public void testRedirectionIfRequestContainsAuthCode() throws Exception
    {
        when(mockRequest.getParameter(AUTH_CODE_GRANT_TYPE)).thenReturn("auth_code");
        when(mockAccessTokenProvider.getAccessTokenForUser(Mockito.any(HttpServletRequest.class))).thenReturn(mockAccessToken);
        when(mockAccessToken.getSubject()).thenReturn("testuser");
        when(mockAccessToken.toString()).thenReturn("access_token");
        when(mockAccessToken.getExpiresIn()).thenReturn(32400);
        when(mockEnv.getProperty("sessionCookieName")).thenReturn("UserSession");
        when(mockRequest.getRequestURL()).thenReturn(new StringBuffer("http://localhost"));

        assertFalse(classToTest.preHandle(mockRequest, mockResponse, mockHandler));
        Mockito.verify(mockResponse, Mockito.times(1)).sendRedirect(Mockito.anyString());
    }
}