/*****************************************************************************
 * Copyright Avaya Inc., All Rights Reserved.
 * THIS IS UNPUBLISHED PROPRIETARY SOURCE CODE OF Avaya Inc.
 * The copyright notice above does not evidence any actual or intended publication of such source code.
 * Some third-party source code components may have been modified from their original versions by Avaya Inc.
 * The modifications are Copyright Avaya Inc., All Rights Reserved.
 * Avaya  Confidential & Restricted. May not be distributed further without written permission of
 * the Avaya owner.
 ****************************************************************************/

package com.avaya.collaboration.authorization.impl;

import java.io.IOException;
import java.io.UnsupportedEncodingException;
import java.security.KeyStoreException;
import java.security.NoSuchAlgorithmException;
import java.security.UnrecoverableKeyException;
import java.security.interfaces.RSAPrivateKey;
import java.util.Collection;
import java.util.Date;
import java.util.List;
import java.util.Map;
import java.util.UUID;

import javax.servlet.ServletRequest;
import javax.servlet.http.HttpServletRequest;

import org.apache.commons.lang.StringUtils;
import org.apache.http.HttpHeaders;
import org.apache.http.client.methods.HttpPost;
import org.apache.http.entity.StringEntity;
import org.apache.oltu.oauth2.client.request.OAuthClientRequest;
import org.apache.oltu.oauth2.common.exception.OAuthSystemException;
import org.apache.oltu.oauth2.common.message.types.GrantType;

import com.avaya.collaboration.authorization.AccessToken;
import com.avaya.collaboration.authorization.AccessTokenProvider;
import com.avaya.collaboration.authorization.AuthorizationHelperException;
import com.avaya.collaboration.authorization.HttpResponseException;
import com.avaya.collaboration.authorization.TokenAggregate;
import com.avaya.collaboration.authorization.http.HttpClientProvider;
import com.avaya.collaboration.authorization.http.HttpRequestExecutor;
import com.avaya.collaboration.authorization.modules.KeyProvider;
import com.google.common.base.Strings;
import com.google.inject.Inject;
import com.google.inject.name.Named;
import com.nimbusds.jose.JOSEException;
import com.nimbusds.jose.JWSAlgorithm;
import com.nimbusds.jose.JWSHeader;
import com.nimbusds.jose.JWSSigner;
import com.nimbusds.jose.crypto.RSASSASigner;
import com.nimbusds.jwt.JWTClaimsSet;
import com.nimbusds.jwt.SignedJWT;

/**
 * Implementation of the {@link AccessTokenProvider} interface.
 *
 * @author Avaya
 */
public class AccessTokenProviderImpl implements AccessTokenProvider
{
    private final HttpClientProvider clientProvider;
    private final String tokenEndpoint;
    private final String clientId;
    private static final String AUTH_CODE = "code";

    private JWSSigner signer;

    @Inject
    public AccessTokenProviderImpl(final HttpClientProvider clientProvider,
            final KeyProvider<RSAPrivateKey> clientPrivateKeyProvider,
            @Named("tokenEndpoint") final String tokenEndpoint,
            @Named("clientId") final String clientId)
            throws AuthorizationHelperException, UnrecoverableKeyException, KeyStoreException, NoSuchAlgorithmException
    {
        this.clientProvider = clientProvider;
        this.tokenEndpoint = tokenEndpoint;
        this.clientId = clientId;

        final RSAPrivateKey aKey = clientPrivateKeyProvider.get();
        setJwtSigner(aKey);
    }

    /*
     * Implements the Client Credentials Grant Type.
     *
     * (non-Javadoc)
     *
     * @see com.avaya.collaboration.authorization.AccessTokenProvider#getAccessToken()
     */
    @Override
    public AccessToken getAccessToken() throws AuthorizationHelperException, HttpResponseException
    {
        final String clientCredentials = buildSignedJwt(buildJwtClaimsSet(clientId, clientId));

        try
        {
            final OAuthClientRequest tokenRequest =
                    OAuthClientRequest
                            .tokenLocation(tokenEndpoint)
                            .setGrantType(GrantType.CLIENT_CREDENTIALS)
                            .setParameter("client_assertion_type",
                                    "urn:ietf:params:oauth:client-assertion-type:jwt-bearer")
                            .setParameter("client_assertion", clientCredentials)
                            .buildBodyMessage();
            tokenRequest.addHeader(HttpHeaders.CONTENT_TYPE, "application/x-www-form-urlencoded");

            return getTokenResponse(tokenRequest);
        }
        catch (final OAuthSystemException exception)
        {
            throw new AuthorizationHelperException("Caught exception while building token request: ", exception);
        }
    }

    /*
     * Implements the Client Credentials Grant Type. Also asks specific scopes in the request.
     *
     * (non-Javadoc)
     *
     * @see com.avaya.collaboration.authorization.AccessTokenProvider#getAccessToken(java.util.List)
     */
    @Override
    public AccessToken getAccessToken(final List<String> scopes) throws AuthorizationHelperException, HttpResponseException
    {
        final String clientCredentials = buildSignedJwt(buildJwtClaimsSet(clientId, clientId));

        try
        {
            if (scopes != null && scopes.size() > 0 && !((scopes.contains(null) || scopes.contains(""))))
            {
                final String scope = StringUtils.join(scopes, " ");

                final OAuthClientRequest tokenRequest =
                        OAuthClientRequest
                                .tokenLocation(tokenEndpoint)
                                .setGrantType(GrantType.CLIENT_CREDENTIALS)
                                .setParameter("client_assertion_type",
                                        "urn:ietf:params:oauth:client-assertion-type:jwt-bearer")
                                .setParameter("client_assertion", clientCredentials)
                                .setScope(scope)
                                .buildBodyMessage();
                tokenRequest.addHeader(HttpHeaders.CONTENT_TYPE, "application/x-www-form-urlencoded");
                return getTokenResponse(tokenRequest);
            }
            else
            {
                throw new AuthorizationHelperException("Requested scopes are empty.");
            }
        }
        catch (final OAuthSystemException exception)
        {
            throw new AuthorizationHelperException("Caught exception while building token request: ",
                    exception);
        }
    }

    /*
     * Implements the Authorization Code Grant Type. It fetches the authorization code from the request and makes an access token request
     * using the code.
     *
     * (non-Javadoc)
     *
     * @see com.avaya.collaboration.authorization.AccessTokenProvider#getAccessTokenForUser(javax.servlet.ServletRequest)
     */
    @Override
    public AccessToken getAccessTokenForUser(ServletRequest servletRequest) throws AuthorizationHelperException,
            HttpResponseException
    {
        final HttpServletRequest httpServletRequest = (HttpServletRequest) servletRequest;

        final String authCode = servletRequest.getParameter(AUTH_CODE);
        final String clientCredentials = buildSignedJwt(buildJwtClaimsSet(clientId, clientId));
        try
        {
            final OAuthClientRequest tokenRequest =
                    OAuthClientRequest
                            .tokenLocation(tokenEndpoint)
                            .setGrantType(GrantType.AUTHORIZATION_CODE)
                            .setCode(authCode)
                            .setRedirectURI(httpServletRequest.getRequestURL().toString())
                            .setClientId(clientId)
                            .buildBodyMessage();

            tokenRequest.addHeader("Authorization", "Bearer " + clientCredentials);
            tokenRequest.addHeader(HttpHeaders.CONTENT_TYPE, "application/x-www-form-urlencoded");
            return getTokenResponse(tokenRequest);

        }
        catch (final OAuthSystemException exception)
        {
            throw new AuthorizationHelperException("Caught exception while building token request: ",
                    exception);
        }
    }

    /*
     * Implements the Resource Owner Password Credentials Grant Type.
     *
     * (non-Javadoc)
     *
     * @see com.avaya.collaboration.authorization.AccessTokenProvider#getAccessTokenForUser(java.lang.String, java.lang.String)
     */
    @Override
    public AccessToken getAccessTokenForUser(final String userName, final String userPassword) throws AuthorizationHelperException,
            HttpResponseException
    {
        final String clientCredentials = buildSignedJwt(buildJwtClaimsSet(clientId, userName));

        try
        {
            if (!Strings.isNullOrEmpty(userName) && !Strings.isNullOrEmpty(userPassword))
            {
                final OAuthClientRequest tokenRequest =
                        OAuthClientRequest.tokenLocation(tokenEndpoint)
                                .setGrantType(GrantType.PASSWORD)
                                .setUsername(userName)
                                .setPassword(userPassword)
                                .buildBodyMessage();

                tokenRequest.addHeader("Authorization", "Bearer " + clientCredentials);
                tokenRequest.addHeader(HttpHeaders.CONTENT_TYPE, "application/x-www-form-urlencoded");
                return getTokenResponse(tokenRequest);
            }
            else
            {
                throw new AuthorizationHelperException("Username/Password cannot be null/empty.");
            }
        }
        catch (final OAuthSystemException exception)
        {
            throw new AuthorizationHelperException("Caught exception while getting token request: ",
                    exception);
        }
    }

    /*
     * Implements the Resource Owner Password Credentials Grant Type. Also asks specific scopes in the request.
     *
     * (non-Javadoc)
     *
     * @see com.avaya.collaboration.authorization.AccessTokenProvider#getAccessTokenForUser(java.lang.String, java.lang.String,
     * java.util.List)
     */
    @Override
    public AccessToken getAccessTokenForUser(final String userName, final String userPassword, final List<String> scopes)
            throws AuthorizationHelperException, HttpResponseException
    {
        final String clientCredentials = buildSignedJwt(buildJwtClaimsSet(clientId, userName));

        try
        {
            if (!(isNullOrEmpty(scopes)) && !Strings.isNullOrEmpty(userName) &&
                    !Strings.isNullOrEmpty(userPassword))
            {
                final String scope = StringUtils.join(scopes, " ");

                final OAuthClientRequest tokenRequest =
                        OAuthClientRequest.tokenLocation(tokenEndpoint)
                                .setGrantType(GrantType.PASSWORD)
                                .setUsername(userName)
                                .setPassword(userPassword)
                                .setScope(scope)
                                .buildBodyMessage();

                tokenRequest.addHeader("Authorization", "Bearer " + clientCredentials);
                tokenRequest.addHeader(HttpHeaders.CONTENT_TYPE, "application/x-www-form-urlencoded");
                return getTokenResponse(tokenRequest);
            }
            else
            {
                throw new AuthorizationHelperException("Scopes/Username/Password cannot be null/empty");
            }
        }
        catch (final OAuthSystemException exception)
        {
            throw new AuthorizationHelperException("Caught exception while building token request: ",
                    exception);
        }
    }

    @Override
    public void shutDown() throws AuthorizationHelperException
    {
        try
        {
            clientProvider.shutDown();
        }
        catch (final IOException e)
        {
            throw new AuthorizationHelperException("Caught exception while shutting down provider: ", e);
        }
    }

    private boolean isNullOrEmpty(final Collection<?> collection)
    {
        return collection == null || collection.isEmpty() || collection.contains(null) ||
                collection.contains("");
    }

    private void setJwtSigner(final RSAPrivateKey privateKey) throws AuthorizationHelperException
    {
        if (privateKey == null)
        {
            throw new AuthorizationHelperException(
                    "Unable to determine signer key for client. Check if the keystore has a valid private key.");
        }

        signer = new RSASSASigner(privateKey);
    }

    private HttpPost preparePostRequest(final OAuthClientRequest request, final Map<String, String> headers)
            throws AuthorizationHelperException
    {
        final HttpPost postRequest = new HttpPost(request.getLocationUri());

        if (headers != null && !headers.isEmpty())
        {
            for (final Map.Entry<String, String> header : headers.entrySet())
            {
                postRequest.addHeader(header.getKey(), header.getValue());
            }
        }
        try
        {
            postRequest.setEntity(new StringEntity(request.getBody()));
        }
        catch (final UnsupportedEncodingException e)
        {
            throw new AuthorizationHelperException("Caught exception while building POST request: ", e);
        }
        return postRequest;
    }

    /*
     * Builds JWT claims set using the provided issuer id and subject id.
     */
    private JWTClaimsSet buildJwtClaimsSet(final String jwtIssuer, final String jwtSubject)
    {
        return new JWTClaimsSet.Builder()
                .issuer(jwtIssuer)
                .subject(jwtSubject)
                .issueTime(new Date())
                .jwtID(UUID.randomUUID().toString())
                .build();
    }

    /*
     * Builds a JWT with the provided claims and signs it.
     */
    private String buildSignedJwt(final JWTClaimsSet claimsSet) throws AuthorizationHelperException
    {
        final SignedJWT signedJWT = new SignedJWT(new JWSHeader(JWSAlgorithm.RS256), claimsSet);

        try
        {
            signedJWT.sign(signer);
        }
        catch (final JOSEException e)
        {
            throw new AuthorizationHelperException("Caught exception while building JWT request: ", e);
        }
        return signedJWT.serialize();
    }

    private AccessToken getTokenResponse(final OAuthClientRequest tokenRequest)
            throws AuthorizationHelperException, HttpResponseException
    {
        final HttpPost postRequest = preparePostRequest(tokenRequest, tokenRequest.getHeaders());

        final HttpRequestExecutor requestExecutor =
                new HttpRequestExecutor(clientProvider.getClientIntance(), postRequest);

        return requestExecutor.execute();
    }

    private TokenAggregate getAggregateResponse(final OAuthClientRequest tokenRequest)
            throws AuthorizationHelperException, HttpResponseException
    {
        final HttpPost postRequest = preparePostRequest(tokenRequest, tokenRequest.getHeaders());

        final HttpRequestExecutor requestExecutor =
                new HttpRequestExecutor(clientProvider.getClientIntance(), postRequest);

        return requestExecutor.getTokenAggregate();
    }

    @Override
    public TokenAggregate getTokenAggregate() throws AuthorizationHelperException, HttpResponseException
    {
        final String clientCredentials = buildSignedJwt(buildJwtClaimsSet(clientId, clientId));

        try
        {
            final OAuthClientRequest tokenRequest =
                    OAuthClientRequest
                            .tokenLocation(tokenEndpoint)
                            .setGrantType(GrantType.CLIENT_CREDENTIALS)
                            .setParameter("client_assertion_type",
                                    "urn:ietf:params:oauth:client-assertion-type:jwt-bearer")
                            .setParameter("client_assertion", clientCredentials)
                            .buildBodyMessage();
            tokenRequest.addHeader(HttpHeaders.CONTENT_TYPE, "application/x-www-form-urlencoded");
            tokenRequest.addHeader(HttpHeaders.ACCEPT, "application/avaya.auth.token.v2+json");

            return getAggregateResponse(tokenRequest);
        }
        catch (final OAuthSystemException exception)
        {
            throw new AuthorizationHelperException("Caught exception while building token request: ", exception);
        }
    }

    @Override
    public TokenAggregate getTokenAggregate(List<String> scopes) throws AuthorizationHelperException, HttpResponseException
    {
        final String clientCredentials = buildSignedJwt(buildJwtClaimsSet(clientId, clientId));
        try
        {
            if (scopes != null && scopes.size() > 0 && !((scopes.contains(null) || scopes.contains(""))))
            {
                final String scope = StringUtils.join(scopes, " ");

                final OAuthClientRequest tokenRequest =
                        OAuthClientRequest
                                .tokenLocation(tokenEndpoint)
                                .setGrantType(GrantType.CLIENT_CREDENTIALS)
                                .setParameter("client_assertion_type",
                                        "urn:ietf:params:oauth:client-assertion-type:jwt-bearer")
                                .setParameter("client_assertion", clientCredentials)
                                .setScope(scope)
                                .buildBodyMessage();
                tokenRequest.addHeader(HttpHeaders.CONTENT_TYPE, "application/x-www-form-urlencoded");
                tokenRequest.addHeader(HttpHeaders.ACCEPT, "application/avaya.auth.token.v2+json");

                return getAggregateResponse(tokenRequest);
            }
            else
            {
                throw new AuthorizationHelperException("Requested scopes are empty.");
            }
        }
        catch (final OAuthSystemException exception)
        {
            throw new AuthorizationHelperException("Caught exception while building token request: ",
                    exception);
        }
    }

    @Override
    public TokenAggregate getTokenAggregateForUser(ServletRequest servletRequest) throws AuthorizationHelperException, HttpResponseException
    {
        final HttpServletRequest httpServletRequest = (HttpServletRequest) servletRequest;

        final String authCode = servletRequest.getParameter(AUTH_CODE);
        final String clientCredentials = buildSignedJwt(buildJwtClaimsSet(clientId, clientId));
        try
        {
            final OAuthClientRequest tokenRequest =
                    OAuthClientRequest
                            .tokenLocation(tokenEndpoint)
                            .setGrantType(GrantType.AUTHORIZATION_CODE)
                            .setCode(authCode)
                            .setRedirectURI(httpServletRequest.getRequestURL().toString())
                            .setClientId(clientId)
                            .buildBodyMessage();

            tokenRequest.addHeader("Authorization", "Bearer " + clientCredentials);
            tokenRequest.addHeader(HttpHeaders.CONTENT_TYPE, "application/x-www-form-urlencoded");
            tokenRequest.addHeader(HttpHeaders.ACCEPT, "application/avaya.auth.token.v2+json");

            return getAggregateResponse(tokenRequest);

        }
        catch (final OAuthSystemException exception)
        {
            throw new AuthorizationHelperException("Caught exception while building token request: ",
                    exception);
        }
    }

    @Override
    public TokenAggregate getTokenAggregateForUser(String userName, String userPassword)
            throws AuthorizationHelperException, HttpResponseException
    {
        final String clientCredentials = buildSignedJwt(buildJwtClaimsSet(clientId, userName));

        try
        {
            if (!Strings.isNullOrEmpty(userName) && !Strings.isNullOrEmpty(userPassword))
            {
                final OAuthClientRequest tokenRequest =
                        OAuthClientRequest.tokenLocation(tokenEndpoint)
                                .setGrantType(GrantType.PASSWORD)
                                .setUsername(userName)
                                .setPassword(userPassword)
                                .buildBodyMessage();

                tokenRequest.addHeader("Authorization", "Bearer " + clientCredentials);
                tokenRequest.addHeader(HttpHeaders.CONTENT_TYPE, "application/x-www-form-urlencoded");
                tokenRequest.addHeader(HttpHeaders.ACCEPT, "application/avaya.auth.token.v2+json");

                return getAggregateResponse(tokenRequest);
            }
            else
            {
                throw new AuthorizationHelperException("Username/Password cannot be null/empty.");
            }
        }
        catch (final OAuthSystemException exception)
        {
            throw new AuthorizationHelperException("Caught exception while getting token request: ",
                    exception);
        }
    }

    @Override
    public TokenAggregate getTokenAggregateForUser(String userName, String userPassword, List<String> scopes)
            throws AuthorizationHelperException, HttpResponseException
    {
        final String clientCredentials = buildSignedJwt(buildJwtClaimsSet(clientId, userName));

        try
        {
            if (!(isNullOrEmpty(scopes)) && !Strings.isNullOrEmpty(userName) &&
                    !Strings.isNullOrEmpty(userPassword))
            {
                final String scope = StringUtils.join(scopes, " ");

                final OAuthClientRequest tokenRequest =
                        OAuthClientRequest.tokenLocation(tokenEndpoint)
                                .setGrantType(GrantType.PASSWORD)
                                .setUsername(userName)
                                .setPassword(userPassword)
                                .setScope(scope)
                                .buildBodyMessage();

                tokenRequest.addHeader("Authorization", "Bearer " + clientCredentials);
                tokenRequest.addHeader(HttpHeaders.CONTENT_TYPE, "application/x-www-form-urlencoded");
                tokenRequest.addHeader(HttpHeaders.ACCEPT, "application/avaya.auth.token.v2+json");

                return getAggregateResponse(tokenRequest);
            }
            else
            {
                throw new AuthorizationHelperException("Scopes/Username/Password cannot be null/empty");
            }
        }
        catch (final OAuthSystemException exception)
        {
            throw new AuthorizationHelperException("Caught exception while building token request: ",
                    exception);
        }
    }

    @Override
    public TokenAggregate getTokenAggregate(String refreshToken) throws AuthorizationHelperException, HttpResponseException
    {
        final String clientCredentials = buildSignedJwt(buildJwtClaimsSet(clientId, clientId));

        try
        {
            if (!Strings.isNullOrEmpty(refreshToken))
            {
                final OAuthClientRequest tokenRequest =
                        OAuthClientRequest.tokenLocation(tokenEndpoint)
                                .setGrantType(GrantType.REFRESH_TOKEN)
                                .setRefreshToken(refreshToken)
                                .buildBodyMessage();

                tokenRequest.addHeader("Authorization", "Bearer " + clientCredentials);
                tokenRequest.addHeader(HttpHeaders.CONTENT_TYPE, "application/x-www-form-urlencoded");
                tokenRequest.addHeader(HttpHeaders.ACCEPT, "application/avaya.auth.token.v2+json");

                return getAggregateResponse(tokenRequest);
            }
            else
            {
                throw new AuthorizationHelperException("Username/Password cannot be null/empty.");
            }
        }
        catch (final OAuthSystemException exception)
        {
            throw new AuthorizationHelperException("Caught exception while getting token request: ",
                    exception);
        }
    }
}