/*****************************************************************************
 * Copyright Avaya Inc., All Rights Reserved.
 * THIS IS UNPUBLISHED PROPRIETARY SOURCE CODE OF Avaya Inc.
 * The copyright notice above does not evidence any actual or intended publication of such source code.
 * Some third-party source code components may have been modified from their original versions by Avaya Inc.
 * The modifications are Copyright Avaya Inc., All Rights Reserved.
 * Avaya  Confidential & Restricted. May not be distributed further without written permission of
 * the Avaya owner.
 ****************************************************************************/

package com.avaya.collaboration.authorization.modules;

import java.io.IOException;
import java.security.GeneralSecurityException;
import java.security.KeyManagementException;
import java.security.KeyStore;
import java.security.KeyStoreException;
import java.security.NoSuchAlgorithmException;
import java.security.SecureRandom;
import java.security.UnrecoverableKeyException;
import java.security.interfaces.RSAPrivateKey;

import javax.net.ssl.KeyManagerFactory;
import javax.net.ssl.SSLContext;
import javax.net.ssl.TrustManagerFactory;

import com.avaya.collaboration.authorization.AccessTokenProvider;
import com.avaya.collaboration.authorization.ClientParams;
import com.avaya.collaboration.authorization.http.HttpClientProvider;
import com.avaya.collaboration.authorization.http.HttpClientProviderImpl;
import com.avaya.collaboration.authorization.impl.AccessTokenProviderImpl;
import com.google.inject.AbstractModule;
import com.google.inject.Provides;
import com.google.inject.Singleton;
import com.google.inject.name.Names;
import com.google.inject.throwingproviders.CheckedProvides;
import com.google.inject.throwingproviders.ThrowingProviderBinder;

public class ClientModule extends AbstractModule
{
    private KeyManagerFactory keyManagerFactoryInstance;
    private TrustManagerFactory trustManagerFactoryInstance;
    private ClientParams clientParams;

    public ClientModule(ClientParams clientParams) throws IOException, GeneralSecurityException
    {
        this.clientParams = clientParams;

        initKeyManager();
        initTrustManager();
    }

    private void initKeyManager() throws IOException, GeneralSecurityException
    {
        keyManagerFactoryInstance = KeyManagerFactory.getInstance("SunX509");
        keyManagerFactoryInstance.init(clientParams.getKeyStore(), clientParams.getKeyStorePassword().toCharArray());
    }

    private void initTrustManager() throws IOException, GeneralSecurityException
    {
        trustManagerFactoryInstance = TrustManagerFactory.getInstance("SunX509");
        trustManagerFactoryInstance.init(clientParams.getTrustStore());
    }

    @Override
    protected void configure()
    {
        install(ThrowingProviderBinder.forModule(this));

        bindConstant().annotatedWith(Names.named("tokenEndpoint")).to(clientParams.getTokenEndpoint());
        bindConstant().annotatedWith(Names.named("keyStorePassword")).to(clientParams.getKeyStorePassword());
        bindConstant().annotatedWith(Names.named("clientId")).to(clientParams.getClientId());
        bindConstant().annotatedWith(Names.named("privateKeyAlias")).to(clientParams.getKeyAlias());

        bind(HttpClientProvider.class).to(HttpClientProviderImpl.class);
        bind(AccessTokenProvider.class).to(AccessTokenProviderImpl.class);
    }

    @Provides
    @ClientKeyStore
    KeyStore provideKeyStore()
    {
        return clientParams.getKeyStore();
    }

    @Provides
    @ClientTrustStore
    KeyStore provideTrustStore()
    {
        return clientParams.getTrustStore();
    }

    @Singleton
    @CheckedProvides(SslProvider.class)
    SSLContext provideSslContext() throws KeyManagementException, NoSuchAlgorithmException
    {
        final SecureRandom secureRandom = new SecureRandom();
        secureRandom.nextInt();

        SSLContext sslContext = SSLContext.getInstance("TLS");
        sslContext.init(keyManagerFactoryInstance.getKeyManagers(),
                trustManagerFactoryInstance.getTrustManagers(),
                secureRandom);

        return sslContext;
    }

    @CheckedProvides(KeyProvider.class)
    RSAPrivateKey provideProvateKey() throws UnrecoverableKeyException, KeyStoreException, NoSuchAlgorithmException
    {
        return (RSAPrivateKey) clientParams.getKeyStore().getKey(clientParams.getKeyAlias(),
                clientParams.getKeyStorePassword().toCharArray());
    }
}
