package com.avaya.collaboration.authorization;

import static org.junit.Assert.assertNotNull;
import static org.junit.Assert.assertSame;
import static org.mockito.Matchers.any;
import static org.mockito.Matchers.anyListOf;
import static org.mockito.Matchers.anyString;
import static org.mockito.Mockito.times;
import static org.mockito.Mockito.verify;
import static org.mockito.Mockito.when;

import java.io.File;
import java.io.FileInputStream;
import java.io.FileNotFoundException;
import java.io.IOException;
import java.security.KeyStore;
import java.security.KeyStoreException;
import java.security.NoSuchAlgorithmException;
import java.security.cert.CertificateException;
import java.util.Arrays;

import javax.servlet.http.HttpServletRequest;

import org.junit.After;
import org.junit.Before;
import org.junit.BeforeClass;
import org.junit.Test;
import org.mockito.Mock;
import org.mockito.MockitoAnnotations;

import com.avaya.zephyr.common.testutils.deencapsulation.FieldReflection;

public class AuthorizationClientHelperTest
{

    private static KeyStore clientKeyStore;
    private static KeyStore clientTrustStore;

    @Mock
    AccessTokenProvider mockAccessTokenProvider;

    @Mock
    AccessToken mockAccessToken;

    @Mock
    HttpServletRequest mockHttpServletRequest;

    @BeforeClass
    public static void oneTimeSetUp() throws KeyStoreException, NoSuchAlgorithmException, CertificateException, FileNotFoundException,
    IOException
    {
        final ClassLoader classLoader = AuthorizationClientHelperTest.class.getClassLoader();
        final File keyStoreFile = new File(classLoader.getResource("clientkeystore.jks").getFile());

        clientKeyStore = KeyStore.getInstance("JKS");
        clientKeyStore.load(new FileInputStream(keyStoreFile), "password".toCharArray());

        final File trustStoreFile = new File(classLoader.getResource("clienttruststore.jks").getFile());

        clientTrustStore = KeyStore.getInstance("JKS");
        clientTrustStore.load(new FileInputStream(trustStoreFile), "password".toCharArray());
    }

    @Before
    public void setUp() throws Exception
    {
        MockitoAnnotations.initMocks(this);
    }

    @After
    public void tearDown() throws Exception
    {
    }

    @Test
    public void testBuilderInitSunnyDay() throws AuthorizationHelperException
    {
        final AuthorizationClientHelper asHelper = new AuthorizationClientHelper.Builder()
                .tokenEndpoint("https://1.1.1.1/services/AuthorizationService/token")
                .clientIdentifier("client_id")
                .keyStore(clientKeyStore, "password", "clientcert")
                .trustStore(clientTrustStore)
                .build();

        assertNotNull(asHelper);
    }

    @Test(expected = IllegalArgumentException.class)
    public void testBuilderUnsecureTokenEndpoint() throws AuthorizationHelperException
    {
        new AuthorizationClientHelper.Builder()
        .tokenEndpoint("http://1.1.1.1/services/AuthorizationService/token")
        .clientIdentifier("client_id")
        .keyStore(clientKeyStore, "password", "clientcert")
        .trustStore(clientTrustStore)
        .build();
    }

    @Test(expected = IllegalArgumentException.class)
    public void testBuilderInvalidTokenEndpoint() throws AuthorizationHelperException
    {
        new AuthorizationClientHelper.Builder()
        .tokenEndpoint("abracadabra")
        .clientIdentifier("client_id")
        .keyStore(clientKeyStore, "password", "clientcert")
        .trustStore(clientTrustStore)
        .build();
    }

    @Test(expected = IllegalArgumentException.class)
    public void testBuilderNullClientId() throws AuthorizationHelperException
    {
        new AuthorizationClientHelper.Builder()
        .tokenEndpoint("https://1.1.1.1/services/AuthorizationService/token")
        .clientIdentifier(null)
        .keyStore(clientKeyStore, "password", "clientcert")
        .trustStore(clientTrustStore)
        .build();
    }

    @Test(expected = IllegalArgumentException.class)
    public void testBuilderNullKeyStore() throws AuthorizationHelperException
    {
        new AuthorizationClientHelper.Builder()
        .tokenEndpoint("https://1.1.1.1/services/AuthorizationService/token")
        .clientIdentifier("client_id")
        .keyStore(null, "password", "clientcert")
        .trustStore(clientTrustStore)
        .build();
    }

    @Test(expected = IllegalArgumentException.class)
    public void testBuilderNullKeyStorePassword() throws AuthorizationHelperException
    {
        new AuthorizationClientHelper.Builder()
        .tokenEndpoint("https://1.1.1.1/services/AuthorizationService/token")
        .clientIdentifier("client_id")
        .keyStore(clientKeyStore, null, "clientcert")
        .trustStore(clientTrustStore)
        .build();
    }

    @Test(expected = IllegalArgumentException.class)
    public void testBuilderNullKeyStoreAlias() throws AuthorizationHelperException
    {
        new AuthorizationClientHelper.Builder()
        .tokenEndpoint("https://1.1.1.1/services/AuthorizationService/token")
        .clientIdentifier("client_id")
        .keyStore(clientKeyStore, "password", null)
        .trustStore(clientTrustStore)
        .build();
    }

    @Test(expected = IllegalArgumentException.class)
    public void testBuilderNullTrustStore() throws AuthorizationHelperException
    {
        new AuthorizationClientHelper.Builder()
        .tokenEndpoint("https://1.1.1.1/services/AuthorizationService/token")
        .clientIdentifier("client_id")
        .keyStore(clientKeyStore, "password", "clientcert")
        .trustStore(null)
        .build();
    }

    @Test
    public void testGetAccessToken() throws AuthorizationHelperException, HttpResponseException
    {
        final AuthorizationClientHelper asHelper = new AuthorizationClientHelper.Builder()
                .tokenEndpoint("https://1.1.1.1/services/AuthorizationService/token")
                .clientIdentifier("client_id")
                .keyStore(clientKeyStore, "password", "clientcert")
                .trustStore(clientTrustStore)
                .build();

        when(mockAccessTokenProvider.getAccessToken()).thenReturn(mockAccessToken);
        FieldReflection.setField(AuthorizationClientHelper.class, asHelper, "tokenProvider", mockAccessTokenProvider);

        assertSame(asHelper.getAccessToken(), mockAccessToken);
    }

    @Test
    public void testAccessTokenWithScopes() throws AuthorizationHelperException, HttpResponseException
    {
        final AuthorizationClientHelper asHelper = new AuthorizationClientHelper.Builder()
                .tokenEndpoint("https://1.1.1.1/services/AuthorizationService/token")
                .clientIdentifier("client_id")
                .keyStore(clientKeyStore, "password", "clientcert")
                .trustStore(clientTrustStore)
                .build();

        when(mockAccessTokenProvider.getAccessToken(anyListOf(String.class))).thenReturn(mockAccessToken);
        FieldReflection.setField(AuthorizationClientHelper.class, asHelper, "tokenProvider", mockAccessTokenProvider);

        assertSame(asHelper.getAccessToken(Arrays.asList("scopes")), mockAccessToken);
    }

    @Test
    public void testGetAccessTokenForUser() throws AuthorizationHelperException, HttpResponseException
    {
        final AuthorizationClientHelper asHelper = new AuthorizationClientHelper.Builder()
                .tokenEndpoint("https://1.1.1.1/services/AuthorizationService/token")
                .clientIdentifier("client_id")
                .keyStore(clientKeyStore, "password", "clientcert")
                .trustStore(clientTrustStore)
                .build();

        when(mockAccessTokenProvider.getAccessTokenForUser(anyString(), anyString())).thenReturn(mockAccessToken);
        FieldReflection.setField(AuthorizationClientHelper.class, asHelper, "tokenProvider", mockAccessTokenProvider);

        assertSame(asHelper.getAccessTokenForUser("user", "pass"), mockAccessToken);
    }

    @Test
    public void testGetAccessTokenForUserWithCode() throws AuthorizationHelperException, HttpResponseException
    {
        final AuthorizationClientHelper asHelper = new AuthorizationClientHelper.Builder()
                .tokenEndpoint("https://1.1.1.1/services/AuthorizationService/token")
                .clientIdentifier("client_id")
                .keyStore(clientKeyStore, "password", "clientcert")
                .trustStore(clientTrustStore)
                .build();

        when(mockAccessTokenProvider.getAccessTokenForUser(any(HttpServletRequest.class))).thenReturn(mockAccessToken);
        FieldReflection.setField(AuthorizationClientHelper.class, asHelper, "tokenProvider", mockAccessTokenProvider);

        assertSame(asHelper.getAccessTokenForUser(mockHttpServletRequest), mockAccessToken);
    }

    @Test
    public void testGetAccessTokenForUserWithScopes() throws AuthorizationHelperException, HttpResponseException
    {
        final AuthorizationClientHelper asHelper = new AuthorizationClientHelper.Builder()
                .tokenEndpoint("https://1.1.1.1/services/AuthorizationService/token")
                .clientIdentifier("client_id")
                .keyStore(clientKeyStore, "password", "clientcert")
                .trustStore(clientTrustStore)
                .build();

        when(mockAccessTokenProvider.getAccessTokenForUser(anyString(), anyString(), anyListOf(String.class))).thenReturn(mockAccessToken);
        FieldReflection.setField(AuthorizationClientHelper.class, asHelper, "tokenProvider", mockAccessTokenProvider);

        assertSame(asHelper.getAccessTokenForUser("user", "pass", Arrays.asList("scopes")), mockAccessToken);
    }

    @Test
    public void testShutDown() throws AuthorizationHelperException
    {

        final AuthorizationClientHelper asHelper = new AuthorizationClientHelper.Builder()
                .tokenEndpoint("https://1.1.1.1/services/AuthorizationService/token")
                .clientIdentifier("client_id")
                .keyStore(clientKeyStore, "password", "clientcert")
                .trustStore(clientTrustStore)
                .build();

        FieldReflection.setField(AuthorizationClientHelper.class, asHelper, "tokenProvider", mockAccessTokenProvider);

        asHelper.shutdown();

        verify(mockAccessTokenProvider, times(1)).shutDown();
    }
}
