package com.avaya.collaboration.authorization.impl;

import static org.junit.Assert.assertEquals;
import static org.junit.Assert.assertNotNull;
import static org.junit.Assert.assertTrue;
import static org.mockito.Matchers.any;
import static org.mockito.Matchers.anyString;
import static org.mockito.Mockito.when;

import java.io.File;
import java.io.FileInputStream;
import java.io.InputStream;
import java.security.KeyPair;
import java.security.KeyPairGenerator;
import java.security.KeyStoreException;
import java.security.NoSuchAlgorithmException;
import java.security.SecureRandom;
import java.security.UnrecoverableKeyException;
import java.security.interfaces.RSAPrivateKey;
import java.util.ArrayList;
import java.util.Arrays;

import javax.servlet.http.HttpServletRequest;

import org.apache.http.HttpEntity;
import org.apache.http.HttpStatus;
import org.apache.http.StatusLine;
import org.apache.http.client.methods.CloseableHttpResponse;
import org.apache.http.client.methods.HttpPost;
import org.apache.http.impl.client.CloseableHttpClient;
import org.junit.After;
import org.junit.Before;
import org.junit.BeforeClass;
import org.junit.Test;
import org.mockito.Mock;
import org.mockito.MockitoAnnotations;

import com.avaya.collaboration.authorization.AccessToken;
import com.avaya.collaboration.authorization.AuthorizationHelperException;
import com.avaya.collaboration.authorization.HttpResponseException;
import com.avaya.collaboration.authorization.http.HttpClientProvider;
import com.avaya.collaboration.authorization.modules.KeyProvider;
import com.nimbusds.jose.JOSEException;

public class AccessTokenProviderImplTest
{

    AccessTokenProviderImpl classToTest;

    @Mock
    CloseableHttpClient mockHttpClient;

    @Mock
    CloseableHttpResponse mockCloseableHttpResponse;

    @Mock
    StatusLine mockStatusLine;

    @Mock
    HttpEntity mockHttpEntity;

    @Mock
    HttpClientProvider mockClientProvider;

    @Mock
    HttpServletRequest mockHttpServletRequest;

    @Mock
    KeyProvider<RSAPrivateKey> mockKeyProvider;

    private static RSAPrivateKey privateKey;

    @BeforeClass
    public static void oneTimeSetUp() throws NoSuchAlgorithmException, JOSEException
    {
        initializeKeyPairs();
    }

    @Before
    public void setUp() throws Exception
    {
        MockitoAnnotations.initMocks(this);

        when(mockKeyProvider.get()).thenReturn(privateKey);

        when(mockClientProvider.getClientIntance()).thenReturn(mockHttpClient);
        when(mockHttpClient.execute(any(HttpPost.class))).thenReturn(mockCloseableHttpResponse);
        when(mockCloseableHttpResponse.getStatusLine()).thenReturn(mockStatusLine);
        when(mockStatusLine.getStatusCode()).thenReturn(HttpStatus.SC_OK);
        when(mockCloseableHttpResponse.getEntity()).thenReturn(mockHttpEntity);

        final ClassLoader classLoader = getClass().getClassLoader();
        final File file = new File(classLoader.getResource("sampleResponse.json").getFile());
        final InputStream fileInputStream = new FileInputStream(file);

        when(mockHttpEntity.getContent()).thenReturn(fileInputStream);

        classToTest =
                new AccessTokenProviderImpl(mockClientProvider, mockKeyProvider, "https://localhost/services/AuthorizationService/token",
                        "client_id");
    }

    @Test(expected = AuthorizationHelperException.class)
    public void testConstructorThrowsExceptionIfPrivateKeyIsNull() throws AuthorizationHelperException, UnrecoverableKeyException,
    KeyStoreException, NoSuchAlgorithmException
    {
        when(mockKeyProvider.get()).thenReturn(null);
        new AccessTokenProviderImpl(mockClientProvider, mockKeyProvider, "https://localhost/services/AuthorizationService/token",
                "client_id");
    }

    @Test
    public void testSunnyDayGetAccessToken() throws AuthorizationHelperException, HttpResponseException
    {
        final AccessToken tokenResponse = classToTest.getAccessToken();
        assertNotNull(tokenResponse);
        assertEquals(tokenResponse.toString(), "asdasdadadadad");
        assertTrue(tokenResponse.getScopes().size() == 2);
    }

    @Test
    public void testSunnyDayGetAccessTokenWithScopes() throws AuthorizationHelperException,
    HttpResponseException
    {
        final AccessToken tokenResponse = classToTest.getAccessToken(Arrays.asList("read", "write"));
        assertNotNull(tokenResponse);
        assertEquals(tokenResponse.toString(), "asdasdadadadad");
        assertTrue(tokenResponse.getScopes().size() == 2);
    }

    @Test(expected = AuthorizationHelperException.class)
    public void testGetAccessTokenWithEmptyScopes() throws AuthorizationHelperException,
    HttpResponseException
    {
        classToTest.getAccessToken(Arrays.asList(""));
    }

    @Test(expected = AuthorizationHelperException.class)
    public void testGetAccessTokenWithNullScopes() throws AuthorizationHelperException, HttpResponseException
    {
        classToTest.getAccessToken(Arrays.asList("sdsd", null));
    }

    @Test(expected = AuthorizationHelperException.class)
    public void testGetAccessTokenWithNullArgument() throws AuthorizationHelperException,
    HttpResponseException
    {
        classToTest.getAccessToken(null);
    }

    @Test
    public void testGetAccessTokenForUserSunnyDay() throws AuthorizationHelperException,
    HttpResponseException
    {
        final AccessToken tokenResponse = classToTest.getAccessTokenForUser("user", "pass");
        assertNotNull(tokenResponse);
        assertEquals(tokenResponse.toString(), "asdasdadadadad");
        assertEquals(tokenResponse.getExpiresIn(), 21423423);
    }

    //    @Test(expected = AuthorizationHelperException.class)
    //    public void testGetAccessTokenForUserWithNullUserName() throws AuthorizationHelperException,
    //    HttpResponseException
    //    {
    //        classToTest.getAccessTokenForUser(null, "pass");
    //    }

    @Test(expected = AuthorizationHelperException.class)
    public void testGetAccessTokenForUserWithEmptyUserName() throws AuthorizationHelperException,
    HttpResponseException
    {
        classToTest.getAccessTokenForUser("", "pass");
    }

    @Test(expected = AuthorizationHelperException.class)
    public void testGetAccessTokenForUserWithEmptyPassword() throws AuthorizationHelperException,
    HttpResponseException
    {
        classToTest.getAccessTokenForUser("user", "");
    }

    @Test(expected = AuthorizationHelperException.class)
    public void testGetAccessTokenForUserWithNullPassword() throws AuthorizationHelperException,
    HttpResponseException
    {
        classToTest.getAccessTokenForUser("user", null);
    }

    //    @Test(expected = AuthorizationHelperException.class)
    //    public void testGetAccessTokenForUserWithAllNull() throws AuthorizationHelperException,
    //    HttpResponseException
    //    {
    //        classToTest.getAccessTokenForUser(null, null);
    //    }

    @Test(expected = AuthorizationHelperException.class)
    public void testGetAccessTokenForUserWithAllEmpty() throws AuthorizationHelperException,
    HttpResponseException
    {
        classToTest.getAccessTokenForUser("", "");
    }

    @Test(expected = AuthorizationHelperException.class)
    public void testGetAccessTokenWithZeroSizeList() throws AuthorizationHelperException,
    HttpResponseException
    {
        classToTest.getAccessToken(new ArrayList<String>());
    }

    @Test
    public void testGetTokenForUserWithScopesSunnyDay() throws AuthorizationHelperException,
    HttpResponseException
    {
        final AccessToken tokenResponse =
                classToTest.getAccessTokenForUser("user", "pass", Arrays.asList("as", "rt"));
        assertNotNull(tokenResponse);
        assertEquals(tokenResponse.toString(), "asdasdadadadad");
    }

    @Test(expected = AuthorizationHelperException.class)
    public void testGetAccessTokenForUserWithEmptyScopes() throws AuthorizationHelperException,
    HttpResponseException
    {
        classToTest.getAccessTokenForUser("Abhi", "pass", Arrays.asList(""));
    }

    @Test(expected = AuthorizationHelperException.class)
    public void testGetAccessTokenForUserWithNullScopes() throws AuthorizationHelperException,
    HttpResponseException
    {
        classToTest.getAccessTokenForUser("Abhi", "pass", Arrays.asList("sdsd", null));
    }

    @Test(expected = AuthorizationHelperException.class)
    public void testGetAccessTokenForUserWithNullArgument() throws AuthorizationHelperException,
    HttpResponseException
    {
        classToTest.getAccessTokenForUser("Abhi", "pass", null);
    }

    @Test(expected = AuthorizationHelperException.class)
    public void testGetAccessTokenForUserWithZeroSizeList() throws AuthorizationHelperException,
    HttpResponseException
    {
        classToTest.getAccessTokenForUser("Abhi", "pass", new ArrayList<String>());
    }

    @Test(expected = AuthorizationHelperException.class)
    public void testGetAccessTokenForUserWithScopesWithNullUserName() throws AuthorizationHelperException,
    HttpResponseException
    {
        classToTest.getAccessTokenForUser(null, "pass", Arrays.asList("asd"));
    }

    @Test(expected = AuthorizationHelperException.class)
    public void testGetAccessTokenForUserWithScopesEmptyUserName() throws AuthorizationHelperException,
    HttpResponseException
    {
        classToTest.getAccessTokenForUser("", "pass", Arrays.asList("asd"));
    }

    @Test(expected = AuthorizationHelperException.class)
    public void testGetAccessTokenForUserWithScopesEmptyPassword() throws AuthorizationHelperException,
    HttpResponseException
    {
        classToTest.getAccessTokenForUser("user", "", Arrays.asList("asd"));
    }

    @Test(expected = AuthorizationHelperException.class)
    public void testGetAccessTokenForUserWithScopesNullPassword() throws AuthorizationHelperException,
    HttpResponseException
    {
        classToTest.getAccessTokenForUser("user", null, Arrays.asList("asd"));
    }

    @Test(expected = AuthorizationHelperException.class)
    public void testGetAccessTokenForUserWithScopesAllNull() throws AuthorizationHelperException,
    HttpResponseException
    {
        classToTest.getAccessTokenForUser(null, null, Arrays.asList("asd"));
    }

    @Test(expected = AuthorizationHelperException.class)
    public void testGetAccessTokenForUserWithScopesAllEmpty() throws AuthorizationHelperException,
    HttpResponseException
    {
        classToTest.getAccessTokenForUser("", "", Arrays.asList("asd"));
    }

    @Test
    public void testGetTokenForUserUsingCodeGrantSunnyDay() throws AuthorizationHelperException,
    HttpResponseException
    {
        final StringBuffer requestURL = new StringBuffer("http://example.com");
        when(mockHttpServletRequest.getParameter(anyString())).thenReturn("example_code");
        when(mockHttpServletRequest.getRequestURL()).thenReturn(requestURL);

        final AccessToken tokenResponse = classToTest.getAccessTokenForUser(mockHttpServletRequest);
        assertNotNull(tokenResponse);
        assertEquals(tokenResponse.toString(), "asdasdadadadad");
        assertTrue(tokenResponse.getScopes().size() == 2);
    }

    @After
    public void tearDown() throws Exception
    {
        classToTest.shutDown();
    }

    private static void initializeKeyPairs() throws NoSuchAlgorithmException
    {
        final KeyPairGenerator keyGenerator = KeyPairGenerator.getInstance("RSA");

        final SecureRandom random = SecureRandom.getInstance("SHA1PRNG");
        final byte[] values = new byte[20];
        random.nextBytes(values);

        keyGenerator.initialize(2048, random);

        final KeyPair kp = keyGenerator.genKeyPair();

        privateKey = (RSAPrivateKey) kp.getPrivate();
    }
}