/*
 * Copyright Avaya Inc., All Rights Reserved.
 *
 * THIS IS UNPUBLISHED PROPRIETARY SOURCE CODE OF Avaya Inc.
 *
 * The copyright notice above does not evidence any actual or intended
 * publication of such source code.
 *
 * Some third-party source code components may have been modified from their
 * original versions by Avaya Inc.
 *
 * The modifications are Copyright Avaya Inc., All Rights Reserved.
 */

package com.avaya.oceanalytics.openinterface.websocketclient.subscription;

import com.avaya.oceanalytics.openinterface.websocketclient.websocket.RealtimeDataProvider;
import com.fasterxml.jackson.databind.ObjectMapper;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Component;

/**
 * Producer of subscription redirect request send to websocket interface.
 *
 * @author Ivan Kovalev
 */

@Component
public class SubscriptionProducer {
    private static final Logger LOGGER = LoggerFactory.getLogger(SubscriptionProducer.class);
    private final ObjectMapper mapper;
    private final RealtimeDataProvider rtdProvider;

    @Autowired
    public SubscriptionProducer(ObjectMapper mapper,RealtimeDataProvider rtdProvider) {
        this.mapper = mapper;
        this.rtdProvider = rtdProvider;
    }

    /**
     * Send pumpup request for provided subscription data.
     *
     * @param authToken
     *          of the subscription initiator
     * @param subscription
     *          represents the data for subscribe request
     */

    public void sendPumpupRequest(String authToken, Subscription subscription) {
        sendRequest(authToken,subscription, SubscriptionRedirectRequestType.SUBSCRIPTIONREDIRECT);
    }

    /**
     * Send unsubscribe request for provided subscription data.
     *
     * @param authToken
     *          of the subscription initiator
     * @param subscription
     *          represents the data for subscribe request
     */


    public void sendUnsubscribeRequest(String authToken, Subscription subscription) {
        sendRequest(authToken,subscription, SubscriptionRedirectRequestType.UNSUBSCRIPTIONREDIRECT);
    }

    /**
     * Send subscribe request for provided subscription.
     *
     * @param authToken
     *          of the subscription initiator
     * @param subscription
     *          represents the data for subscribe request
     */
    public void sendSubscribeRequest(String authToken, Subscription subscription) {
        sendRequest(authToken,subscription, SubscriptionRedirectRequestType.SUBSCRIPTIONREDIRECT);
    }

    public void sendRequest(String authToken, Subscription subscription, SubscriptionRedirectRequestType requestType) {

        SubscriptionRedirectRequest request = new SubscriptionRedirectRequest();
        request.setAuthorization(authToken);
        request.setEventName(requestType.toString());
        request.setGuid(subscription.getGuid());
        request.setProducer(subscription.getProducer());
        request.setSource(subscription.getSource());
        request.setTransport(subscription.getTransport());
        request.setVersion("3.5");
        request.setEndpoint(subscription.getEndpoint());
        request.setProducerId(subscription.getProducerId());
        request.setStreamType(subscription.getStreamType());
        request.setSourceId(subscription.getSourceId());

        try {

            rtdProvider.send(mapper.writeValueAsString(request));

        } catch (Exception e) {
            LOGGER.error("Unexpected exception while sending subscription redirect request", e);
        }
    }
}
