package com.avaya.customer.example.servlet.api;

import org.codehaus.jackson.JsonNode;
import org.codehaus.jackson.annotate.JsonIgnore;
import org.codehaus.jackson.map.ObjectMapper;
import org.codehaus.jackson.node.ObjectNode;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import java.io.IOException;

public class MessageTranscriptV2 extends ChatMessage {

    private static final Logger LOG = LoggerFactory.getLogger(MessageTranscriptV2.class);
    private static final ObjectMapper OBJECT_MAPPER = new ObjectMapper();

    private String messageId;

    private String customData;

    private String richMediaType;

    public MessageTranscriptV2() {
    	// used in JSON conversion
    }

    /**
     * Create a new V2 transcript message.
     * @param sender
     * @param message
     * @param date
     * @param messageType
     * @param channelType
     * @param messageId
     * @param customData
     * @param richMediaType
     * @param isFallbackMessage
     */
    public MessageTranscriptV2(String sender, String message, long date, MessageType messageType,
            ChannelType channelType, String messageId, String customData, String richMediaType, boolean isFallbackMessage) {
        super(sender, message, date, messageType, channelType, isFallbackMessage);
        this.messageId = messageId;
        this.customData = customData;
        this.richMediaType = richMediaType;
    }

    public String getMessageId() {
        return messageId;
    }

    public void setMessageId(String messageId) {
        this.messageId = messageId;
    }

    public String getCustomData() {
        return customData;
    }

    public void setCustomData(String customData) {
        this.customData = customData;
    }

    public String getRichMediaType() {
        return richMediaType;
    }

    public void setRichMediaType(String richMediaType) {
        this.richMediaType = richMediaType;
    }

    /**
     * @return the filtered transcript message
     */
    @Override
    public MessageTranscriptV2 getFilteredMessage() {
        MessageType type = this.getMessageType();

        String customDataFiltered = customData;
        if(customData != null) {
            try {
                JsonNode parsedCustomData = OBJECT_MAPPER.readTree(customData);
                JsonNode messageNode = parsedCustomData.path("asyncMessageInfo").path("message");
                updateTextInCustomData(parsedCustomData, messageNode);
                updateTextInCustomData(parsedCustomData, messageNode.path("quotedMessage"));

                customDataFiltered = parsedCustomData.toString();
            } catch (IOException e) {
                LOG.info("Failed to parse customData. Will not filter");
            }
        }
        LOG.info("IsFallback: " + isFallbackMessage);

        return (type == MessageType.PAGEPUSH || isAsyncFile()) ? this : new MessageTranscriptV2(
                sender,
                message + " been filtered",
                date,
                messageType,
                channelType,
                messageId,
                customDataFiltered,
                richMediaType,
                isFallbackMessage
        );
    }

    private void updateTextInCustomData(JsonNode customData, JsonNode messageNode) {
        JsonNode messageText = messageNode.path("text");
        String filteredQuotedMessage = !messageText.isMissingNode() ?
                messageText.asText() + " been filtered" : "";
        if (!filteredQuotedMessage.isEmpty()) {
            ((ObjectNode)messageNode).put("text", filteredQuotedMessage);
        }
    }

    @JsonIgnore
    public boolean isAsyncFile() {
        return this.getMessageType() == MessageType.FILETRANSFER ||
                (this.getMessageType() == MessageType.RICHMEDIAMESSAGE && "file".equals(this.getRichMediaType()));
    }

    @Override
    public String toString() {
        return "MessageTranscriptV2{" + "messageId='" + messageId + '\'' + ", customData='" + customData + '\''
                + ", richMediaType='" + richMediaType + '\'' + ", sender='" + sender + '\'' + ", message='" + message
                + '\'' + ", date=" + date + ", messageType=" + messageType + ", channelType=" + channelType + ", isFallbackMessage=" + isFallbackMessage +'}';
    }
}
