package com.avaya.generic.channel.testclient;

import com.avaya.generic.channel.testclient.utils.ExternalException;
import com.avaya.generic.channel.testclient.utils.Utils;
import org.apache.commons.configuration2.PropertiesConfiguration;
import org.apache.commons.configuration2.builder.FileBasedBuilderParametersImpl;
import org.apache.commons.configuration2.builder.ReloadingDetectorFactory;
import org.apache.commons.configuration2.builder.ReloadingFileBasedConfigurationBuilder;
import org.apache.commons.configuration2.builder.fluent.Parameters;
import org.apache.commons.configuration2.convert.DisabledListDelimiterHandler;
import org.apache.commons.configuration2.ex.ConfigurationException;
import org.apache.commons.configuration2.ex.ConversionException;
import org.apache.commons.configuration2.io.ConfigurationLogger;
import org.apache.commons.configuration2.io.FileHandler;
import org.apache.commons.configuration2.reloading.ManagedReloadingDetector;
import org.apache.commons.configuration2.reloading.ReloadingDetector;

import java.io.File;
import java.util.EnumMap;
import java.util.Iterator;
import java.util.Map;

/**
 * The properties configuration.
 * This class does not parse property values.
 *
 * Properties that must be non-empty are accessed via getters.
 * Use opt() for properties that may be empty or commented out.
 */
class Configuration {

    Configuration(File propertiesFile) {
        builder = new ReloadingFileBasedConfigurationBuilder<PropertiesConfiguration>(PropertiesConfiguration.class)
                .configure(new Parameters().properties()
                        .setFile(propertiesFile)
                        .setThrowExceptionOnMissing(true)
                        .setListDelimiterHandler(new DisabledListDelimiterHandler())
                        .setIncludesAllowed(false)
                        .setReloadingDetectorFactory(new MyReloadingDetectorFactory())
                        .setLogger(ConfigurationLogger.newDummyLogger()));
    }

    File getPropertiesFile() {
        return builder.getFileHandler().getFile();
    }

    /**
     * Reload the properties and check if they are basically correct
     */
    void reload() throws ExternalException {

        // this hack forces the reloading
        myReloadingDetector.refresh();
        builder.getReloadingController().checkForReloading(null);

        // refreshing actual keys and checking for duplications
        actualKeys.clear();
        try {
            final Iterator<String> iterator = builder.getConfiguration().getKeys();

            while (iterator.hasNext()) {

                // check for duplication
                final String actualKey = iterator.next();
                try {
                    if (builder.getConfiguration().getList(actualKey).size() > 1) {
                        throw new ExternalException("Property %s duplicated", actualKey); // e.g. clientId and clientId
                    }
                } catch (ConversionException ce) {
                    // nothing. Catching just in case.
                }

                // map the key to the actual key
                final PropertyKey key = PropertyKey.byStringIgnoreCase(actualKey);

                if (key == null) {
                    // unknown property. It would be useful to show the list of supported properties
                    final StringBuilder sb = new StringBuilder()
                            .append("Unknown property ").append(actualKey).append("\n\n")
                            .append("Supported properties:");
                    for (PropertyKey supportedPropertyKey : PropertyKey.values()) {
                        sb.append("\n    ").append(supportedPropertyKey);
                    }
                    throw new ExternalException(sb.toString());
                }

                if (actualKeys.get(key) != null) {
                    throw new ExternalException("Property %s duplicated", key); // e.g. clientID and clientId
                }

                actualKeys.put(key, actualKey);
            }
        } catch (ConfigurationException ex) {
               //nothing. Catching just in case.
        }

    }

    String getAuthorizationHost() throws ExternalException, ConfigurationException  {
        return get(PropertyKey.authorizationHost);
    }

    String getApiHost() throws ExternalException, ConfigurationException  {
        return get(PropertyKey.apiHost);
    }

    String getCoreDataServiceHost() throws ExternalException, ConfigurationException  {
        return get(PropertyKey.coreDataServiceHost);
    } 

    String getClientId() throws ExternalException, ConfigurationException  {
        return get(PropertyKey.clientId);
    }

    String getKeyStore() throws ExternalException, ConfigurationException  {
        return get(PropertyKey.keyStore);
    }

    String getKeyStorePassword() throws ExternalException, ConfigurationException  {
        return get(PropertyKey.keyStorePassword);
    }

    String getKeyAlias() throws ExternalException, ConfigurationException  {
        return get(PropertyKey.keyAlias);
    }

    String getTrustStore() throws ExternalException, ConfigurationException  {
        return get(PropertyKey.trustStore);
    }

    String getTrustStorePassword() throws ExternalException, ConfigurationException  {
        return get(PropertyKey.trustStorePassword);
    }

    String getEventConnectorHost() throws ExternalException, ConfigurationException  {
        return get(PropertyKey.eventConnectorHost);
    }

    String getEventFamily() throws ExternalException, ConfigurationException  {
        return get(PropertyKey.eventFamily);
    }

    String getEventCallbackHost() throws ExternalException, ConfigurationException  {
        return get(PropertyKey.eventCallbackHost);
    }

    String getEventCallbackPort() throws ExternalException, ConfigurationException  {
        return get(PropertyKey.eventCallbackPort);
    }

    String getEventSubscription() throws ExternalException, ConfigurationException  {
        return get(PropertyKey.eventSubscription);
    }
    
    String getUesrname() throws ExternalException, ConfigurationException  {
        return get(PropertyKey.username);
    }
    
    String getPassword() throws ExternalException, ConfigurationException  {
        return get(PropertyKey.password);
    }
    
    String getAuthPort() throws ExternalException, ConfigurationException  {
        return get(PropertyKey.authPort);
    }

    void setSsoToken(String ssoToken) throws ExternalException, ConfigurationException {
        set(PropertyKey.ssoToken, ssoToken);
    }

    void setEventSubscription(String eventSubscription) throws ExternalException, ConfigurationException {
        set(PropertyKey.eventSubscription, eventSubscription);
    }
    
    String getNumContactsForClosing()throws ExternalException, ConfigurationException  {
        return get(PropertyKey.numContactsForClosing);
    } 
    
    String getNumContactsForShowing()throws ExternalException, ConfigurationException  {
        return get(PropertyKey.numContactsForShowing);
    } 

    /**
     * @returns the property value, or null when the property not set
     */
    String opt(PropertyKey key) throws ConfigurationException {
        return builder.getConfiguration().getString(actualKeyFor(key), null);
    }

    //=========================== private ===========================

    /**
     * @returns a non-empty property value
     * @throws exception when the property is empty or not set
     */
    private String get(PropertyKey key) throws ExternalException, ConfigurationException{
        final String value = opt(key);
        if (Utils.isNotEmpty(value)) {
            return value;
        } else {
            throw new ExternalException("Property %s %s", key, (value != null ? "is empty" : "not specified"));
        }
    }

    private void set(PropertyKey key, String newValue) throws ConfigurationException{
        builder.getConfiguration().setProperty(actualKeyFor(key), (newValue != null) ? newValue : "");
        builder.save();
        Utils.println("Property %s %s", key, (Utils.isNotEmpty(newValue) ? "updated" : "cleared"));
    }

    private String actualKeyFor(PropertyKey key) {
        String actualKey = actualKeys.get(key);
        if (actualKey == null) {
            actualKey = key.name();
        }
        return actualKey;
    }

    private class MyReloadingDetectorFactory implements ReloadingDetectorFactory {
        @Override
        public ReloadingDetector createReloadingDetector(FileHandler fileHandler, FileBasedBuilderParametersImpl fileBasedBuilderParameters) throws ConfigurationException {
            return myReloadingDetector;
        }
    }
    private final ManagedReloadingDetector myReloadingDetector = new ManagedReloadingDetector();
    private final ReloadingFileBasedConfigurationBuilder<PropertiesConfiguration> builder;

    // PropertyKey-to-KeyInPropertiesFile map.
    // Purpose: to ignore case of keys in the properties file. Otherwise it would complain a lot. "clientID" should be as correct as "clientId".
    // The actual key always equalsIgnoreCase the key.
    private final Map<PropertyKey, String> actualKeys = new EnumMap<PropertyKey, String>(PropertyKey.class);
}
