package com.avaya.generic.channel.testclient;

import com.avaya.generic.channel.testclient.utils.ExternalException;
import com.avaya.generic.channel.testclient.utils.Rest;
import com.avaya.generic.channel.testclient.utils.RestResponse;
import com.avaya.generic.channel.testclient.utils.Utils;
import java.io.IOException;
import java.security.KeyManagementException;
import java.security.NoSuchAlgorithmException;
import net.minidev.json.JSONArray;
import org.apache.commons.configuration2.ex.ConfigurationException;
import org.apache.http.HttpStatus;
import org.json.JSONObject;

/**
 * REST client for EventingConnector
 * Some kinds of responses are recognized and printed in short forms.
 * If a response if not recognized, it is printed as is.
 *
 * This class does not know of EventListener.
 */
public class EventingConnectorClient {

    EventingConnectorClient(Configuration configuration) {
        this.configuration = configuration;
    }


    void newSubscription() throws ConfigurationException, ExternalException, IOException {

        // prepare the request payload
        final String callbackHost = configuration.getEventCallbackHost();
        final int callbackPort = Integer.parseInt(configuration.getEventCallbackPort());
        final String family = configuration.getEventFamily();
        final String[] eventTypes = optEventTypesAsArray();
        final String metadata = configuration.opt(PropertyKey.eventMetadata);
   
        JSONObject filter = new JSONObject();
        JSONArray criteria = new JSONArray();
        int criteriaIndex = 0;

        JSONObject requestJson = new JSONObject()
                .put("family", family)
                .putOpt("eventTypes", eventTypes)
                .put("callbackUrl", Utils.buildUrl(callbackHost + ":" + callbackPort))
                .put("callbackContentType", "application/json");
        

        // add metadata parameters
        if (Utils.isNotEmpty(metadata)) {
            final String[] parts = metadata.split("\\s*,\\s*");
            for (final String part : parts) {
                final String[] keyValuePair = part.split("\\s*=\\s*");
                JSONObject criterias = new JSONObject();
                if (keyValuePair.length == 2 && Utils.isNotEmpty(keyValuePair[0]) && Utils.isNotEmpty(keyValuePair[1])) {
                    criterias.put("key", keyValuePair[0]);
                    criterias.put("value", keyValuePair[1]);
                    criteria.add(criteriaIndex++, criterias);
                    filter.putOpt("criteria",criteria);
                } else {
                    throw new ExternalException("%s: '%s' is not a key=value pair", PropertyKey.eventMetadata, part);
                }
            }
            requestJson.put("filter", filter);
        }
      
        
        // execute the request
        final String connectorHost = configuration.getEventConnectorHost();
        final String requestUrl = Utils.buildUrl(connectorHost, SUBSCRIPTIONS_PATH);
        final RestResponse response = Rest.post(requestUrl, requestJson);

        if (response.isOK()) {

            // save the subscription
            final String id = new JSONObject(response.getContent()).getString("subscriptionId");
            Utils.println("New subscription created");
            final JSONObject subscriptionSnapshotJson = new JSONObject()
                    .put("id", id)
                    .put(PropertyKey.eventConnectorHost.name(), connectorHost)
                    .put(PropertyKey.eventFamily.name(), family)
                    .putOpt(PropertyKey.eventTypes.name(), eventTypes)
                    .put(PropertyKey.eventCallbackHost.name(), callbackHost)
                    .put(PropertyKey.eventCallbackPort.name(), callbackPort);

            if (Utils.isNotEmpty(metadata)) {
                subscriptionSnapshotJson.put(PropertyKey.eventMetadata.name(), metadata);
            }

            configuration.setEventSubscription(subscriptionSnapshotJson.toString());

        } else {
            Utils.println(response.toString()); // something's wrong. Print the response as is.
        }
    }

    void deleteSubscription() throws Exception {

        final RestResponse response = Rest.delete(getSubscriptionControlUrl());

        if (response.isOK()) {

            Utils.println("Subscription deleted");
            configuration.setEventSubscription("");
            return;

        } else if (response.getCode() == HttpStatus.SC_NOT_FOUND) {

            // method not found or subscription not found?
            try {
                final String message = new JSONObject(response.getContent()).getString("message");
                if (Utils.isNotEmpty(message) && message.toLowerCase().contains("not found")) {
                    Utils.println("Subscription not found");
                    configuration.setEventSubscription("");
                    return;
                }
            } catch (Exception ex) {
                // nothing
            }
        }

        Utils.println(response.toString()); // something's wrong. Print the response as is.
    }

    void querySubscription() throws Exception {
        final RestResponse response = Rest.get(getSubscriptionControlUrl());
        Utils.println(response.toString());
    }

    void renewSubscription() throws Exception {
        final RestResponse response = Rest.put(getSubscriptionControlUrl());
        if (response.isOK()) {
            Utils.println("Subscription renewed");
        } else {
            Utils.println(response.toString()); // something's wrong. Print the response as is.
        }
    }

    //========================================== private ======================================================

    /**
     * Reads the eventSubscription property and
     * @returns the EventingConnector service endpoint for query/renew/delete this subscription
     */
    private String getSubscriptionControlUrl() throws ExternalException, ConfigurationException  {
        final JSONObject json = new JSONObject(configuration.getEventSubscription());
        final String eventConnectorHost = json.getString(PropertyKey.eventConnectorHost.name());
        final String id = json.getString("id");
        return Utils.buildUrl(eventConnectorHost, SUBSCRIPTIONS_PATH, id);
    }

    /**
     * @return the eventTypes property as an array, or an empty array if the property is empty, or null if the property is not specified
     */
    private String[] optEventTypesAsArray() throws ConfigurationException  {
        final String eventTypes = configuration.opt(PropertyKey.eventTypes);
        if (Utils.isNotEmpty(eventTypes)) {
            return eventTypes.split("\\s*,\\s*");
        } else if (eventTypes != null) {
            return new String[]{};
        } else {
            return null;
        }
    }

    private final Configuration configuration;

    private static final String SUBSCRIPTIONS_PATH = "/services/EventingConnector/subscriptions";
}
