package com.avaya.generic.channel.testclient.utils;

import java.io.BufferedReader;
import java.io.BufferedWriter;
import java.io.FileNotFoundException;
import java.io.FileReader;
import java.io.FileWriter;
import java.io.IOException;
import java.util.Arrays;

/**
 * Encapsulates the "<<|>|>> file" functionality.
 * << read
 * > overwrite
 * >> append
 *
 * The open*() methods cut the "<<|>|>> file" part from the args if the file is opened.
 */
public class FileIO {

    /**
     * Opens a file if << or > or >> found in the args.
     * @return the rest of the args
     */
    public String[] open(final String[] args) throws IOException {
        return openImpl(args, new String[]{"<<", ">", ">>"});
    }

    /**
     * Opens a file if << found in the args.
     * @return the rest of the args
     */
    public String[] openForReading(final String[] args) throws IOException  {
        return openImpl(args, new String[]{"<<"});
    }

    /**
     * Opens a file if > or >> found in the args.
     * @return the rest of the args
     */
    public String[] openForWriting(final String[] args) throws IOException  {
        return openImpl(args, new String[]{">", ">>"});
    }

    public boolean hasFile() {
        return hasReader() || hasWriter();
    }

    public boolean hasReader() {
        return reader != null;
    }

    public boolean hasWriter() {
        return writer != null;
    }

    /**
     * @return a non-empty line or null if EOF reached or there is no reader
     */
    public String readLine() throws IOException  {

        if (reader == null) {
            return null;
        }

        while (true) {
            String line = reader.readLine();
            if (line == null) {
                return null;
            }
            line = line.trim();
            if (!line.isEmpty()) {
                return line;
            }
        }
    }

    public void writeLine(String line) throws IOException {
        if (writer != null) {
            writer.write(line);
            writer.newLine();
            writer.flush();
        }
    }


    public void close() throws IOException  {

        if (reader != null) {
            reader.close();
            reader = null;
        }

        if (writer != null) {
            writer.close();
            writer = null;
        }
    }

    //=============================== private =============================================

    private String[] openImpl(final String[] args, final String[] expectedOperators) throws FileNotFoundException, IOException  {

        close();

        final int length = args.length;
        if (length < 2) {
            return args;
        }

        final String operator = args[length - 2];

        if (!Arrays.asList(expectedOperators).contains(operator)) {
            return args;
        }

        final String fileName = args[length - 1];

        switch (operator) {
            case "<<":
                reader = new BufferedReader(new FileReader(fileName));  // read
                break;

            case ">":
                writer = new BufferedWriter(new FileWriter(fileName, false)); // overwrite
                break;

            case ">>":
                writer = new BufferedWriter(new FileWriter(fileName, true)); // append
                break;

            default:
                return args;
        }

        return Arrays.copyOfRange(args, 0, length - 2);
    }


    private BufferedReader reader = null;
    private BufferedWriter writer = null;
}
