package com.avaya.generic.channel.testclient;

import com.avaya.generic.channel.testclient.utils.ExternalException;
import com.avaya.generic.channel.testclient.utils.FileIO;
import com.avaya.generic.channel.testclient.utils.Rest;
import com.avaya.generic.channel.testclient.utils.RestResponse;
import com.avaya.generic.channel.testclient.utils.SynchronousTimer;
import com.avaya.generic.channel.testclient.utils.Utils;
import com.google.gson.JsonObject;
import com.google.gson.JsonParser;
import org.apache.commons.cli.CommandLine;
import org.apache.commons.cli.DefaultParser;
import org.apache.commons.cli.HelpFormatter;
import org.apache.commons.cli.Option;
import org.apache.commons.cli.Options;
import org.apache.commons.cli.ParseException;
import org.json.JSONObject;

import java.io.BufferedReader;
import java.io.BufferedWriter;
import java.io.FileReader;
import java.io.FileWriter;
import java.io.IOException;
import java.security.KeyManagementException;
import java.security.NoSuchAlgorithmException;
import java.text.SimpleDateFormat;
import java.util.ArrayList;
import java.util.Date;
import java.util.EnumSet;
import java.util.List;
import java.util.Locale;
import java.util.logging.Level;
import java.util.logging.Logger;
import jdk.nashorn.internal.objects.NativeArray;
import org.apache.commons.configuration2.ex.ConfigurationException;
import org.apache.commons.lang.exception.ExceptionUtils;
import org.apache.http.HttpStatus;
import org.apache.http.client.methods.HttpPost;
import org.apache.http.entity.StringEntity;

/**
 * Parses and executes user commands
 * Supported commands:
 *      status
 *      getToken
 *      getBearerToken
 *      attributes
 *      routepoints
 *      create -n integer [-r integer] [>|>> file]
 *      create [-r integer] << file
 *      create contactId
 *      drop [-r integer] << file
 *      drop contactId
 *      events -s new
 *      events -s del
 *      events -s query
 *      events -s renew
 *      events
 */

class CommandProcessor {

    CommandProcessor(Configuration configuration, EventListener eventListener) {
        this.eventListener = eventListener;
        this.gcapiClient = new GenericChannelAPIClient(configuration);
        this.сoreDataServiceClientClient = new CoreDataServiceClient(configuration);
        this.eventingConnectorClient = new EventingConnectorClient(configuration);
        this.configuration = configuration;
    }

    /**
     * Executes an test client command
     * @param line command line as it is
     */
    void execute(String line) throws Exception  {

        // split the line
        final String[] splittedLine = line.split("\\s+");
        if (splittedLine.length == 0) {
            return;
        }

        // recognize command name
        final CommandName commandName = CommandName.byStringIgnoreCase(splittedLine[0]);
        if (commandName == null) {
            Utils.println("Unknown command");
            return;
        }

        // populate the array of arguments
        String[] args = new String[splittedLine.length-1];
        for (int i=1; i<splittedLine.length; i++) {
            args[i-1] = splittedLine[i];
        }

        // reload properties
        configuration.reload();

        // execute command
        switch (commandName) {
            case STATUS:        status();                         break;
            case GETTOKEN:      getToken();                       break;
            case GETBEARERTOKEN: getBearerToken();                 break;
            case GETPUBLICTOKEN: getPublicToken();                break;
            case CREATE:        create(args);                     break;
            case DROP:          drop(args);                       break;
            case ATTRIBUTES:    gcapiClient.getAttributes();      break;
            case ROUTEPOINTS:   gcapiClient.getRoutePoints();     break;
            case EVENTS:        events(args);                     break;
            case CREATECONTEXT: getСontextID(args);               break;
            case DROPALLCONTACTS: dropAllContacts();                break;
            case GETACTIVECONTACTS: getActiveContacts();            break;
            default: Utils.println("Unknown command");            break;
        }
    }

    //========================== private =================================

    void status() throws ConfigurationException  {
        List<EnumSet<PropertyKey>> keySetList = new ArrayList<>(3);
        keySetList.add(PropertyKey.authorizationGroup());
        keySetList.add(PropertyKey.eventingGroup());
        keySetList.add(PropertyKey.createContactGroup());

        for (EnumSet<PropertyKey> keySet : keySetList) {

            int maxLen = 0; // the length of the longest key in this group
            for (PropertyKey key : keySet) {
                if (key.name().length() > maxLen) {
                    maxLen = key.name().length();
                }
            }

            for (PropertyKey key : keySet) {
                String value = configuration.opt(key);
                if (key == PropertyKey.ssoToken) {
                    value = Utils.trimTo16chars(value);
                }
                // print key
                Utils.print(key.name());
                for (int i=key.name().length(); i<maxLen; i++) {
                    Utils.print(" ");
                }
                // print value
                if (value != null) {
                    Utils.println(" = '%s'", value);
                } else {
                    Utils.println(" not set");
                }
            }
            Utils.printSeparator();
        }
    }

    private void getToken() throws Exception {
        final String ssoToken = Authorization.getAccessToken(configuration).toString();
        configuration.setSsoToken(ssoToken);
    }
    
    private void getBearerToken() throws Exception {
        final String ssoToken = Authorization.getAccessToken(configuration).getTokenType() + " " + Authorization.getAccessToken(configuration).toString();
        configuration.setSsoToken(ssoToken);
    }
    
    private void getPublicToken() throws Exception{
        String username = configuration.getUesrname();
        String password = configuration.getPassword();
        StringBuilder stringRequest  = new StringBuilder(256).append("grant_type=password&username=").append(username).append("&password=").append(password);
        final String url = Utils.buildHttpSecureUrl(configuration.getAuthorizationHost() + ":" + configuration.getAuthPort(), "/services/AuthorizationService/token");
        final HttpPost request = new HttpPost(url);
        request.addHeader("content-type", "application/x-www-form-urlencoded");
        final StringEntity stringEntity = new StringEntity(stringRequest.toString());        
        request.setEntity(stringEntity);
        try {
            RestResponse response = Rest.execute(request);
            JsonParser parser = new JsonParser();
            JsonObject responseJson = parser.parse(response.getContent()).getAsJsonObject();
            String ssoToken = responseJson.get("access_token").getAsString();
            configuration.setSsoToken(ssoToken);
        } catch (Exception ex) {
            Utils.println("Incorrect credentials");
        }
    }

    /**
     * parse args and call an impl method
     */
    private void create(String[] args) throws Exception {

        // todo: consider using FileIO class when changing something here
        final Options  options = new Options().addOption(contactIDOption).addOption(contextIDOption).addOption(numberOption).addOption(rateOption).addOption(dataOption);            

        final String[] examples = {
                "create -n integer [-r integer] [>|>> file]",
                "create [-r integer] << file",
                "create contactId", 
                "create -c contextID -i contactID",
                "create -d key:value -i contactID -c contextID"};

        try {
            CommandLine cmdLine = new DefaultParser().parse(options, args);
            if (cmdLine.hasOption("n")) {
                createTimeStampedImpl(cmdLine);
            } else {
                createAndDropImpl(CommandName.CREATE, cmdLine); // we are using unified code where possible
            }

        } catch (ParseException pe) {
            Utils.println(pe.getMessage());
            printUsage(examples, options);
        }
    }

    /**
     * parse args and call an impl method
     */    
    private void getСontextID(String[] args) {
        final Options options = new Options().addOption(contactIDOption);
        final String[] examples = {
                "createcontext contextId"};

        try {
            CommandLine cmdLine = new DefaultParser().parse(options, args);
            createAndDropImpl(CommandName.CREATECONTEXT, cmdLine); // we are using unified code where possible

        } catch (ParseException pe) {
            Utils.println(pe.getMessage());
            printUsage(examples, options);
        } catch (Exception ex) {
            Utils.println("CommandProcessor.class.getName()" + ex.getMessage());
        }        
    }

    /**
     * parse args and call an impl method
     */
    private void drop(String[] args) throws Exception {

        final Options options = new Options().addOption(rateOption);
        final String[] examples = {
                "drop [-r integer] << file",
                "drop contactId"};

        try {
            CommandLine cmdLine = new DefaultParser().parse(options, args);
            createAndDropImpl(CommandName.DROP, cmdLine);
        } catch (ParseException pe) {
            Utils.println(pe.getMessage());
            printUsage(examples, options);
        }


    }
    
    private void dropAllContacts() throws Exception{
        try{
            gcapiClient.dropAllContacts();
        } catch (Exception e) {
            Utils.println(ExceptionUtils.getStackTrace(e));
        }
    }
    
    private void getActiveContacts() throws Exception{
        try{
            gcapiClient.getActiveContacts();
        } catch (Exception e) {
            Utils.println(ExceptionUtils.getStackTrace(e));
        }
    }

    /**
     * Unified implementation for create and drop
     *      create [-r rate] << file
     *      create contactId
     *      drop [-r rate] << file
     *      drop contactId
     */
    private void createAndDropImpl(CommandName commandName, CommandLine cmdLine) throws Exception {

        SynchronousTimer timer = new SynchronousTimer();
        BufferedReader reader = null;
        try {
            Integer rate = getOptionInteger(cmdLine, "r");
            String contactId = null;
            String contextID = null;
            String extraData = null;

            final List<String> leftOverArgs = cmdLine.getArgList();

            switch (leftOverArgs.size()) {
                case 0:
                    if (cmdLine.hasOption("c") && cmdLine.hasOption("i") && cmdLine.hasOption("d")) {
                        contextID = cmdLine.getOptionValue("c");
                        contactId = cmdLine.getOptionValue("i");
                        extraData = cmdLine.getOptionValue("d");
                    } else if (cmdLine.hasOption("i") && cmdLine.hasOption("c")) {
                        contactId = cmdLine.getOptionValue("i");
                        contextID = cmdLine.getOptionValue("c");
                    }
                    break;
                case 1:
                    contactId = leftOverArgs.get(0);
                    break;
                case 2:
                        assertTrue("<<".equals(leftOverArgs.get(0)));
                        reader = new BufferedReader(new FileReader(leftOverArgs.get(1)));
                    break;
                default:
                    throw new ParseException("Too many arguments");
            }

            if (reader != null) {
                rate = (rate != null) ? rate : 1; // rate is optional
                assertTrue(rate > 0);
                timer.start(rate);
                int linesCounter = 0;
                while ((contactId = reader.readLine()) != null) {
                    contactId = contactId.trim();
                    if (validateContactId(contactId)) {
                        timer.waitForTick();
                        finishCreateOrDrop(commandName, contactId, "", "");
                        linesCounter++;
                    }
                }
                if (linesCounter == 0) {
                    Utils.println("No contact ID found in the file");
                }
            } else if (contactId != null) {
                assertTrue(rate == null); // rate must be absent
                finishCreateOrDrop(commandName, contactId, (contextID!=null ? contextID : ""), (extraData!=null ? extraData : ""));
            } else {
                throw new ParseException("Too few arguments");
            }

        } catch (NumberFormatException nfe) {
            throw new ParseException(nfe.getMessage());
        } finally {
            timer.stop();
            if (reader != null) {
                reader.close();
            }
        }
    }

    private void finishCreateOrDrop(CommandName commandName, String contactId, String contextID, String extraData) throws Exception {
        switch (commandName) {
            case CREATE:
                gcapiClient.createContact(contactId, contextID, extraData);
                break;
            case DROP:
                gcapiClient.dropContact(contactId);
                break;
            case CREATECONTEXT:
                сoreDataServiceClientClient.getСontextID(contextID, extraData.trim());
            default:
                Utils.println("Unknown command");
                break;
        }
    }


    /**
     * Special implementation for "create -n integer [-r integer] [(>|>>) file]"
     */
    private void createTimeStampedImpl(CommandLine cmdLine) throws Exception {

        SynchronousTimer timer = new SynchronousTimer();
        BufferedWriter writer = null;
        try {

            final Integer number = getOptionInteger(cmdLine, "n");
            assertTrue(number != null && number > 0);

            Integer rate = getOptionInteger(cmdLine, "r");
            rate = (rate != null) ? rate : 1;
            assertTrue(rate > 0);
            
            final String optionalContactIdPrefix = cmdLine.getOptionValue("i");
            final List<String> leftOverArgs = cmdLine.getArgList();
            switch (leftOverArgs.size()) {
                case 0:
                    break;
                case 2: {
                    switch (leftOverArgs.get(0)) {
                        case ">":
                            writer = new BufferedWriter(new FileWriter(leftOverArgs.get(1), false)); // overwrite
                            break;
                        case ">>":
                            writer = new BufferedWriter(new FileWriter(leftOverArgs.get(1), true)); // append
                            break;
                        default:
                            assertTrue(false); // illegal arguments
                            break;
                    }
                    break;
                }
                default:
                    assertTrue(false); // illegal arguments
                    break;
            }

            final double expectedTimeInSec = (double)number * (1.0 / (double)rate);
            if (expectedTimeInSec > 10.0) {
                Utils.println("Expected time is %.1f s", expectedTimeInSec);
            }

            timer.start(rate);
            for (int i=0; i<number; i++) {
                timer.waitForTick();
                String contactIdPrefix;
                if(Utils.isNullOrEmpty(optionalContactIdPrefix)) {
                    contactIdPrefix = configuration.opt(PropertyKey.contactIdPrefix);
                } else {
                    contactIdPrefix = optionalContactIdPrefix;
                }
                final SimpleDateFormat dateFormat = new SimpleDateFormat("yyyyMMMddHHmmssSSS", Locale.ENGLISH);
                final String contactId = (contactIdPrefix != null ? contactIdPrefix : "") + dateFormat.format(new Date());
                if (gcapiClient.createContact(contactId, "", "") && writer != null) {
                    writer.write(contactId);
                    writer.newLine();
                }
            }

        } catch (NumberFormatException nfe) {
            throw new ParseException(nfe.getMessage());
        } finally {
            timer.stop();
            if (writer != null) {
                writer.close();
            }
        }
    }


    private void events(final String[] args) throws Exception {

        final Options options = new Options().addOption(
                Option.builder("s").hasArg().argName("subscription operation").desc("new|del|query|renew").build());
        final String[] examples = {
                "events",
                "events >|>> file",
                "events -s new|del|query|renew"
        };

        final FileIO fileIO = new FileIO();
        try {
            // open file if any
            final String[] argsNoFile = fileIO.openForWriting(args);

            // parse the rest of the arguments
            final CommandLine cmdLine = new DefaultParser().parse(options, argsNoFile);
            assertTrue(cmdLine.getArgList().isEmpty());

            // execute the command
            if (cmdLine.hasOption('s')) {
                assertFalse(fileIO.hasFile());
                final String operation = cmdLine.getOptionValue('s').toLowerCase();
                switch (operation) {
                    case "new": newSubscription(); break;
                    case "del": eventingConnectorClient.deleteSubscription(); break;
                    case "query": eventingConnectorClient.querySubscription(); break;
                    case "renew": eventingConnectorClient.renewSubscription(); break;
                    default: throw new ParseException("Illegal argument of the -s option");
                }
            } else {
                listenToEvents(fileIO);
            }

        } catch (ParseException pe) {
            Utils.println(pe.getMessage());
            printUsage(examples, options);

        } finally {
            fileIO.close();
        }
    }

    //============================= private =============================================================

    /**
     * 'events -s new' implementation
     */
    private void newSubscription() throws ExternalException, IOException, NoSuchAlgorithmException, KeyManagementException, ConfigurationException {

        // check that the subscription does not exist
        final String eventSubscription = configuration.opt(PropertyKey.eventSubscription);
        if (Utils.isNotEmpty(eventSubscription)) {
            throw new ExternalException("Please delete the current subscription first");
        }

        // start EventListener so that EventingConnector can check the callbackUrl
        final String callbackHost = configuration.getEventCallbackHost();
        final int callbackPort = Integer.parseInt(configuration.getEventCallbackPort());
        eventListener.start(callbackHost, callbackPort); // so that EventingConnector can check the callbackUrl

        // execute a create subscription request to EventingConnector
        eventingConnectorClient.newSubscription();
    }

    /**
     * 'events' no arguments implementation
     * @param fileIO the FileIO object for optional writing the events to a file
     */
    private void listenToEvents(final FileIO fileIO) throws IOException, ExternalException, KeyManagementException, NoSuchAlgorithmException, ConfigurationException  {

        try {
            // start EventListener
            final JSONObject json = new JSONObject(configuration.getEventSubscription());
            final String subscriptionId = json.getString("id");
            final String callbackHost = json.getString(PropertyKey.eventCallbackHost.name());
            final int callbackPort = json.getInt(PropertyKey.eventCallbackPort.name());
            eventListener.start(callbackHost, callbackPort);

            // Print some info about what we are listening to
            Utils.println("Listening to events.");
            final PropertyKey[] keys = { PropertyKey.eventFamily, PropertyKey.eventTypes, PropertyKey.eventMetadata };
            for (PropertyKey key : keys) {
                final String value = json.optString(key.name(), "");
                if (Utils.isNotEmpty(value)) {
                    Utils.println("  %s: %s", key.name().substring(5), value);
                }
            }

            // listen to events and wait for user interruption
            Utils.println("Press Enter to stop.");
            eventListener.showEvents(fileIO, subscriptionId);
            Utils.readLine();

        } finally {
            eventListener.hideEvents();
        }
    }

    private Integer getOptionInteger(CommandLine cmdLine, String option) {
        final String str = cmdLine.getOptionValue(option);
        return (str != null) ? Integer.parseInt(str) : null;
    }

    private void assertTrue(boolean conditionAboutArgs) throws ParseException {
        if (!conditionAboutArgs) {
            throw new ParseException("Illegal arguments");
        }
    }

    private void assertFalse(boolean conditionAboutArgs) throws ParseException {
        if (conditionAboutArgs) {
            throw new ParseException("Illegal arguments");
        }
    }

    private void printUsage(String[] examples, Options options) {
        HelpFormatter formatter = new HelpFormatter();
        StringBuilder sb = new StringBuilder();
        for (String example : examples) {
            if (Utils.isNullOrEmpty(sb.toString())) {
                sb = new StringBuilder();
            } else {
                sb.append("       ");
            }
            sb.append(example).append("\n");
        }
        sb.append("options:");
        formatter.printHelp(sb.toString(), options);
    }

    private boolean validateContactId(String contactId) {
        return !("".equals(contactId)); // todo: add a regex property for this validation
    }

    private final GenericChannelAPIClient gcapiClient;
    private final CoreDataServiceClient сoreDataServiceClientClient;
    private final EventingConnectorClient eventingConnectorClient;
    private final EventListener eventListener;
    private final Configuration configuration;

    private static final Option numberOption = Option.builder("n").hasArg().argName("number").desc("Number of contacts to create").build();
    private static final Option rateOption = Option.builder("r").hasArg().argName("rate").desc("Request rate in contacts per second, default is 1").build();
    private static final Option contactIDOption = Option.builder("i").hasArg().argName("contactID").desc("contactID prefix").build();
    private static final Option contextIDOption = Option.builder("c").hasArg().argName("contextID").desc("Context ID").build();
    private static final Option dataOption = Option.builder("d").hasArg().argName("extraData").desc("extra Data field").build();
}
