package com.avaya.generic.channel.testclient;

import com.avaya.generic.channel.testclient.utils.ExternalException;
import com.avaya.generic.channel.testclient.utils.FileIO;
import com.avaya.generic.channel.testclient.utils.Rest;
import com.avaya.generic.channel.testclient.utils.RestResponse;
import com.avaya.generic.channel.testclient.utils.Utils;
import org.apache.http.ConnectionClosedException;
import org.apache.http.ExceptionLogger;
import org.apache.http.HttpEntity;
import org.apache.http.HttpEntityEnclosingRequest;
import org.apache.http.HttpException;
import org.apache.http.HttpRequest;
import org.apache.http.HttpResponse;
import org.apache.http.HttpStatus;
import org.apache.http.client.methods.HttpPost;
import org.apache.http.config.SocketConfig;
import org.apache.http.entity.StringEntity;
import org.apache.http.impl.bootstrap.HttpServer;
import org.apache.http.impl.bootstrap.ServerBootstrap;
import org.apache.http.protocol.HttpContext;
import org.apache.http.protocol.HttpRequestHandler;
import org.apache.http.util.EntityUtils;
import org.json.JSONObject;

import java.io.IOException;
import java.net.SocketException;
import java.net.SocketTimeoutException;
import java.security.KeyManagementException;
import java.security.NoSuchAlgorithmException;
import java.text.SimpleDateFormat;
import java.util.Date;
import java.util.concurrent.TimeUnit;
import java.util.concurrent.atomic.AtomicBoolean;
import java.util.concurrent.atomic.AtomicReference;

/**
 * Based on this example:
 * https://hc.apache.org/httpcomponents-core-ga/httpcore/examples/org/apache/http/examples/HttpFileServer.java
 *
 * This class does not know about Configuration as asynchronous access to Configuration could be unsafe.
 */
public class EventListener {

    public EventListener() {

        final SocketConfig socketConfig = SocketConfig.custom()
                .setSoTimeout(5000)
                .setTcpNoDelay(true)
                .build();

        this.serverBootstrap = ServerBootstrap.bootstrap()
                .setServerInfo("GenericChannelAPITestClient/2.0")
                .setSocketConfig(socketConfig)
                .setSslContext(null)
                .setExceptionLogger(new ExceptionLoggerImpl())
                .registerHandler("*", new HttpRequestHandlerImpl());
    }

    /**
     * Start HTTP server, or restart if the port changed.
     * And after that ensures the events can arrive to the specified host and port
     */
    synchronized void start(String host, int port) throws IOException, ExternalException, NoSuchAlgorithmException, KeyManagementException {

        hideEvents();

        // shutdown the server if we need to change the port number
        if (server != null && server.getLocalPort() != port) {
            Utils.println("Shutting down EventListener on port %d...", server.getLocalPort());
            shutdown();
            server = null;
        }

        // create a new server if required
        if (server == null) {
            Utils.println("Starting EventListener on port %d...", port);
            server = serverBootstrap.setListenerPort(port).create();
            server.start();
        }

        // check the server
        final String url = Utils.buildUrl(host + ":" + port);
        final RestResponse response = Rest.execute(new HttpPost(url));
        if (!(response.isOK())) {
            throw new ExternalException("EventListener check failed. Check your firewall or try another port.");
        } else if (!(getSignature().equals(response.getContent()))) {
            throw new ExternalException("Port %d is used by another program", port);
        }
    }

    /**
     * Sets the event visibility flag.
     * @param fileIO for optional output to a file
     * @param subscriptionId the subscriptionId to filter the events by. If empty, all events will be printed.
     */
    void showEvents(FileIO fileIO, String subscriptionId) {
        this.eventsVisible.set(true);
        this.fileIO.set(fileIO);
        this.subscriptionId.set(subscriptionId);
    }

    /**
     * Clears the event visibility flag
     */
    void hideEvents() {
        this.eventsVisible.set(false);
        this.fileIO.set(null);
        this.subscriptionId.set(null);
    }

    /**
     * Shutdown HTTP server i.e. terminate the associated thread
     */
    synchronized void shutdown() {
        hideEvents();
        if (server != null) {
            server.shutdown(10, TimeUnit.SECONDS);
            server = null;
        }
    }

    //================================ private ==============================================

    private static class ExceptionLoggerImpl implements ExceptionLogger {
        @Override
        public void log(final Exception ex) {

            // Exceptions are frequently thrown and most of them do not actually mean something wrong.
            if (!(ex instanceof SocketTimeoutException) && !(ex instanceof ConnectionClosedException) &&
                    !((ex instanceof SocketException) && Utils.isNotEmpty(ex.getMessage()) && ex.getMessage().toLowerCase().contains("socket closed"))) {

                Utils.printException(EventListener.class.getSimpleName(), ex);
            }
        }
    }

    private class HttpRequestHandlerImpl implements HttpRequestHandler {

        public HttpRequestHandlerImpl() {
            super();
        }
        
        @Override
        public void handle(
                final HttpRequest request,
                final HttpResponse response,
                final HttpContext context) throws HttpException, IOException {

            if (eventsVisible.get() && (request instanceof HttpEntityEnclosingRequest)) {

                String entityContent = "";
                final HttpEntity entity = ((HttpEntityEnclosingRequest) request).getEntity();

                if (entity != null) {

                    try {
                        entityContent = EntityUtils.toString(entity);
                    } catch (Exception ex) {
                        entityContent = ""; // failed to read the content, proceeding with empty content
                    }

                    if (entityContent == null) {
                        entityContent = "";
                    }

                    printEvent(entityContent);
                }
            }

            // always return 200 with the RESPONSE_MESSAGE
            response.setStatusCode(HttpStatus.SC_OK);
            response.setEntity(new StringEntity(getSignature()));
        }
    }

    /**
     * Checks if the subscriptionId matches the current subscription.
     * If the event has family, type, and contactId, it is pronted in a short format. Otherwise, the event is printed as is.
     * @param event a JSON string expected
     */
    private void printEvent(String event) {

        String publicationTimeStamp = "";
        String formattedEvent = "";

        // parse
        try {

            final JSONObject json = new JSONObject(event);

            // check if the subscriptionId matches the current subscription
            final String subscriptionId = this.subscriptionId.get();
            if (Utils.isNotEmpty(subscriptionId) && !subscriptionId.equals(json.optString("subscriptionId", ""))) {
                return;
            }

            // continue parsing if the publication timestamp is not present
            try {
                publicationTimeStamp = DATEFORMAT.format(new Date(json.getLong("publicationTimestamp"))); // small 's'
            } catch (Exception ex) {
                publicationTimeStamp = "";
            }

            // family, type, and ContactId are mandaroty
            final String family = json.getString("family");
            final String type = json.getString("type");
            final String theContactId = json.getJSONObject("eventBody").getString("ContactId"); // capital C
            String theResourceID = theResourceID = "";
            if(type.equalsIgnoreCase("CONTACT_ANSWERED")) {
                try {
                    theResourceID = json.getJSONObject("eventBody").getString("NativeResourceId"); // capital C 
                } catch (Exception e) {
                    theResourceID = event; // the event does not have NativeResourceId. Print the event as is.
                }
                formattedEvent = String.format("%s.%s ContactId = %s  ResourceID = %s", family, type, theContactId, theResourceID);
            } else {
                formattedEvent = String.format("%s.%s ContactId = %s", family, type, theContactId);
            }
            
        } catch (Exception ex) {
            formattedEvent = event; // the event does not have family, type, or contactId. Print the event as is.
        }

        // print
        final String arrivalTimeStamp = DATEFORMAT.format(new Date());
        String line = "";
        if (Utils.isNotEmpty(publicationTimeStamp)) {
            line = String.format("[pub %s, arr %s] %s", publicationTimeStamp, arrivalTimeStamp, formattedEvent);
        } else {
            line = String.format("[arr %s] %s", arrivalTimeStamp, formattedEvent);
        }

        Utils.println(line);

        final FileIO fileIO = this.fileIO.get();
        if (fileIO != null) {
            try {
                fileIO.writeLine(line);
            } catch (Exception ex) {
                Utils.printException(EventListener.class.getSimpleName(), ex);
            }
        }

    }

    /**
     * The signature is used to check whether a response is from this object or from something else.
     */
    private String getSignature() {
        return String.format("~~~ EventListener %s OK ~~~", Integer.toHexString(hashCode()));
    }

    private final ServerBootstrap serverBootstrap;
    private HttpServer server = null;
    private final AtomicBoolean eventsVisible = new AtomicBoolean(false);
    private final AtomicReference<FileIO> fileIO = new AtomicReference(null);
    private final AtomicReference<String> subscriptionId = new AtomicReference(null);

    private final SimpleDateFormat DATEFORMAT = new SimpleDateFormat("HH:mm:ss");
}