package com.avaya.generic.channel.testclient.utils;

import com.avaya.generic.channel.testclient.PropertyKey;
import java.io.IOException;
import java.io.UnsupportedEncodingException;
import java.security.KeyManagementException;
import java.security.NoSuchAlgorithmException;
import org.apache.http.HttpEntity;
import org.apache.http.StatusLine;
import org.apache.http.client.methods.CloseableHttpResponse;
import org.apache.http.client.methods.HttpDelete;
import org.apache.http.client.methods.HttpGet;
import org.apache.http.client.methods.HttpPost;
import org.apache.http.client.methods.HttpPut;
import org.apache.http.client.methods.HttpRequestBase;
import org.apache.http.client.utils.HttpClientUtils;
import org.apache.http.conn.ssl.SSLConnectionSocketFactory;
import org.apache.http.conn.ssl.SSLContexts;
import org.apache.http.entity.StringEntity;
import org.apache.http.impl.client.CloseableHttpClient;
import org.apache.http.impl.client.HttpClients;
import org.apache.http.util.EntityUtils;
import org.json.JSONObject;

import javax.net.ssl.SSLContext;
import javax.net.ssl.X509TrustManager;
import java.security.SecureRandom;
import java.security.cert.CertificateException;
import java.security.cert.X509Certificate;
import org.apache.http.HttpStatus;

/**
 * Encapsulates HTTP GET/POST requests building and execution.
 * Prints request, does not print responses.
 */
public class Rest {

    public static RestResponse get(String url) throws UnsupportedEncodingException, ExternalException, IOException {
        printRequest("GET", url, null);
        return execute(new HttpGet(url));
    }

    public static RestResponse put(String url) throws UnsupportedEncodingException, ExternalException, IOException {
        printRequest("PUT", url, null);
        return execute(new HttpPut(url));
    }

    public static RestResponse delete(String url) throws  UnsupportedEncodingException, ExternalException, IOException {
        printRequest("DELETE", url, null);
        return execute(new HttpDelete(url));
    }

    public static RestResponse post(String url, JSONObject json) throws UnsupportedEncodingException, ExternalException, IOException{
        printRequest("POST", url, json);
        if (json == null) {
            return new RestResponse(HttpStatus.SC_BAD_REQUEST, "");
        } else {
            final HttpPost request = new HttpPost(url);
            request.addHeader("content-type", "application/json");
            final StringEntity stringEntity = new StringEntity(json.toString());
            request.setEntity(stringEntity);
            return execute(request);
        }
    }

    public static void shutdown() {
        HttpClientUtils.closeQuietly(httpClientInstance);
        HttpClientUtils.closeQuietly(httpSecureClientInstance);
    }

    /**
     * The low-level method for custom requests
     */
    public static RestResponse execute(HttpRequestBase request) throws IOException, ExternalException{

        CloseableHttpResponse httpResponse = null;
        HttpEntity entity = null;

        try {

            // execute the request
            if (request.getURI().toString().toLowerCase().startsWith("https:")) {
                httpResponse = getHttpSecureClient().execute(request);
            } else {
                httpResponse = getHttpClient().execute(request);
            }

            // read the response code
            int code = 0;
            final StatusLine statusLine = httpResponse.getStatusLine();
            if (statusLine != null) {
                code = statusLine.getStatusCode();
            } else {
                throw new ExternalException("No status code found in the response"); // a response without the code is useless
            }

            // read the response message
            String content = "";
            entity = httpResponse.getEntity();
            if (entity != null) {
                try {
                    content = EntityUtils.toString(entity);
                } catch (Exception ex) {
                    content = ""; // failed to read the content, proceeding with empty content
                }
            }
            // do not print the response here as it may be too long
            return new RestResponse(code, content);
        }catch( NoSuchAlgorithmException|KeyManagementException ex){
            return new RestResponse(HttpStatus.SC_NOT_IMPLEMENTED, "");
        } finally {
            EntityUtils.consumeQuietly(entity);
            HttpClientUtils.closeQuietly(httpResponse);
        }
    }

    //================================ private ============================================

    private static void printRequest(String method, String url, JSONObject json) {
        Utils.printSeparator();
        Utils.print("%s %s", method, url);
        if (json != null) {
            // trim the ssoToken to the first 16 chars.
            JSONObject jsonCopy = json;
            if (json.has(PropertyKey.ssoToken.name())) {
                jsonCopy = new JSONObject(json, JSONObject.getNames(json));
                String ssoToken = (String) (jsonCopy.getString(PropertyKey.ssoToken.name()));
                jsonCopy.put(PropertyKey.ssoToken.name(), Utils.trimTo16chars(ssoToken));
            }
            Utils.println(" %s", jsonCopy.toString());
        } else {
            Utils.println();
        }
        Utils.printSeparator();
    }

    private static synchronized CloseableHttpClient getHttpClient() {
        if (httpClientInstance == null) {
            httpClientInstance = HttpClients.createDefault();
        }
        return httpClientInstance;
    }

    private static synchronized CloseableHttpClient getHttpSecureClient() throws NoSuchAlgorithmException, KeyManagementException {
        if (httpSecureClientInstance == null) {
            SSLContext sslcontext = SSLContexts.custom().useSSL().build();
            sslcontext.init(null, new X509TrustManager[]{new HttpsTrustManager()}, new SecureRandom());
            SSLConnectionSocketFactory factory = new SSLConnectionSocketFactory(sslcontext, SSLConnectionSocketFactory.ALLOW_ALL_HOSTNAME_VERIFIER); // todo: avoid using deprecated code
            httpSecureClientInstance = HttpClients.custom().setSSLSocketFactory(factory).build();
        }

        return httpSecureClientInstance;
    }

    private static class HttpsTrustManager implements X509TrustManager {

        @Override
        public void checkClientTrusted(X509Certificate[] arg0, String arg1) throws CertificateException {
            //do nothing
        }

        @Override
        public void checkServerTrusted(X509Certificate[] arg0, String arg1) throws CertificateException {
            //do nothing
        }

        @Override
        public X509Certificate[] getAcceptedIssuers() {
            return new X509Certificate[]{};
        }
    }

    private static CloseableHttpClient httpClientInstance = null;
    private static CloseableHttpClient httpSecureClientInstance = null;
}
